<?php class ReferenceDataLoader {

    
    /**
     * Triggers the loading of reference data during an upgrade.
     *
     * Caller is responsible for managing database transactions, including commit.
     */
    public static function loadReferenceDataForUpgrade($referenceDataLocation=false){
        if(!Log::isInited()){
            throw new ServerException('loadReferenceDataForUpgrade: Logging is not inited');
        }
        Log::write(Log::LOGLEVEL_DEBUG, 'In loadReferenceDataForUpgrade');
        static::setReferenceDataLocation($referenceDataLocation);
        static::loadReferenceDataForInstallPart1();
        static::loadReferenceDataForInstallPart2();
        Log::write(Log::LOGLEVEL_DEBUG, 'loadReferenceDataForUpgrade complete');
    }
    
    /**
     * Loads the first parts of the reference data - specifically, projects and usergroups.
     * Because later parts assume that this is set up, we must do the initial load in two
     * parts.
     *  
     * Caller is responsible for managing database transactions, including commit.
     */
    public static function loadReferenceDataForInstallPart1(){
        if(!Log::isInited()){
            throw new ServerException('loadReferenceDataForInstallPart1: Logging is not inited');
        }
        if(empty(static::$referenceDataLocation)){
            static::setReferenceDataLocation();
        }
        static::loadProjects();
        static::loadUsergroups();
        
        //give Everyone read on Shared
        $everyone=usergroup::getByName(usergroup::EVERYONE);
        $shared=project::getByName(project::SHARED);
        try {
            permission::create(array(
                'usergroupid'=>$everyone['id'],
                'projectid'=>$shared['id'],
                'type'=>'read'
            ));
        } catch(BadRequestException $e){
            if(false===stripos($e->getMessage(), 'already exists')){
                throw $e;
            }
        }
    }


    /**
     * Loads the reference data not handled by loadReferenceDataForInstallPart1.
     *
     * For both updates and new installations, full IceBear user auth and database handling must be
     * available before calling this.
     *
     * Caller is responsible for managing database transactions, including commit.
     * @throws ServerException
     */
    public static function loadReferenceDataForInstallPart2(){
        if(!Log::isInited()){
            throw new ServerException('loadReferenceDataForInstallPart2: Logging is not inited');
        }
        if(empty(static::$referenceDataLocation)){
            static::setReferenceDataLocation();
        }
        static::loadCrystallizationScreens();
        static::loadContainerTypes();
        static::loadShipmentDestinations();
    }
    
    private static $referenceDataLocation;
    
    private static function setReferenceDataLocation($referenceDataLocation=false){
        if(empty(static::$referenceDataLocation)){
            static::$referenceDataLocation=rtrim($_SERVER['DOCUMENT_ROOT'],'/').'/upgrade/referenceData/';
        } else {
            static::$referenceDataLocation=rtrim($referenceDataLocation,'/').'/';
        }
    }
    
    /**
     * Load the projects CSV into the database.
     * 
     * Caller is responsible for wrapping this method in a transaction if needed.
     * 
     * @throws ForbiddenException if current user is not an admin.
     * @throws Exception if CSV cannot be opened.
     */
    private static function loadProjects(){
        Log::write(Log::LOGLEVEL_DEBUG, 'In loadProjects...');
        if(!session::isAdmin()){ throw new ForbiddenException('Non-administrator cannot load project reference data'); }
        $adminId=session::getUserId();
        $file=@fopen(static::$referenceDataLocation.'project.csv','r');
        if(!$file){ throw new Exception('Could not open projects list for import'); }
        $keys=fgetcsv($file);
        $keys[]='owner';
        
        $query='INSERT INTO project('.implode(',', $keys).') VALUES(:'.implode(',:', $keys).')';
        $keys=explode('|', ':'.implode('|:', $keys)); //prepend : to each key
        
        $values=fgetcsv($file);
        while(!empty($values)){
            $values[]=$adminId;
            try {
                $project=array_combine($keys, $values);
                database::query($query, $project);
                Log::write(Log::LOGLEVEL_DEBUG, 'Created project: '.$project[':name']);
            } catch(BadRequestException $e){
                $message=$e->getMessage();
                if(false===strpos($message, 'constraint violation')){
                    throw $e;
                }
                Log::write(Log::LOGLEVEL_DEBUG, 'Project already exists: '.$project[':name']);
            }
            
            $values=fgetcsv($file);
        }
        @fclose($file);
        Log::write(Log::LOGLEVEL_DEBUG, 'loadProjects complete.');
    }

    
    /**
     * Load the usergroups CSV into the database. Caller is responsible for wrapping this in a transaction if needed.
     *
     * @throws ForbiddenException if current user is not an admin.
     * @throws Exception if CSV cannot be opened.
     */
    private static function loadUsergroups($pdo=false, $adminId=false){
        Log::write(Log::LOGLEVEL_DEBUG, 'In loadUsergroups...');
        if(!$adminId){
            if(!session::isAdmin()){ throw new ForbiddenException('Non-administrator cannot load usergroup reference data'); }
            $adminId=session::getUserId();
        }
        //Usergroups
        $file=@fopen(static::$referenceDataLocation.'usergroup.csv','r');
        if(!$file){ throw new Exception('Could not open usergroups list for import'); }
        $keys=fgetcsv($file);

        $query='INSERT INTO usergroup('.implode(',', $keys).') VALUES(:'.implode(',:', $keys).')';
        $keys=explode('|', ':'.implode('|:', $keys)); //prepend : to each key
        
        $values=fgetcsv($file);
        while(!empty($values)){
            try {
                $usergroup=array_combine($keys, $values);
                database::query($query, $usergroup);
                Log::write(Log::LOGLEVEL_DEBUG, 'Created usergroup: '.$usergroup[':name']);
            } catch(BadRequestException $e){
                $message=$e->getMessage();
                if(false===strpos($message, 'constraint violation')){
                    throw $e;
                }
                Log::write(Log::LOGLEVEL_DEBUG, 'Usergroup already exists: '.$usergroup[':name']);
            }
            $values=fgetcsv($file);
        }
        
        while(!empty($values)){
            $values[]=$adminId;
            try {
                $name=$values[0];
                $existing=usergroup::getByName($name);
                if(!$existing){
                    database::query($query, array_combine($keys, $values));
                }
            } catch(NotFoundException $e){
                database::query($query, array_combine($keys, $values));
            }
            $values=fgetcsv($file);
        }
        @fclose($file);
        Log::write(Log::LOGLEVEL_DEBUG, 'loadUsergroups complete.');
    }
    
    
    /**
     * Parses the containertype.csv file for container types (Unipuck, etc), creating
     * them - along with their container categories (Puck, etc.) - if not already present.
     * @throws Exception if CSV cannot be opened.
     */
    private static function loadContainerTypes(){
        Log::write(Log::LOGLEVEL_DEBUG, 'In loadContainerTypes...');
        $file=@fopen(static::$referenceDataLocation.'containertype.csv','r');
        if(!$file){ throw new Exception('Could not open container types list for import'); }
        $keys=fgetcsv($file); //don't want the header
        Log::write(Log::LOGLEVEL_DEBUG, '[ '. implode(' ][ ', $keys).' ]');
        $values=fgetcsv($file);
        while(!empty($values)){
            Log::write(Log::LOGLEVEL_DEBUG, '[ '. implode(' ][ ', $values).' ]');
            $categoryName=$values[0];
            $typeName=$values[1];
            $positions=$values[2];
            $type=containertype::getByName($typeName);
            if($type){
                Log::write(Log::LOGLEVEL_DEBUG, $typeName.' exists');
            } else {
                $category=containercategory::getByName($categoryName);
                if(!$category){
                    Log::write(Log::LOGLEVEL_DEBUG, 'Creating containercategory '.$categoryName);
                    $category=containercategory::create(array(
                        'name'=>$categoryName
                    ));
                    $category=$category['created'];
                    if(in_array(array('Pin','Puck','Dewar'), $categoryName)){
                        //Used in crystal fishing. Let users create by default.
                        containercategory::update($category['id'], array(
                            'usercancreate'=>'1'
                        ));
                    }
                    Log::write(Log::LOGLEVEL_DEBUG, '...done.');
                }
                $categoryId=$category['id'];
                Log::write(Log::LOGLEVEL_DEBUG, 'Creating containertype'.$typeName);
                containertype::create(array(
                    'name'=>$typeName,
                    'containercategoryid'=>$categoryId, 
                    'positions'=>$positions
                ));
                Log::write(Log::LOGLEVEL_DEBUG, '...done.');
            }
            $values=fgetcsv($file);
        }
        @fclose($file);
        Log::write(Log::LOGLEVEL_DEBUG, 'loadContainerTypes complete.');
    }
    
    
    /**
     * Parses the shipmentdestinations.csv file for shipment destinations.
     * These will be updated by force if they already exist.
     */
    private static function loadShipmentDestinations(){
        Log::write(Log::LOGLEVEL_DEBUG, 'in loadShipmentDestinations...');
        $file=@fopen(static::$referenceDataLocation.'shipmentdestination.csv','r');
        if(!$file){ throw new Exception('Could not open shipment destinations list for import'); }
        $keys=fgetcsv($file); //don't want the header
        $values=fgetcsv($file);
        while(!empty($values)){
            $destinationName=$values[0];
            $shipmentHandler=$values[1];
            $baseUri=$values[2];
            $existing=shipmentdestination::getByName($destinationName);
            if(!$existing){
                shipmentdestination::create(array(
                    'name'=>$destinationName,
                    'shipmenthandler'=>$shipmentHandler,
                    'baseuri'=>$baseUri
                ));
            } else {
                shipmentdestination::update($existing['id'], array(
                    'shipmenthandler'=>$shipmentHandler,
                    'baseuri'=>$baseUri
                ));
            }
            $values=fgetcsv($file);
        }
        @fclose($file);
        Log::write(Log::LOGLEVEL_DEBUG, 'loadShipmentDestinations complete.');
    }
    

    
    
    /**
     * Processes the reference data for crystallization screens.
     * 
     * Where a screen by this name already exists in IceBear, it is ignored. The installation site may have
     * modified it on purpose. Forced updates to the screen must be done through vOLD-to-vNEW/upgrade.php.
     * 
     * Strategy:
     * - Read files one by one, ignoring all that do not end in _icebear.csv
     * - For each _icebear.csv
     *   - parse it into conditions
     *   - Create a standard screen with those conditions
     *   - Attach any manufacturer screen definition files to that screen
     * - Parse the refdata CSV for manufacturer names and catalogue numbers
     */
    private static function loadCrystallizationScreens(){
        Log::write(Log::LOGLEVEL_DEBUG, 'In loadCrystallizationScreens...');
        
        //Get the shared project
        $sharedProject=project::getByName(project::SHARED);
        if(!$sharedProject){ throw new Exception("Shared project does not exist. Cannot create screens."); }
        $sharedProjectId=$sharedProject['id'];
        
        //File store, for ref data with files (such as standard screens)
        $destinationDir=rtrim(config::get('core_filestore'),'/').'/files/project'.$sharedProjectId.'/';
        $backupDir='';
        if(!empty(config::get('core_filestorebackup'))){
            $backupDir=rtrim(config::get('core_filestorebackup'),'/').'/files/project'.$sharedProjectId.'/';
        }
        if(!file_exists($destinationDir)){
            if(!@mkdir($destinationDir, 0700, true)){
                throw new ServerException('Could not make project directory in filestore for shared project');
            }
        }
        
        //Standard screens
        $source=static::$referenceDataLocation."screen/";
        $sourceDir=dir($source);
        while(false!==($filename=$sourceDir->read())) {
            if(!preg_match('/.+_icebear\.csv$/',$filename)){ continue; }
            
            $screenName=str_replace('_icebear.csv','',$filename);
            
            //Skip this file if IceBear already has screen by this name
            $existing=screen::getByName($screenName);
            if($existing){ continue; }
            
            //Parse conditions from the 2-column CSV into a 12x8 array
            $fileContent=file_get_contents($source.$filename);
            $conditions=screen::parseWellNameAndConditionCsv($fileContent);
            
            //Create the screen and the conditions
            $screenName=str_replace('_icebear.csv','',$filename);
            $request=array(
                'name'=>$screenName,
                'rows'=>8,
                'cols'=>12,
                'conditions'=>$conditions
            );
            $screen=screen::create($request);
            $screenId=$screen['created']['id'];
            
            //Attach the IceBear screen definition file to the screen and copy it into place
            $request=array(
                'projectid'=>$sharedProjectId,
                'parentid'=>$screenId,
                'filename'=>$filename,
                'mimetype'=>mime_content_type($source.$filename),
                'bytes'=>filesize($source.$filename),
                'description'=>'IceBear screen definition file'
            );
            $baseObject=baseobject::createByClassName($request, 'file');
            $fileId=$baseObject['created']['id'];
            database::query('UPDATE file SET name=:name WHERE id=:id', array(':name'=>'file'.$fileId, ':id'=>$fileId));
            copy($source.$filename, $destinationDir.'file'.$baseObject['created']['id']);
            if(!empty($backupDir)){
                copy($source.$filename, $backupDir.'file'.$baseObject['created']['id']);
            }
            
            //Look for a manufacturer definition file and attach it to the screen
            $screenExtensions=array('.csv','.txt','.doc','.docx','.xls','.xlsx','.pdf');
            foreach($screenExtensions as $ext){
                $filePath=$source.$screenName.$ext;
                if(file_exists($filePath)){
                    $request['filename']=$screenName.$ext;
                    $request['mimetype']=mime_content_type($filePath);
                    $request['bytes']=filesize($filePath);
                    $request['description']='Manufacturer screen definition file';
                    $baseObject=baseobject::createByClassName($request, 'file');
                    $fileId=$baseObject['created']['id'];
                    database::query('UPDATE file SET name=:name WHERE id=:id', array(':name'=>'file'.$fileId, ':id'=>$fileId));
                    copy($filePath, $destinationDir.'file'.$baseObject['created']['id']);
                    if(!empty($backupDir)){
                        copy($filePath, $destinationDir.'file'.$baseObject['created']['id']);
                    }
                }
            }
        }
        $sourceDir->close();
        
        //parse the refdata CSV for manufacturer, catalogue number - and update the screen records
        $file=@fopen($source.'_screen.csv','r');
	if(!$file){ throw new Exception('Could not open screen list for import'); }
        $keys=fgetcsv($file);
        $values=fgetcsv($file);
        while(false!==$values){
            $screenName=$values[0];
            $screen=screen::getByName($screenName);
            $screenId=$screen['id'];
            $request=array();
            $request['isstandard']=1;
            for($i=1;$i<count($keys);$i++){ // 0 is name, not modifying
                $request[$keys[$i]]=$values[$i];
            }
            screen::update($screenId, $request);
            $values=fgetcsv($file);
        }
        @fclose($file);
        Log::write(Log::LOGLEVEL_DEBUG, 'loadCrystallizationScreens complete.');
    }
    

} ?>