<?php

use PHPUnit\Framework\TestCase;

class userTest extends TestCase {

    protected function setUp():void {
        try {
            session::init(new DummySession());
        } catch (BadRequestException $e) {
            $this->fail('BadRequestException on session::init');
            return;
        } catch (NotFoundException $e) {
            $this->fail('NotFoundException on session::init');
            return;
        } catch (ServerException $e) {
            $this->fail('ServerException on session::init');
            return;
        }
        database::begin();
    }

    protected function tearDown():void {
        database::abort();
        session::destroy();
    }

    public function testCanCreateAsAdmin(){
        session::becomeAdmin();
        $this->assertTrue(user::canCreate());
    }

    public function testCanCreateAsNonAdmin(){
        $this->assertFalse(user::canCreate());
    }

    public function testCreateAsAdmin(){
        session::becomeAdmin();
        try {
            $created=user::create(array(
                'name'=>'test'.time(),
                'fullname'=>'Test '.time(),
                'email'=>time().'@test.user',
                'password'=>'USELESSPASSWORD'
            ));
            $this->assertNotEmpty($created);
            $this->assertNotEmpty(user::getById($created['created']['id']));
        } catch (ForbiddenException $exception){
            $this->fail('Could not create user as admin (ForbiddenException)');
        } catch (Exception $exception){
            $this->fail('Could not create user as admin (not ForbiddenException)');
        }

    }

    /**
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public function testCreateAsNonAdmin(){
        $this->expectException(ForbiddenException::class);
        user::create(array(
                'name'=>'test'.time(),
                'fullname'=>'Test '.time(),
                'email'=>time().'@test.user'
            ));
    }

    /**
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public function testGetById(){
        session::becomeAdmin();
        $username='test'.time();
        $created=user::create(array(
            'name'=>$username,
            'fullname'=>'Test '.time(),
            'email'=>time().'@test.user',
            'password'=>'USELESSPASSWORD'
        ));
        $id=$created['created']['id'];
        //Get by ID as admin
        $user=user::getById($id);
        $this->assertArrayHasKey('fullname', $user);
        $this->assertArrayHasKey('email', $user);
        $this->assertArrayNotHasKey('password', $user);
        //getById as non-admin
        session::revokeAdmin();
        $user=user::getById($id);
        $this->assertArrayHasKey('fullname', $user);
        $this->assertArrayNotHasKey('name', $user);
        $this->assertArrayNotHasKey('email', $user);
        $this->assertArrayNotHasKey('password', $user);
    }

    /**
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public function testGetByName(){
        session::becomeAdmin();
        $username='test'.time();
        $created=user::create(array(
            'name'=>$username,
            'fullname'=>'Test '.time(),
            'email'=>time().'@test.user',
            'password'=>'USELESSPASSWORD'
        ));
        //Get by ID as admin
        $user=user::getByName($username);
        $this->assertArrayHasKey('fullname', $user);
        $this->assertArrayHasKey('email', $user);
        $this->assertArrayNotHasKey('password', $user);
        //getById as non-admin
        session::revokeAdmin();
        $user=user::getByName($username);
        $this->assertArrayHasKey('fullname', $user);
        $this->assertArrayNotHasKey('name', $user);
        $this->assertArrayNotHasKey('email', $user);
        $this->assertArrayNotHasKey('password', $user);
    }

}