<?php 
/**
 * This importer copies plate inspections and their associated images from a Rigaku imager
 * into the LIMS for onward processing. It creates associated records in the LIMS, such as 
 * imagers and plate types, where needed.
 * 
 * Command-line arguments:
 * 
 * -h Show help
 * -c - Commissioning run. Import all plates and inspections, committing after each plate and inspection
 * -b909s - (Re)import images for this barcode
 * -l1 - Set logging level. 1=debug, 2=info, 3=warn, 4=error. (Default 2)
 * 
 * General approach:
 * 
 * We do not connect to the Rigaku database, instead relying on XML and the filesystem state. 
 * 
 * Stage 1: Parse all recently-modified plate XML files to ensure that the plate, plate type,
 * plate owner, project, (screen), etc., exist - creating them if not.
 * 
 * If a plate has a screen that is already attached to another plate in IceBear (i.e., there is
 * another plate whose screen has the same name), that screen is promoted to "standard screen" in 
 * IceBear. Conditions are assumed to be the same if the screen name is the same.
 * 
 * Stage 2: We look in the image store for plate directories modified in the last x hours, 
 * creating new plate inspection and image records as required.
 * 
 * This approach depends on filesystem last-modified dates and is therefore not watertight.
 * It also involves parsing entire directories at one sub-directory per plate, and will be 
 * PAINFULLY SLOW if not COMPLETELY UNWORKABLE on an imaging system seeing even moderate use
 * over an extended period. This is intended as a first attempt only, in order to get one 
 * lightly-used Rigaku installation tied into IceBear. The correct approach is to connect to
 * and use the (rather intimidating) Oracle database used by CrystalTrak.
 * 
 * It is assumed that there is exactly one imager, called "Minstrel", and that all plates are
 * at +20. If a plate XML specifies a different temperature, import will fail.
 */

set_time_limit(0);
setUpClassAutoLoader();
date_default_timezone_set('UTC');

//Plates and images with a filesystem last-modified time more than this many hours ago will not be (re-)imported.
//Does not affect by-barcode import - plate and images for the specified barcode are always imported.
//Zero means "always import, regardless of last-modified time". Think carefully before setting this in a production environment!
$lastModifiedHoursCutoffPlates=24;
$lastModifiedHoursCutoffImages=24;

$isCommissioning=false;

define('IMAGINGSESSIONS_PER_RUN',10);

//These are just defaults. Imagers are imported rarely, and these numbers can be changed later, 
//so no need for a separate config item.
define('IMAGERLOAD_ALERT_THRESHOLD_PERCENT',75);
define('IMAGERLOAD_WARNING_THRESHOLD_PERCENT',90);

define('IMAGER_NAME_UNKNOWN', 'Unknown Imager');
define('IMAGER_NAME', 'Minstrel');
define('IMAGER_TEMP', 20);
define('IMAGER_MICRONS_PER_PIXEL',1.06);

define('MINIMUM_INSPECTION_AGE_MINS',20);  //Don't attempt to parse inspections younger than this. Images are likely not processed yet.

define('LOGLEVEL_DEBUG', 1);
define('LOGLEVEL_INFO', 2);
define('LOGLEVEL_WARN', 3);
define('LOGLEVEL_ERROR', 4);
$logLevel=LOGLEVEL_INFO; //may be overridden by argument

//Mapping of image type code (found in filenames) to name and light path.
//Where multiple images are taken of a drop, the inspection will be split into several IceBear inspections.
//The order of those inspections is the reverse of this array. If adding to this, keep the default 00/Visible FIRST.
$imagingTypes=array(
	'00'=>array('name'=>'Normal', 'light'=>'Visible'), //This entry FIRST
    'E0'=>array('name'=>'Enhanced', 'light'=>'Visible'),
    'HF'=>array('name'=>'High Frequency', 'light'=>'Visible'),
	'U0'=>array('name'=>'UV', 'light'=>'UV'),
);

$shouldntOwnPlates=array('Administrator');
$limitStart=0;
$limitTotal=IMAGINGSESSIONS_PER_RUN;
$fromDate=null;
$imager=null;
$barcode=null;

writeLog(LOGLEVEL_INFO,'Importer started');

for($i=1;$i<count($argv);$i++){
	$arg=$argv[$i];
	if(preg_match('/^-h$/',$arg)){
	    writeLog(LOGLEVEL_DEBUG,'Help requested');
	    showHelp();
	    exit();
	} else if(preg_match('/^-c$/',$arg)){
	    writeLog(LOGLEVEL_INFO,'Commissioning import specified');
	    $isCommissioning=true;
	} else if(preg_match('/^-l[1-4]$/',$arg)){
		$logLevel=(int)substr($arg, 2);
	} else if(preg_match('/^-s\d+$/',$arg)){
		$limitStart=(int)substr($arg, 2);
	} else if(preg_match('/^-i.+$/',$arg)){
		$imager=substr($arg, 2);
	} else if(preg_match('/^-b.+$/',$arg)){
		$barcode=substr($arg, 2);
	} else if(preg_match('/^-d\d\d\d\d-\d\d-\d\d+$/',$arg)){
		$fromDate=substr($arg,2);
		writeLog(LOGLEVEL_INFO,'Start date specified: '.$fromDate);
	} else {
        writeLog(LOGLEVEL_INFO,'No command-line switches, normal import');
	}
}

writeLog(LOGLEVEL_INFO,'Importer started');

writeLog(LOGLEVEL_DEBUG,'Attempting to get LIMS DB connection... ');
	database::connect();
writeLog(LOGLEVEL_DEBUG,'...got LIMS DB connection');

writeLog(LOGLEVEL_DEBUG,'Attempting to get LIMS session... ');
try {
    session::init(new DummySession());
} catch (BadRequestException $e) {
    writeLog(LOGLEVEL_ERROR,'...session::init threw BadRequestException');
    exit();
} catch (NotFoundException $e) {
    writeLog(LOGLEVEL_ERROR,'...session::init threw NotFoundException');
    exit();
} catch (ServerException $e) {
    writeLog(LOGLEVEL_ERROR,'...session::init threw ServerException');
    exit();
}
session::set('isAdmin',true);
writeLog(LOGLEVEL_DEBUG,'...got LIMS session');

$sharedProjectId=null;

try {

	database::begin();
	
	$user=user::getByName('rigakuimporter');
	if(empty($user)){
		$user=user::create(array(
				'name'=>'rigakuimporter',
				'fullname'=>'Rigaku Importer',
				'email'=>'rigimport@null.null',
				'password'=>'USELESSPASSWORD',
				'isactive'=>0
		));
		$user=$user['created'];
	}
	session::init(new DummySession());
	session::set('userId', $user['id']);
	session::set('isAdmin', true);

	$hasRigakuImagers=config::get('rigaku_hasimagers');
	if(!(int)$hasRigakuImagers){
		throw new Exception("IceBear is not configured for import from Rigaku imagers");
	}
	
	//Config items
	$plateStore=config::get('rigaku_platepath');
	$imageStore=config::get('rigaku_imagepath');
	$thumbStore=config::get('rigaku_thumbpath');
	if(empty($plateStore) || empty($thumbStore) ||  empty($imageStore)){
		throw new Exception('Config items rigaku_platepath, rigaku_imagepath, rigaku_thumbpath must all be defined');
	}
	$plateStore=rtrim($plateStore,'/').'/';
	$imageStore=rtrim($imageStore,'/').'/';
	$thumbStore=rtrim($thumbStore,'/').'/';
	
	$limsImageStore=config::get('core_imagestore');
	$limsImageStore=rtrim($limsImageStore,'/\\').'/';
	if(!file_exists($limsImageStore)){
		writeLog(LOGLEVEL_ERROR, 'Cannot find LIMS image store: '.$limsImageStore);
		writeLog(LOGLEVEL_ERROR, 'Check that the drive is mounted and has the correct permissions');
		throw new Exception('Aborting because LIMS image store cannot be read');
	}
	
	//Find some common prerequisites
	
	$sharedProject=project::getByName('Shared');
	if(empty($sharedProject)){ throw new Exception('Could not find "Shared" project in LIMS'); }
	$sharedProjectId=$sharedProject['id'];
	
	$defaultProject=project::getByName('Default Project');
	if(empty($defaultProject)){ throw new Exception('Could not find "Default Project" in LIMS'); }
	$defaultProjectId=$defaultProject['id'];
	
	$scoringSystem=crystalscoringsystem::getByName('Custom');
	if(!$scoringSystem){
		$scoringSystem=crystalscoringsystem::getByName('Rigaku');
		if(!$scoringSystem){
			$scoringSystem=importScoringSystem();
		}
	}
	if(!$scoringSystem){
		throw new Exception("Could not find or import a scoring system");
	}

	$imager=getImager(IMAGER_NAME);
	writeLog(LOGLEVEL_DEBUG, 'Finding IceBear imagingparametersversion for each imaging type...');
	foreach($imagingTypes as &$it){
		//If it isn't in IceBear, create it.
		//Should really check for and support multiple versions of each type, but we're
		//making dummy ones anyway - no real need for versioning, unlike Formulatrix.
		writeLog(LOGLEVEL_DEBUG, 'Imaging type name is '.$it['name'].', looking for imagingparametersversion '.$it['name'].'_v1...');
		$limsParametersVersion=imagingparametersversion::getByName($it['name'].'_v1');
		if(!$limsParametersVersion){
			writeLog(LOGLEVEL_DEBUG, '...not found, creating imagingparameters first...');
			$ip=imagingparameters::create(array(
					'name'=>$it['name'],
					'manufacturer'=>'Rigaku',
					'manufacturerdatabaseid'=>0,
					'projectid'=>$sharedProjectId
			));
			$ip=$ip['created'];
			writeLog(LOGLEVEL_DEBUG, '...done. creating imagingparametersversion...');
			$ipv=imagingparametersversion::create(array(
					'name'=>$it['name'].'_v1',
					'manufacturerdatabaseid'=>0,
					'projectid'=>$sharedProjectId,
					'imagingparametersid'=>$ip['id']
			));
			$ipv=$ipv['created'];
			imagingparameters::update($ip['id'], array(
					'currentversionid'=>$ipv['id']
			));
			writeLog(LOGLEVEL_DEBUG, '...created.');
			$limsParametersVersion=imagingparametersversion::getByName($it['name'].'_v1');
		}
		writeLog(LOGLEVEL_DEBUG, 'imagingparametersversion '.$it['name'].'_v1... exists');
		$it['limsParametersVersion']=$limsParametersVersion;
	}
	database::commit();
	
	//Decide what to import
	$plateBarcodesToImport=array();
	$plateFilesToImport=array();
	$imagesToImport=array();

	$now=time();
	$lastModifiedTimestampCutoffPlates=0;
	$lastModifiedTimestampCutoffImages=0;
	if(0==$lastModifiedHoursCutoffPlates){
		$lastModifiedTimestampCutoffPlates=0;
	} else {
		$lastModifiedTimestampCutoffPlates=$now-(3600*$lastModifiedHoursCutoffPlates);
	}
	if(0==$lastModifiedHoursCutoffImages){
		$lastModifiedTimestampCutoffImages=0;
	} else {
		$lastModifiedTimestampCutoffImages=$now-(3600*$lastModifiedHoursCutoffImages);
	}
	
	if($isCommissioning){
        
	    writeLog(LOGLEVEL_INFO, 'Commissioning import specified');
	    writeLog(LOGLEVEL_INFO, 'Checking plate store...');
	    $dir=dir($plateStore);
	    $filename=$dir->read();
	    while(!empty($filename)){
	        if('.'!=$filename && '..'!=$filename){
                $plateFilesToImport[]=$filename;
	        }
	        $filename=$dir->read();
	    }
	    writeLog(LOGLEVEL_INFO, 'Done checking plate XML store. Found '.count($plateFilesToImport).' plate files to import.');
	    
	} else if(null!=$barcode){
		
		//By-plate import
		writeLog(LOGLEVEL_INFO, 'Single-plate import specified, barcode is '.$barcode);
		$lastModifiedTimestampCutoffPlates=0; //override for full plate history
		$lastModifiedTimestampCutoffImages=0; //override for full plate history
		$plateBarcodesToImport[]=$barcode;
		
	} else {
		
		writeLog(LOGLEVEL_INFO, 'Checking plate store for files modified in last '.$lastModifiedHoursCutoffPlates.'hrs...');
		$dir=dir($plateStore);
		$filename=$dir->read();
		while(!empty($filename)){
			if('.'!=$filename && '..'!=$filename){
				if(0==$lastModifiedTimestampCutoffPlates){
					writeLog(LOGLEVEL_DEBUG, 'No cutoff set, Importing:  '.$filename);
					$plateFilesToImport[]=$filename;
				} else if(filemtime($plateStore.$filename)<$lastModifiedTimestampCutoffPlates){
					writeLog(LOGLEVEL_DEBUG, 'File too old, ignoring:  '.$filename);
				} else {
					writeLog(LOGLEVEL_DEBUG, 'File changed, importing:: '.$filename);
					$plateFilesToImport[]=$filename;
				}
			}
			$filename=$dir->read();
		}
		writeLog(LOGLEVEL_INFO, 'Done checking plate XML store. Found '.count($plateFilesToImport).' plate files to import.');

	}

	writeLog(LOGLEVEL_INFO, 'Beginning plate import');
	writeLog(LOGLEVEL_INFO, '-------------------------------------------------');
	foreach($plateBarcodesToImport as $barcode){
	    writeLog(LOGLEVEL_DEBUG, 'Adding '.$barcode.' to barcodes list for image import');
	    $imagesToImport[]=$barcode;
	    database::begin();
		importPlateByBarcode($barcode, $isCommissioning);
		database::commit();
	}
	foreach($plateFilesToImport as $filename){
	    writeLog(LOGLEVEL_DEBUG, 'Beginning import of plate file '.$filename);
	    database::begin();
	    $plate=importPlateByFilename($filename, $isCommissioning);
	    if(!$plate){
	        writeLog(LOGLEVEL_WARN, 'No plate barcode found in '.$filename);
	        writeLog(LOGLEVEL_WARN, 'Not importing from this file.');
	    } else {
	        $barcode=$plate['name'];
	        writeLog(LOGLEVEL_INFO, 'Plate barcode found in file is '.$barcode);
	        writeLog(LOGLEVEL_DEBUG, 'Adding '.$barcode.' to barcodes list for image import');
	        $imagesToImport[]=$barcode;
	    }
		database::commit();
		writeLog(LOGLEVEL_DEBUG, 'Finished importing plate file '.$filename);
	}
	writeLog(LOGLEVEL_INFO, 'Finished plate import');
	writeLog(LOGLEVEL_INFO, 'Beginning inspection/image import');
	writeLog(LOGLEVEL_INFO, '-------------------------------------------------');
	foreach($imagesToImport as $barcode){
	    if(empty(trim($barcode))){
	        writeLog(LOGLEVEL_WARN, 'No plate barcode specified, not importing images for unknown plate');
	        continue;
	    }
		database::begin();
		importInspectionsForPlateBarcode($barcode,$isCommissioning);
		database::commit();
	}
	writeLog(LOGLEVEL_INFO, 'Finished inspection/image import');
	

} catch(Exception $e){
	writeLog(LOGLEVEL_ERROR, get_class($e).' caught: '.$e->getMessage());
	$trace=$e->getTrace();
	foreach($trace as $t){
		writeLog(LOGLEVEL_ERROR, ': '.$t['file']);
		if(isset($t['type']) && isset($t['class'])){
			writeLog(LOGLEVEL_ERROR, ':    Line '.$t['line'].', Function '.$t['class'].$t['type'].$t['function']);
		} else {
			writeLog(LOGLEVEL_ERROR, ':    Line '.$t['line'].', Function '.$t['function']);
		}
	}
	database::abort();
	writeLog(LOGLEVEL_ERROR, 'Import aborted due to exception');
}

writeLog(LOGLEVEL_INFO,'Importer finished.');
writeLog(LOGLEVEL_INFO,'=================================================');

/**
 * @param $barcode
 * @return string
 * @throws Exception
 */
function getXmlFilenameForPlateBarcode($barcode){
	global $plateStore;
	writeLog(LOGLEVEL_DEBUG, 'In getXmlFilenameForPlateBarcode, barcode="'.$barcode.'"...');
	$filenameToImport='';
	writeLog(LOGLEVEL_DEBUG, 'Filenames do not contain plate barcodes, checking each file for barcode="'.$barcode.'"...');
	$dir=dir($plateStore);
	$filename=$dir->read();
	while(!empty($filename)){
		if('.'!=$filename && '..'!=$filename){
			writeLog(LOGLEVEL_DEBUG, 'Checking file with name '.$filename);
			$contents=file_get_contents($plateStore.$filename);
			if(stripos($contents, ' barcode="'.$barcode.'" ')!==false){
				$filenameToImport=$filename;
				writeLog(LOGLEVEL_DEBUG, 'Filename for barcode="'.$barcode.'" is '.$filenameToImport);
				break;
			}
		}
		$filename=$dir->read();
	}
	if(empty($filenameToImport)){
		throw new Exception('No plate XML found containing barcode="'.$barcode.'" - cannot import this plate by barcode.');
	}
	writeLog(LOGLEVEL_DEBUG, 'Returning from getXmlFilenameForPlateBarcode');
	return $filenameToImport;
}

/**
 * @param $barcode
 * @param bool $isCommissioning
 * @return bool
 * @throws BadRequestException
 * @throws NotFoundException
 * @throws ServerException
 * @throws Exception
 */
function importInspectionsForPlateBarcode($barcode, $isCommissioning=false){
	global $imageStore, $thumbStore, $lastModifiedTimestampCutoffImages;
	writeLog(LOGLEVEL_DEBUG, 'In importInspectionsForPlateBarcode, barcode="'.$barcode.'"...');
	if(empty(trim($barcode))){
	    writeLog(LOGLEVEL_WARN, 'Empty plate barcode - cannot determine plate');
	    writeLog(LOGLEVEL_DEBUG, 'Returning from importInspectionsForPlateBarcode');
	    return false;
	}
	if(!file_exists($imageStore.'/'.$barcode)){
	    writeLog(LOGLEVEL_WARN, 'No images directory for '.$barcode.' - not importing inspections');
	    writeLog(LOGLEVEL_DEBUG, 'Returning from importInspectionsForPlateBarcode');
	    return false;
	}
	if(!file_exists($thumbStore.'/'.$barcode)){
	    writeLog(LOGLEVEL_WARN, 'No thumbnails directory for '.$barcode.' - not importing inspections');
	    writeLog(LOGLEVEL_DEBUG, 'Returning from importInspectionsForPlateBarcode');
	    return false;
	}
	$limsPlate=plate::getByName($barcode);
	if(!$limsPlate){
		writeLog(LOGLEVEL_ERROR, 'No plate in IceBear with barcode '.$barcode.', cannot import images');
		throw new Exception('No plate in IceBear with barcode '.$barcode.', cannot import images');
	}
	$filenamesToImport=array();
	$dir=dir(rtrim($imageStore,'/').'/'.$barcode);
	$filename=$dir->read();
	while($filename){
	    if(substr($filename, -4)==='.xml'){
			if($isCommissioning || filemtime($dir->path.'/'.$filename)>$lastModifiedTimestampCutoffImages){
				$filenamesToImport[]=$filename;
			}
		}
		$filename=$dir->read();
	}
	if(empty($filenamesToImport)){
	    writeLog(LOGLEVEL_WARN, 'No inspections for '.$barcode.' need importing');
	} else {
	    $doImport=true;
	    if($isCommissioning){
	        $numImagerInspections=count($filenamesToImport);
	        $limsInspections=imagingsession::getbyproperty('plateid',$limsPlate['id']);
	        $numLimsInspections=1*($limsInspections['total']);
	        writeLog(LOGLEVEL_DEBUG, 'Commissioning import found '.$numLimsInspections.' in IceBear, '.$numImagerInspections.' in Rigaku');
	        if($numImagerInspections<=$numLimsInspections){
	           writeLog(LOGLEVEL_WARN, 'Not re-importing inspections for this plate');
	           $doImport=false;
	        }
	    }
	    if($doImport){
    	    foreach($filenamesToImport as $filename){
    	        importInspectionFromXml($barcode, $filename);
    	    }
	    }
	}
	writeLog(LOGLEVEL_DEBUG, 'Returning from importInspectionsForPlateBarcode');
	return true;
}

/**
 * Imports a group of images described in an XML file, as one or more IceBear plate inspections.
 *
 * Approach:
 *
 * Iterate through XML image elements, building an array of image paths for each imaging type (UV, Enhanced, etc.) found.
 * When all image paths have been determined, and the images found to exist, create one IceBear imagingsession for
 * each imaging type and either
 * - copy the images and thumbnails to the IceBear image store (if rigaku_imagepath does not begin with core_imagestore)
 * - point the database at the existing images in the Rigaku store
 *
 * Only composite ("extended focus" in Formulatrix terms) or best-slice images are imported, with composite being
 * preferred. Should any image be newer than the minimum age, import of the entire XML file is aborted; composite images
 * may not have finished generating.
 *
 * The XML files do not directly describe the composite image, so we have to munge the "best slice" image name to find it.
 *
 * @param string $barcode The plate barcode.
 * @param string $filename The name of the XML file describing an inspection.
 * @return bool
 * @throws Exception
 * @noinspection PhpUndefinedFieldInspection - Prevents PHPStorm warning for "magic method" access on XML properties
*/
function importInspectionFromXml($barcode, $filename){
	global $imageStore, $thumbStore, $limsImageStore, $imagingTypes, $imager;
	writeLog(LOGLEVEL_DEBUG, 'In importInspectionFromXml');
	writeLog(LOGLEVEL_DEBUG, 'barcode='.$barcode.', filename='.$filename);
	if(empty(trim($barcode))){
	    writeLog(LOGLEVEL_WARN, 'Empty plate barcode - cannot determine plate');
	    writeLog(LOGLEVEL_DEBUG, 'Returning from importInspectionFromXml');
	    return false;
	}
	if(!file_exists($imageStore.'/'.$barcode.'/'.$filename)){
		throw new Exception('No inspection XML '.$filename.' for '.$barcode);
	}
	$limsPlate=plate::getByName($barcode);
	if(!$limsPlate){
		throw new Exception('No plate in IceBear with barcode '.$barcode.', cannot import images');
	}
	
	//All records under this plate need to share its project ID. Imagers, plate types, etc., go under the Shared project.
	$projectId=$limsPlate['projectid'];
	
	$now=time();
	$tooNew=false;
	$xml=simplexml_load_file($imageStore.'/'.$barcode.'/'.$filename);
    $images=$xml->children('rs',true)->data->children('z',true);
	$inspectionNumber=null;
	$inspections=array(); //will hold a list of images against their imaging type
	$lastModifiedTime=null;
	foreach($images as $i){
		$imageName=$i->attributes()->DROP_IMAGE_NAME;
		$row=$i->attributes()->ROW; //Zero-based!
		$col=$i->attributes()->COL; //Zero-based!
		$dropNumber=$i->attributes()->SUBWELL;
		$imagePlateBarcode=$i->attributes()->BARCODE_ID;
		writeLog(LOGLEVEL_DEBUG, 'Image name is '.$imageName);
		if($barcode!=$imagePlateBarcode){
			throw new Exception('Image with name '.$imageName.' is not for plate '.$barcode);
		}
		if(null==$inspectionNumber){
			$inspectionNumber=1*($i->attributes()->INSPECT_NUMBER);
		} else if($inspectionNumber!=$i->attributes()->INSPECT_NUMBER){
			throw new Exception('Image with name '.$imageName.' is not for inspection '.$inspectionNumber);
		}
		$inspectionImageStore=$imageStore.$barcode.'/'.$inspectionNumber.'/';
		$inspectionThumbStore=$thumbStore.$barcode.'/'.$inspectionNumber.'/Thumbnails/160/';
		
		//RM01_03_000_0213_Proj1_Clon1_MC_0000MC000518_002_150730_01_01_02_00_99_031_001_RAI
		//Discard everything up to and including the barcode and its trailing underscore: 002_150730_01_01_02_00_99_031_001_RAI
		//Explode on underscores.
		//002 - Second inspection on this plate?
		//150730 - Date of inspection in YYMMDD format. If you're reading this in the year 2100, sorry. Wasn't me.
		//01 - Row (1-based)
		//01 - Col (1-based)
		//02 - Drop (1-based)
		//00 - Imaging profile, see $imagingTypes near top of this file
		//99 - Slice number. 99 is (a copy of) the best slice, 00 is a composite ("Extended-focus image" in Formulatrix) and the one we want.
		//031 - Unknown
		//001 - Unknown
		//RAI - Unknown
		$filenameParts=explode('_', explode($barcode.'_', $imageName)[1]);
		
		//Interested in composite images, but those aren't described in the XML.
		//Find the best slice and modify its filename to get that of the composite.
		if("99"!=$filenameParts[6]){
			writeLog(LOGLEVEL_DEBUG, 'Image is not composite or best slice, not importing');
			continue;
		}
		writeLog(LOGLEVEL_DEBUG, 'Image is composite or best slice, importing');

		if(!array_key_exists('profile'.$filenameParts[5], $inspections)){
			$inspections['profile'.$filenameParts[5]]=array();
			$inspectionArray=&$inspections['profile'.$filenameParts[5]];
			$inspectionArray['images']=array();
			$inspectionArray['profile']=$filenameParts[5];
		}

		//First we look for the "composite" / "extended focus" image, by changing the _99_ in the best-slice image name to _00_
        $compositeImageName=str_replace('_99_', '_00_', $imageName);
		$imagePath=$inspectionImageStore.$compositeImageName.'.jpg';
		$thumbPath=$inspectionThumbStore.$compositeImageName.'_160.jpg';

		//For some profiles, the imager doesn't generate a composite image, only the best slice is available.
        //If either the full-size or thumbnail composite is missing, fall back to the best-slice filenames
        if(!file_exists($imagePath) || !file_exists($thumbPath)) {
            writeLog(LOGLEVEL_DEBUG, 'Did not find image and thumbnail for composite. Falling back to best slice.');
            $imagePath=$inspectionImageStore.$imageName.'.jpg';
            $thumbPath=$inspectionThumbStore.$imageName.'_160.jpg';
        }

        //If neither composite nor best-slice has both full-size and thumbnail, warn and don't import.
        if(!file_exists($imagePath) || !file_exists($thumbPath)){
			writeLog(LOGLEVEL_WARN, 'Could not find both image and thumbnail for composite, or for best slice. Not importing image.');
            writeLog(LOGLEVEL_DEBUG, 'Best-slice image path: '.$inspectionImageStore.$imageName.'.jpg');
            writeLog(LOGLEVEL_DEBUG, 'Best-slice image exists: '.(file_exists($inspectionImageStore.$imageName.'.jpg')?'Yes':'No'));
            writeLog(LOGLEVEL_DEBUG, 'Best-slice thumbnail path: '.$inspectionThumbStore.$imageName.'_160.jpg');
            writeLog(LOGLEVEL_DEBUG, 'Best-slice thumbnail exists: '.(file_exists($inspectionThumbStore.$imageName.'_160.jpg')?'Yes':'No'));
            writeLog(LOGLEVEL_DEBUG, 'Composite image path: '.$inspectionImageStore.$compositeImageName.'.jpg');
            writeLog(LOGLEVEL_DEBUG, 'Composite image exists: '.(file_exists($inspectionImageStore.$compositeImageName.'.jpg')?'Yes':'No'));
            writeLog(LOGLEVEL_DEBUG, 'Composite thumbnail path: '.$inspectionThumbStore.$compositeImageName.'_160.jpg');
            writeLog(LOGLEVEL_DEBUG, 'Composite thumbnail exists: '.(file_exists($inspectionThumbStore.$compositeImageName.'_160.jpg')?'Yes':'No'));
			continue;
		}

        writeLog(LOGLEVEL_DEBUG, 'Found image and thumbnail');
        writeLog(LOGLEVEL_DEBUG, 'Image: '.$imagePath);
        writeLog(LOGLEVEL_DEBUG, 'Thumb: '.$thumbPath);
		$inspectionArray['images'][]=array(
				'image'=>$imagePath,
				'thumb'=>$thumbPath,
				'row'=>1+(int)$row,
				'col'=>1+(int)$col,
				'sub'=>(int)$dropNumber,
		);
		
		$lastModifiedTime=filemtime($imagePath);
		if($now-(60*MINIMUM_INSPECTION_AGE_MINS)<$lastModifiedTime){
			$tooNew=true;
			break;
		}
		if(!isset($inspectionArray['datetime'])){
			$datetime=gmdate('Y-m-d h:i:s', $lastModifiedTime);
			$inspectionArray['datetime']=$datetime;
		}
	}
	
	if($tooNew){
		writeLog(LOGLEVEL_WARN, 'Not importing this inspection yet. Found image less than '.MINIMUM_INSPECTION_AGE_MINS.' minutes old.');
		writeLog(LOGLEVEL_DEBUG, 'Returning from importInspectionFromXml');
		writeLog(LOGLEVEL_INFO, '-------------------------------------------------');
		return false;
	}
	$imagedTime=$lastModifiedTime;
	foreach($imagingTypes as $label=>$type){
		writeLog(LOGLEVEL_DEBUG, $label);
		if(!isset($inspections['profile'.$label])){ continue; }
		$found=$inspections['profile'.$label];
		if(empty($found['images'])){ continue; }
		writeLog(LOGLEVEL_DEBUG, count($found['images']).' images found for '.$label);
		
		//Create the imagingsession
		$date=gmdate('Y-m-d H:i:s',$imagedTime);
		$imagedTime--; //Import the next imaging profile with timestamp a second before this one
		
		$imagingSessionName=$barcode.'_'.$date.'_profile'.$type['limsParametersVersion']['id'];
		$limsImagingSession=imagingsession::getByName($imagingSessionName);
		if(!empty($limsImagingSession)){
			writeLog(LOGLEVEL_DEBUG, 'LIMS imagingsession already exists');
		} else {
			writeLog(LOGLEVEL_DEBUG, 'LIMS imagingsession was not found, creating it');
			$params=array(
					'name'=>$imagingSessionName,
					'manufacturerdatabaseid'=>0,
					'imagerid'=>$imager['id'],
					'plateid'=>$limsPlate['id'],
					'imageddatetime'=>$date,
					'lighttype'=>$type['light'],
					'imagingparametersversionid'=>$type['limsParametersVersion']['id']
			);
			imagingsession::create($params);
			$limsImagingSession=imagingsession::getByName($imagingSessionName);
		}
		if(!$limsImagingSession){
			throw new Exception('Could not find imagingsession in LIMS after create');
		}
		$limsImagingSessionId=$limsImagingSession['id'];
		
		$copyImagesToStore=true;
        $destinationdir=substr($barcode,0,4).'/'.substr($barcode,0,6).'/'.$barcode.'/imagingsession'.$limsImagingSession['id'].'/';

		if(!empty(trim($limsImageStore, '/')) && stripos($imageStore, $limsImageStore)===0){
		    writeLog(LOGLEVEL_DEBUG, 'LIMS and Rigaku image stores appear common, will use Rigaku images in place (no copy)');
		    $copyImagesToStore=false;
		} else {
		    writeLog(LOGLEVEL_DEBUG, 'LIMS and Rigaku image stores appear different, will copy Rigaku images into LIMS image store');
		    //Create the directory of images from this imagingsession if it does not exist
		    //Should be at image_store/MC00/MC0005/MC000518/imagingsession1234/thumbs/
		    writeLog(LOGLEVEL_DEBUG, 'Checking for imagingsession directory in LIMS image store...');
		    @mkdir($limsImageStore.$destinationdir.'thumbs',0755,true);
		    if(!file_exists($limsImageStore.$destinationdir.'thumbs')){
		        writeLog(LOGLEVEL_ERROR, 'Path does not exist: '.$limsImageStore.$destinationdir.'thumbs');
		        throw new Exception('Could not create destination directory in LIMS image store');
		    }
		    writeLog(LOGLEVEL_DEBUG, '...exists.');
		}
		
		
		foreach($found['images'] as $i){

		    if($copyImagesToStore){
    			writeLog(LOGLEVEL_DEBUG, 'Copying images to LIMS store');
    			$imageDestination=$destinationdir.platetype::$rowLabels[(int)($i['row'])].str_pad($i['col'],2,'0',STR_PAD_LEFT).'.'.$i['sub'].'.jpg';
    			$thumbDestination=$destinationdir.'thumbs/'.platetype::$rowLabels[(int)($i['row'])].str_pad($i['col'],2,'0',STR_PAD_LEFT).'.'.$i['sub'].'.jpg';
    			copy($i['image'], $limsImageStore.$imageDestination);
    			copy($i['thumb'], $limsImageStore.$thumbDestination);
    			if(!file_exists($limsImageStore.$imageDestination)){
    				writeLog(LOGLEVEL_ERROR, 'Compound image not copied: '.$limsImageStore.$imageDestination);
    				throw new Exception('Could not create compound image in LIMS image store');
    			}
    			if(!file_exists($limsImageStore.$thumbDestination)){
    				writeLog(LOGLEVEL_ERROR, 'Thumbnail image not copied: '.$limsImageStore.$thumbDestination);
    				throw new Exception('Could not create thumbnail in LIMS image store');
    			}
		    } else {
		        $imageDestination=ltrim(str_replace($limsImageStore, '', $i['image']),'/');
		        $thumbDestination=ltrim(str_replace($limsImageStore, '', $i['thumb']),'/');
		    }
			
			$imageSize=getimagesize($i['image']);
			if(!$imageSize){
				writeLog(LOGLEVEL_WARN, ''.$i['image'].' may be corrupt, could not get pixel dimensions');
				writeLog(LOGLEVEL_WARN, 'Not linking image '.$i['image'].' in LIMS database');
			} else {
				$imageWidth=$imageSize[0];
				$imageHeight=$imageSize[1];
				writeLog(LOGLEVEL_DEBUG, 'Creating database record of image...');
				$imageDbName=$barcode.'_'.platetype::$rowLabels[(int)($i['row'])].str_pad($i['col'],2,'0',STR_PAD_LEFT).'.'.$i['sub'].'_is'.$limsImagingSession['id'];
				
				
				$preExisting=dropimage::getByName($imageDbName);
				if(!$preExisting){
					$welldropName=$barcode.'_'.platetype::$rowLabels[(int)($i['row'])].str_pad($i['col'],2,'0',STR_PAD_LEFT).'.'.$i['sub'];
					$welldrop=welldrop::getByName($welldropName);
					if(!$welldrop){
						throw new Exception('welldrop not found in LIMS for '.$welldropName);
					}
					dropimage::create(array(
							'name'=>$imageDbName,
							'projectid'=>$projectId,
							'imagingsessionid'=>$limsImagingSessionId,
							'welldropid'=>$welldrop['id'],
							'pixelheight'=>$imageHeight,
							'pixelwidth'=>$imageWidth,
							'micronsperpixelx'=>IMAGER_MICRONS_PER_PIXEL,
							'micronsperpixely'=>IMAGER_MICRONS_PER_PIXEL,
							'imagestorepath'=>$limsImageStore,
							'imagepath'=>$imageDestination,
							'thumbnailpath'=>$thumbDestination
					));
					carryForwardScores($welldrop['id']);
				}
			}
			
		}
		
	}
	writeLog(LOGLEVEL_DEBUG, 'Returning from importInspectionFromXml');
	writeLog(LOGLEVEL_INFO, '-------------------------------------------------');
	return true;
}


/**
 * Creates a record in the LIMS of the imager with the specified serial/name.
 * Capacity calculations may be somewhat complicated by the existence in the Rigaku DB of a second hotel that is not installed.
 * Note that IceBear determines an imagingsession temperature from that of the imager, so a twin +4/+20 Rigaku with a shared
 * camera will need to be treated as two separate imagers. This is NOT SUPPORTED at present. @TODO Support multiple temperatures.
 * @param string $imagerName The imager name.
 * @return array|mixed
 * @throws Exception
 */
function getImager($imagerName){
	writeLog(LOGLEVEL_DEBUG, 'In importImager, imagerName='.$imagerName);
	$imager=imager::getByName($imagerName);
	if(!$imager){
		writeLog(LOGLEVEL_WARN, 'Imager "'.$imagerName.'" does not exist, creating it');
		$capacity=500;
		//TODO Calculate capacity from table structure
		//Looks like Hotels are swappable, Helsinki's has Linbro and SBS? 
		$imager=imager::create(array(
				'name'=>$imagerName,
				'friendlyname'=>$imagerName,
				'manufacturer'=>'Rigaku',
				'temperature'=>20,
				'platecapacity'=>$capacity,
				'alertlevel'=>floor($capacity*IMAGERLOAD_ALERT_THRESHOLD_PERCENT/100),
				'warninglevel'=>floor($capacity*IMAGERLOAD_WARNING_THRESHOLD_PERCENT/100)
		));
		$imager=$imager['created'];
	}
	writeLog(LOGLEVEL_DEBUG, 'Returning from importImager');
	return $imager;
}


/**
 * Creates a record in the LIMS of the plate with the specified barcode.
 * If the plate type does not exist in IceBear, it is created.
 * If the plate owner does not exist in IceBear, the user is created.
 * If screen information is present in the XML, a screen is created, or an existing one is associated with the plate.
 * @param string $barcode The plate barcode.
 * @param bool $isCommissioning
 * @return array the plate
 * @throws BadRequestException
 * @throws ForbiddenException
 * @throws NotFoundException
 * @throws NotModifiedException
 * @throws ServerException
 * @throws Exception
 */
function importPlateByBarcode($barcode, $isCommissioning=false){
	writeLog(LOGLEVEL_DEBUG, 'In importPlateByBarcode, barcode='.$barcode);
	$filename=getXmlFilenameForPlateBarcode($barcode);
	$plate=importPlateByFilename($filename, $isCommissioning);
	writeLog(LOGLEVEL_DEBUG, 'Returning from importPlateByBarcode');
	return $plate;
}

/**
 * Creates a record in the LIMS of a plate, by parsing the specified Rigaku plate XML..
 * If the plate type does not exist in IceBear, it is created.
 * If the plate owner does not exist in IceBear, the user is created.
 * If screen information is present in the XML, a screen is created, or an existing one is associated with the plate.
 * @param $filename
 * @param bool $isCommissioning
 * @return array|boolean the plate, or false if the file does not look like plate XML.
 * @throws BadRequestException
 * @throws ForbiddenException
 * @throws NotFoundException
 * @throws NotModifiedException
 * @throws ServerException
 * @throws Exception
 * @noinspection PhpUndefinedFieldInspection - Prevents PHPStorm warning for "magic method" access on XML properties
 */
function importPlateByFilename($filename, $isCommissioning=false){
	global $lastModifiedTimestampCutoffPlates, $plateStore, $shouldntOwnPlates, $scoringSystem;
	writeLog(LOGLEVEL_DEBUG, 'In importPlateByFilename, filename='.$filename);
	if(!file_exists($plateStore.$filename)){
		throw new Exception('Plate filename '.$filename.' not found in plate XML store');
	}
	if(stripos($filename, 'Plate_')!==0){
	    writeLog(LOGLEVEL_WARN, 'Not attempting to import file: '.$filename);
	    writelog(LOGLEVEL_WARN, 'File does not look like a plate');
	    return false;
	}
	if(!$isCommissioning && 0!=$lastModifiedTimestampCutoffPlates && filemtime($plateStore.$filename)<$lastModifiedTimestampCutoffPlates){
		writeLog(LOGLEVEL_WARN, 'File was not modified recently enough. Ignoring.');
		writeLog(LOGLEVEL_DEBUG, 'Returning from importPlateByFilename, nothing to do');
		return false;
	}
	$xml=@simplexml_load_file($plateStore.$filename);
	if(false===$xml){
		throw new Exception('Could not parse plate XML '.$filename);
	}
	$xmlPlate=$xml->plate->attributes();
	$barcode=$xmlPlate['barcode'];
	$xmlPlateType=$xml->plate->format->attributes();
	
	if(''==$xmlPlate->temperature){
	    writeLog(LOGLEVEL_WARN, 'Plate temperature: not specified. Assuming '.IMAGER_TEMP);
	} else if(IMAGER_TEMP!=$xmlPlate->temperature){
	    writeLog(LOGLEVEL_WARN, 'Plate is at wrong temperature: '.$xmlPlate->temperature.' - should be '.IMAGER_TEMP);
		//throw new Exception('Plate is at wrong temperature: '.$xmlPlate->temperature.' - should be '.IMAGER_TEMP);
	}
	
	if(''==trim($barcode)){
	    writeLog(LOGLEVEL_WARN, 'No plate barcode in XML file. Not importing plate from XML.');
	    writeLog(LOGLEVEL_DEBUG, 'Returning from importPlateByFilename');
	    return false;
	}
	
	$plate=plate::getByName($barcode);
	if($plate){

		writeLog(LOGLEVEL_DEBUG, 'Plate '.$barcode.' exists in IceBear with ID '.$plate['id']);
		//We can assume that plate type and user already exist in IceBear
		$projectId=$plate['projectid'];
		
	} else {
		writeLog(LOGLEVEL_INFO, 'Plate '.$barcode.' does not exist in IceBear, need to create it');

		//check for plate type
		$plateType=getPlateType($xmlPlateType->name, $xmlPlateType->rows, $xmlPlateType->cols, $xmlPlateType->subs);
		platetype::update($plateType['id'], array('defaultdropsize'=>$xmlPlateType->def_drop_vol.'uL'));
		
		//check for user
		$username=trim($xmlPlate->user);
		if(empty($username) || in_array($username, $shouldntOwnPlates)){
			writeLog(LOGLEVEL_WARN, 'Username '.$username.' should not own plates, finding first username in IceBear');
			//First user ought to be system administrator - username may vary
			$user=user::getFirstAdmin();
			$username=$user['name'];
			writeLog(LOGLEVEL_WARN, 'Setting plate '.$barcode.' owner to '.$username.'');
		}
		$plateOwner=getUser($username);		
		
		//check for project
		try {
			$project=project::getByName($xmlPlate->project);
		} catch(NotFoundException $e){
		    try {
    			$project=project::create(array(
    					'name'=>$xmlPlate->project->__toString(),
    					'owner'=>$plateOwner['id'],
    					'description'=>'Created automatically by Rigaku importer'
    			));
    			$project=$project['created'];
		    } catch(Exception $e2){
		        session::set('isAdmin',true);
		        $project=project::getByName($xmlPlate->project->__toString());
		    }
		}
		$projectId=$project['id'];
		
		//create the plate
		$plate=plate::create(array(
				//'screenid'=>whatever,
				'name'=>$barcode,
				'description'=>$xmlPlate->name,
				'ownerid'=>$plateOwner['id'],
				'projectid'=>$projectId,
				'platetypeid'=>$plateType['id'],
				'crystalscoringsystemid'=>$scoringSystem['id']
		));
		$plate=$plate['created'];
		writeLog(LOGLEVEL_INFO, 'Plate created with ID '.$plate['id']);
	}		

	$xmlScreen=$xml->screen;
	$screenName='';
	if(empty($xmlScreen)){
		writeLog(LOGLEVEL_DEBUG, 'XML has no screen element');
		writeLog(LOGLEVEL_INFO,'No screen info attached to plate. Screen will need to be set manually.');
	} else {
		writeLog(LOGLEVEL_DEBUG, 'XML has screen element, parsing');
		$screenName=$xmlScreen->attributes()->name;
		$limsScreen=screen::getByName($screenName);
		if(!empty($limsScreen)){
				
			writeLog(LOGLEVEL_INFO,'Screen with name '.$screenName.' exists in LIMS. Using existing screen.');

		} else {

			//Have not seen the screen before. Assume optimization screen.
			//Parse screen XML, create screen and conditions.
			writeLog(LOGLEVEL_INFO,'Screen with name '.$screenName.' does not exist in LIMS. Parsing screen XML.');
			
			$screenRows=$xmlScreen->format->attributes()->rows;
			$screenCols=$xmlScreen->format->attributes()->cols;
			
			$limsScreen=screen::create(array(
					'projectid'=>$projectId,
					'name'=>$screenName,
					'rows'=>$screenRows,
					'cols'=>$screenCols
			));
			$limsScreen=$limsScreen['created'];
			
			$conditionsFound=0;
			foreach($xmlScreen->children() as $c){
				if("well"!=$c->getName()){ continue; } //not the format or comments element
				$wellNumber=$c->attributes()->number;
				$conditionParts=array();
				$parts=$c->children();
				foreach($parts as $p){
					$part=$p->attributes()->class.': '. $p->attributes()->conc . $p->attributes()->units.' '.$p->attributes()->name;
					$pH=$p->attributes()->ph;
					if(!empty($pH) && ""!=$pH){ $part.=', pH '.$pH; }
					$conditionParts[]=$part;
				}
				$condition=implode('; ', $conditionParts);
				$row=floor(($wellNumber-1)/$screenCols)+1;
				$col=floor(($wellNumber-1)%$screenCols)+1;
				writeLog(LOGLEVEL_DEBUG, 'Condition in well '.$wellNumber.' is '.$condition);
				writeLog(LOGLEVEL_DEBUG, 'Updating dummy condition in row '.$row.' col '.$col.'...');
				$existingCondition=screencondition::getByName('screen'.$limsScreen['id'].'_well'.str_pad($wellNumber, 3, '0', STR_PAD_LEFT));
				screencondition::update($existingCondition['id'], array(
						'description'=>$condition
				));
				writeLog(LOGLEVEL_DEBUG, '...created');
				$conditionsFound++;
			}
			if($conditionsFound!=$screenCols*$screenRows){
			    writeLog(LOGLEVEL_WARN, 'Screen has '.$conditionsFound.' conditions, expected '.$screenCols*$screenRows.' ('.$screenCols.'x'.$screenRows.')');
			}
			writeLog(LOGLEVEL_DEBUG, 'Finished creating screen');
		}
		
		//Lastly, attach the screen to the plate.
		writeLog(LOGLEVEL_DEBUG, 'Attaching screen '.$limsScreen['id'].' to plate '.$plate['id'].'...');
		plate::update($plate['id'], array('screenid'=>$limsScreen['id']));
		writeLog(LOGLEVEL_DEBUG, 'Finished creating screen');
	}

	//If more than one plate has the screen found, promote it to a standard screen
	//(If the screen is newly created, this almost certainly won't happen.)
	if(!empty($limsScreen)){
		$platesWithScreen=plate::getByProperty('screenid', $limsScreen['id']);
		if(!empty($platesWithScreen) && isset($platesWithScreen['rows']) && count($platesWithScreen['rows'])>1){
			writeLog(LOGLEVEL_INFO,'More than one plate has screen '.$limsScreen['name'].'.');
			writeLog(LOGLEVEL_INFO,'Promoting '.$screenName.' to standard screen.');
			screen::update($limsScreen['id'], array(
					'isstandard'=>true,
			));
		} else {
            //Leave as optimization screen
            writeLog(LOGLEVEL_INFO,'First time seeing screen '.$limsScreen['name'].'.');
            writeLog(LOGLEVEL_INFO,'Assuming optimization screen. Will promote to standard screen if seen again.');
		}
	}
	
	writeLog(LOGLEVEL_DEBUG, 'Returning from importPlateByFilename');
	return $plate;
}

/**
 * Creates the Rigaku scoring system in the LIMS, if it does not exist
 * @throws BadRequestException
 * @throws ForbiddenException
 * @throws NotFoundException
 * @throws ServerException
 * @throws Exception
 */
function importScoringSystem(){
	writeLog(LOGLEVEL_DEBUG, 'In importScoringSystem');
	
	$existing=crystalscoringsystem::getByName('Rigaku');
	if($existing){ return $existing; }
	
	$system=crystalscoringsystem::create(array(
		'name'=>'Rigaku'
	));
	if(!$system){ throw new Exception('Could not create scoring system'); }
	$system=$system['created'];
	
	$scores=array(
			array('0','0','Clear','ffffff'),
			array('1','9','Crystals','ff0000'),
			array('2','8','Micro-crystals','ffff00'),
			array('3','7','Crystalline','999999'),
			array('4','6','Clusters','999999'),
			array('5','5','Spherulites','999999'),
			array('6','4','Precipitation','999999'),
			array('7','3','Heavy precipitation','999999'),
			array('8','2','Phase separation','999999'),
			array('9','1','Matter','999999'),
	);
	$index=0;
	foreach($scores as $s){
		$request=array(
				'crystalscoringsystemid'=>$system['id'],
				'hotkey'=>$s[0],
				'scoreindex'=>$s[1],
				'color'=>$s[3],
				'label'=>$s[2],
				'name'=>'Rigaku_'.$s[2],
		);
		crystalscore::create($request);
		$index++;
	}
	writeLog(LOGLEVEL_DEBUG, 'returning from importScoringSystem');
	return $system;
}


/**
 * Retrieves the IceBear platetype with the specified name, or creates and returns it.
 * @param $name
 * @param $rows
 * @param $cols
 * @param $subs
 * @return array|mixed
 * @throws BadRequestException
 * @throws ForbiddenException
 * @throws NotFoundException
 * @throws ServerException
 * @throws Exception
 */
function getPlateType($name, $rows, $cols, $subs){
	writeLog(LOGLEVEL_DEBUG, 'In getPlateType, name='.$name);
	$plateType=platetype::getByName($name);
	if(!$plateType){
		//create it
		writeLog(LOGLEVEL_WARN, 'Plate type '.$name.' does not exist in LIMS, creating it');
		$sharedProject=project::getByName('Shared');
		$dropMapping='1,R';
		if($subs>1){
			$top='';
			$bottom='';
			for($i=1;$i<=$subs;$i++){
				$top.=$i;
				$bottom.='R';
			}
			$dropMapping=$top.','.$bottom;
		}
		$created=platetype::create(array(
			'name'=>$name,
			'rows'=>$rows,
			'cols'=>$cols,
			'subs'=>$subs,
			'dropmapping'=>$dropMapping,
			'projectid'=>$sharedProject['id'],
		));
		$plateType=$created['created'];
		writeLog(LOGLEVEL_DEBUG, 'Plate type '.$name.' created in LIMS with ID '.$plateType['id']);
	}
	if($rows!=$plateType['rows'] || $cols!=$plateType['cols'] || $subs!=$plateType['subs']){
		$msg='Plate type geometry mismatch.';
		$msg.=' IceBear: '.$plateType['rows'].'x'.$plateType['cols'].'x'.$plateType['subs'];
		$msg.=' Plate XML: '.$rows.'x'.$cols.'x'.$subs;
		throw new Exception($msg);
	}
	writeLog(LOGLEVEL_DEBUG, 'Returning from getPlateType');
	return $plateType;
}


/**
 * Returns the LIMS user with the specified username, or creates and returns it.
 * The user is created in an inactive state, with a bogus email address and name, and will be unable to log in
 * until an administrator activates their account.
 * @param string $username The user's Rigaku username
 * @return array|mixed
 * @throws BadRequestException
 * @throws ForbiddenException
 * @throws NotFoundException
 * @throws ServerException
 */
function getUser($username){
	writeLog(LOGLEVEL_DEBUG, 'In getUser, username='.$username);
	$user=user::getByName($username);
	if(!$user){
		writeLog(LOGLEVEL_WARN, 'User '.$username.' does not exist in LIMS, creating it');
		$user=user::create(array(
				'name'=>$username,
				'fullname'=>$username,
				'email'=>$username.'@bogus.bogus',
				'password'=>'USELESSPASSWORD',
				'isactive'=>0
		));
		$user=$user['created'];
	} else {
		writeLog(LOGLEVEL_DEBUG, 'User exists in IceBear');
	}
	writeLog(LOGLEVEL_DEBUG, 'Returning from getUser');
	return $user;
}



/**
 * Echoes a log message to the output, prefixed by a level indicator and date/time.
 * @param int $level The log level, see top of this file for definitions
 * @param string $message The log message
 */

function writeLog($level, $message){
	global $logLevel;
	if($level<$logLevel){ return; }
	$labels=array(
		LOGLEVEL_DEBUG=>'DEBUG',
		LOGLEVEL_INFO =>'INFO ',
		LOGLEVEL_WARN =>'WARN ',
		LOGLEVEL_ERROR=>'ERROR',
	);
	$out=$labels[$level].' '.gmdate('d M y H:i:s').' '.$message."\n";
	echo $out;
}

function setUpClassAutoLoader(){
	spl_autoload_register(function($className){
		$paths=array(
				'../classes/',
				'../classes/core/',
				'../classes/core/exception/',
				'../classes/core/authentication/',
				'../classes/core/interface/',
				'../classes/model/',
		);
		foreach($paths as $path){
			if(file_exists($path.$className.'.class.php')){
                /** @noinspection PhpIncludeInspection */
                include_once($path.$className.'.class.php');
			}
		}
	});
}


function showHelp(){
	echo "\nImporter help\n\n";
	echo "By default this will (re-)import all images and plates created or modified within the last 24 hours.\n";
	echo "\nThe following arguments can be supplied to modify this behaviour:\n";
	echo "\n -h Show help";
	echo "\n -c Commissioning import - import ALL found plates and images";
	echo "\n -bBARCODE - (Re)import all inspections and images for this barcode - not compatible with other options";
	exit;	
}


/**
 * Carries forward a score from a previous of this drop to the new image.
 * For the given well drop, finds the most recent image with a score and applies that
 * score to all subsequent images. Where there are multiple scores on that image, the
 * most recent is applied to the subsequent images.
 * @param int $limsWellDropId The ID of the well drop in IceBear
 * @throws BadRequestException
 * @throws ForbiddenException
 * @throws NotFoundException
 * @throws ServerException
 */
function carryForwardScores($limsWellDropId){
    Log::write(Log::LOGLEVEL_DEBUG, 'Carrying forward scores for LIMS drop ID '.$limsWellDropId);
    $timeCourseImages=welldrop::gettimecourseimages($limsWellDropId);
    if(!empty($timeCourseImages && isset($timeCourseImages['rows']))){
        $timeCourseImages=array_reverse($timeCourseImages['rows']);
        $hasUnscoredAtEnd=false;
        $lastScoreId=null;
        foreach($timeCourseImages as $t){
            if(""==$t['latestcrystalscoreid']){
                $hasUnscoredAtEnd=true;
            } else {
                $lastScoreId=$t['latestcrystalscoreid'];
                break;
            }
        }
        if($hasUnscoredAtEnd && !empty($lastScoreId)){
            foreach($timeCourseImages as $t){
                if(""==$t['latestcrystalscoreid']){
                    dropimage::update($t['id'], array('latestcrystalscoreid'=>$lastScoreId));
                } else {
                    break;
                }
            }
            
        }
    }
    Log::write(Log::LOGLEVEL_DEBUG, 'Finished carrying forward scores');
}

