window.SynchwebShippingHandler={
		
		/**
		 * The base URI of the SynchWeb instance. 
		 * Populated from the shipmentdestination's baseuri property.
		 */
		baseUri:null,
		
		/**
		 * The authentication token to be passed to the synchrotron with each request after authentication.
		 * Received from the synchrotron on authentication.
		 */
		token:null,

		safetyLevels:[
			{ "code":"Green", "risk":"Low", "background":"#cfc", "color":"#060"},
			{ "code":"Yellow", "risk":"Medium", "background":"#ffc", "color":"#660"},
			{ "code":"Red", "risk":"High", "background":"#fcc", "color":"#600"}
		],

		/*
		 * These are set and used during shipment submission
		 */
		shipmentIdAtFacility:"",
		safetyLevel:null,
		labContact:null,
		proteins:null,
		proposal:null,
		visit:null,

		UNATTENDED:"Unattended",
		
		/**
		 * Returns the URL of this shipment at the remote synchrotron, or false if the shipment has not been submitted.
		 * Strips /api(/) from base API and appends /shipment/sid/[REMOTE ID].
		 */
		getShipmentUrlAtFacility:function(shipmentDestination, shipmentIdAtFacility){
			if(!shipmentIdAtFacility &&!data.idatremotefacility){
				return false;
			} else if(!shipmentIdAtFacility){
				shipmentIdAtFacility=data.idatremotefacility;
			}
			let baseUri=shipmentDestination["baseuri"];
			if(!baseUri){ 
				alert("Cannot get shipment URL. No shipping API base URI set for "+shipmentDestination.name+".");
				return false;
			}
			baseUri=baseUri.replace(/[/]api[/]?$/g,"");
			return baseUri+"/shipments/sid/"+shipmentIdAtFacility;
		},
		
		/**
		 * Returns the URL of this crystal at the remote synchrotron, or false if the crystal does not have a remote ID.
		 * Strips /api(/) from base API and appends /samples/sid/[REMOTE ID].
		 *
		 */
		getCrystalUrlAtFacility:function(shipmentDestination, crystalIdAtFacility){
			if(!crystalIdAtFacility || ""===crystalIdAtFacility){
				return false;
			}
			let baseUri=shipmentDestination["baseuri"];
			if(!baseUri){ 
				alert("Cannot get crystal URL. No shipping API base URI set for "+shipmentDestination.name+".");
				return false;
			}
			baseUri=baseUri.replace(/[/]api[/]?$/g,"");
			return baseUri+"/samples/sid/"+crystalIdAtFacility;
		},
		
		/**
		 * Begins the process of authenticating, choosing a proposal/session, and submitting to 
		 * the synchrotron. 
		 * 
		 * It is assumed that local validation of the shipment has already been done! This includes 
		 * basic checks such as: at least one dewar; each dewar has at least one puck; all crystals
		 * have protein acronyms. Some validation can only be done at the synchrotron, but we should
		 * not be sending them shipments that fail these basic checks.
		 */
		begin:function(){
			if(data["remoteidatfacility"]){
				alert("Shipment has already been sent.");
				return false;
			}
			if(!window.shipmentDestination){ 
				alert("No shipment destination set - cannot determine shipping API location.");
				return false;
			}
			let baseUri=window.shipmentDestination["baseuri"];
			if(!baseUri){ 
				alert("No shipping API base URI set for "+window.shipmentDestination.name+".");
				return false;
			}
			SynchwebShippingHandler.baseUri=baseUri.replace(/\/+$/g,"")+"/";
			//Check still logged into IceBear, reset session clock - don't want to fail at the end.
			new Ajax.Request("/api/homepagebrick", {
				"method":"get",
				"onSuccess":SynchwebShippingHandler.openShippingDialog,
				"onFailure":function(transport){
					if(401===transport.status){
						//IceBear session timed out. Reload page to show login form.
						ui.forceReload();
					} else {
						ui.keepAlive();
						SynchwebShippingHandler.openShippingDialog();
					}
				}
			});
			
		},

		openShippingDialog:function(){
			ui.modalBox({ title:"Send shipment to "+window.shipmentDestination.name, content:"Getting list of proposals from "+window.shipmentDestination.name+"..." });
			SynchwebShippingHandler.getProposals();
		},

		/**
		 * Renders the login form into the modal box.
		 */
		showLoginForm:function(afterAuthenticate){
			let mb=document.getElementById("modalBox");
			mb.querySelector(".boxbody").innerHTML="";
			ui.setModalBoxTitle("Authenticate at "+window.shipmentDestination.name);
			let f=mb.form({
				action:SynchwebShippingHandler.baseUri+"authenticate",
				method:"post",
			});
			f.afterAuthenticate=afterAuthenticate;
			f.style.maxWidth="800px";
			let h=f.formField({
				label:"Authenticate with your "+window.shipmentDestination.name+" credentials", content:'&nbsp;'
			});
			h.classList.add("radiohead");
			f.textField({
				name:'remoteusername',
				label:window.shipmentDestination.name+" username",
				value:""
			});
			f.passwordField({
				name:'remotepassword',
				label:window.shipmentDestination.name+" password",
				value:""
			});
			fieldValidations.remoteusername="required";
			fieldValidations.remotepassword="required";
	
			f.submitButton({ label:"Authenticate" });
			f.onsubmit=function(){ SynchwebShippingHandler.authenticate(); return false; };
			document.getElementById("remoteusername").focus();
		},
		
		/**
		 * Validates that both username and password are present, then submits them to the remote API.
		 * Expects a token in return.
		 */
		authenticate:function(){
			let frm=document.getElementById("modalBox").querySelector(".boxbody form");
			let isValid=true;
			frm.select("input").each(function(f){
				if(!validator.validate(f)){ isValid=false; }
			});
			if(!isValid){ return false; }
			let username = frm.remoteusername.value;
			let password = frm.remotepassword.value;
			frm.querySelector("input[type=submit]").closest("label").classList.add("updating");
			AjaxUtils.remoteAjax(
					SynchwebShippingHandler.baseUri+"authenticate",
					'post',
					'{ "login":"'+username+'", "password":"'+password+'" }', //Note SynchWeb wants JSON as a string, not an object 
					SynchwebShippingHandler.authenticate_onSuccess,
					SynchwebShippingHandler.authenticate_onFailure
			);
			return false;
		},
		/**
		 * Success handler for remote synchrotron authentication. Calls getProposals().
		 * Note that Synchweb returns a 200 OK with the "correct" response code in a status value (i.e., 
		 * transport.responseJSON.status instead of transport.status, which is always 200). We check for
		 * an "error" value and handle anything with it present as a failure.
		 * @param {XMLHttpRequest} transport The response object.
		 */
		authenticate_onSuccess:function(transport){
			if(!transport.responseJSON || !transport.responseJSON["jwt"]){
				if(transport.responseJSON.error){
					SynchwebShippingHandler.authenticate_onFailure(transport);
				} else {
					SynchwebShippingHandler.showErrorAtAuthenticate("Bad response from remote server. Cannot authenticate");
				}
			} else {
				SynchwebShippingHandler.token=transport.responseJSON["jwt"];
				let frm=document.getElementById("modalBox").querySelector(".boxbody form");
				window.setTimeout(frm.afterAuthenticate,50);
			}
		},
		/**
		 * Failure handler for remote synchrotron authentication.
		 * @param transport
		 */
		authenticate_onFailure:function(transport){
			let msg="Could not log you in. The remote server gave no further information.";
			if(transport.responseJSON && transport.responseJSON.error){
				msg=window.shipmentDestination.name+" said: "+transport.responseJSON.error;
			}
			SynchwebShippingHandler.showErrorAtAuthenticate(msg);
		},
		/**
		 * Renders the authentication error message below the form.
		 * @param {String} msg The error message.
		 */
		showErrorAtAuthenticate:function(msg){
			let frm=document.getElementById("modalBox").querySelector(".boxbody").querySelector("form");
			frm.querySelector("input[type=submit]").closest("label").classList.remove("updating");
			let ae=document.getElementById("autherror");
			if(!ae){
				frm.innerHTML+='<div style="text-align:left" id="autherror" class="errorbar"></div>';
			}
			ae.innerHTML=msg;
		},


		/**
		 * Fetches a list of the user's proposals from the synchrotron.
		 * On failure, triggers authentication.
		 */
		getProposals:function(){
			if(!SynchwebShippingHandler.baseUri){ 
				alert("No base URI supplied. Call begin() first, with base URI."); 
				return false;
			}
			if(!SynchwebShippingHandler.token){ 
				SynchwebShippingHandler.showLoginForm(SynchwebShippingHandler.getProposals);
				return false;
			}
			AjaxUtils.remoteAjax(
					SynchwebShippingHandler.baseUri+'proposal?page=1&per_page=50&all=1&next=1',
					'get',
					{},
					SynchwebShippingHandler.getProposals_onSuccess,
					SynchwebShippingHandler.getProposals_onFailure,
					{ 'Authorization':'Bearer '+SynchwebShippingHandler.token }
			);
		},
		getProposals_onSuccess:function(transport){
			if(transport.responseJSON.error){
				//Synchweb returns 200 OK despite error
				return SynchwebShippingHandler.getProposals_onFailure(transport);
			}
			SynchwebShippingHandler.renderProposalList(transport.responseJSON);
		},
		getProposals_onFailure:function(transport){
			let msg="Error getting list of proposals";
			if(transport.responseJSON && transport.responseJSON.error){
				msg=window.shipmentDestination.name+" said: "+transport.responseJSON.error;
			}
			let mb=(document.getElementById("modalBox").querySelector(".boxbody"));
			ui.setModalBoxTitle("Could not retrieve list of proposals from "+window.shipmentDestination.name);
			mb.innerHTML=msg;
		},
		renderProposalList:function(response){
			let mb=document.getElementById("modalBox").querySelector(".boxbody");
			if(parseInt(response.total)===0){
				ui.setModalBoxTitle("No proposals at "+window.shipmentDestination.name);
				mb.innerHTML='No proposals found. Cannot submit shipment.';
				return false;
			}
			ui.setModalBoxTitle("Your proposals at "+window.shipmentDestination.name);
			mb.innerHTML='';
			response.data.each(function(p){
				let prop=mb.treeItem({
					record:p,
					header:p["PROPOSAL"]+": "+p["TITLE"],
					updater:function(tb){
						AjaxUtils.remoteAjax(
							SynchwebShippingHandler.baseUri+'proposal/visits?page=1&per_page=50&prop='+p["PROPOSAL"],
							'get',
							'',
							function (transport) {
								//DLS offers unattended data collection with no explicit visit
								if(-1===window.shipmentDestination["name"].indexOf("diamond")){
									transport.responseJSON.data.push({
										"VISIT":SynchwebShippingHandler.UNATTENDED,
										"SESSIONID":"",
										"BL":"", "ST":"When available", "EN":"",
										"ACTIVE":"1"
									})
								}
								tb.table({
										headers:["Visit","Beamline","Active","Start","End",""],
										cellTemplates:["{{VISIT}}", "{{BL}}", [ui.checkmark,"ACTIVE", ], "{{ST}}", "{{EN}}", [SynchwebShippingHandler.getSendButton,"ACTIVE"] ],
									},
									transport.responseJSON
								);
							},
							function (transport) {
								if(404===transport.status){
									//No visits for proposal. Offer unattended if DLS
									if(-1===window.shipmentDestination["name"].indexOf("diamond")) {
										tb.table({
												headers: ["Visit", "Beamline", "Active", "Start", "End", ""],
												cellTemplates: ["{{VISIT}}", "{{BL}}", [ui.checkmark, "ACTIVE",], "{{ST}}", "{{EN}}", [SynchwebShippingHandler.getSendButton, "ACTIVE"]],
											},
											[{
												"VISIT": SynchwebShippingHandler.UNATTENDED,
												"SESSIONID": "",
												"BL": "", "ST": "When available", "EN": "",
												"ACTIVE": "1"
											}]
										);
									} else {
										tb.innerHTML="There are no visits associated with this proposal."
									}
								} else if(transport.responseJSON && transport.responseJSON.error){
									tb.innerHTML=transport.responseJSON.error;
								} else {
									tb.innerHTML="There was a problem getting the visits for this proposal."
								}
							},
							{ 'Authorization':'Bearer '+SynchwebShippingHandler.token }
						);
					},
					url:SynchwebShippingHandler.baseUri+'proposal/visits?page=1&per_page=50&prop='+p["PROPOSAL"],
					headers:["Visit","Beamline","Active","Start","End",""],
					cellTemplates:["{{VISIT}}", "{{BL}}", [ui.checkmark,"ACTIVE", ], "{{ST}}", "{{EN}}", [SynchwebShippingHandler.getSendButton,"ACTIVE"] ],
					requestHeaders:{ 'Authorization':'Bearer '+SynchwebShippingHandler.token }
				});
				SynchwebShippingHandler.getProposalLabContacts(prop);
			});
		},
		
		getProposalLabContacts:function(elem){
			let p=elem.record;
			if(!p){ return; }
			AjaxUtils.remoteAjax(
					SynchwebShippingHandler.baseUri+"contact?page=1&per_page=100&all=1&prop="+p["PROPOSAL"],
					'get',
					'', 
					function(transport){ SynchwebShippingHandler.getProposalLabContacts_onSuccess(transport,elem); },
					function(transport){ SynchwebShippingHandler.getProposalLabContacts_onFailure(transport,elem); },
					{ 'Authorization':'Bearer '+SynchwebShippingHandler.token }
			);
		},
		getProposalLabContacts_onSuccess:function(transport, elem){
			if(!transport.responseJSON || !transport.responseJSON.data){
				return SynchwebShippingHandler.getProposalLabContacts_onFailure(transport,elem);
			}
			elem.labContacts=transport.responseJSON.data;
		},
		getProposalLabContacts_onFailure:function(transport, elem){
			elem.labContacts=[];
		},

		/**
		 * Returns the HTML for a "send shipment" button.
		 * @param {Object} record A visit object from Synchweb.
		 */
		getSendButton:function(record){
			if(SynchwebShippingHandler.UNATTENDED===record["VISIT"]){
				return '<input type="button" value="Unattended collection" onclick="SynchwebShippingHandler.chooseLabContact(this)" />';
			} else if(1===parseInt(record["ACTIVE"])){
				return '<input type="button" value="Use this visit" onclick="SynchwebShippingHandler.chooseLabContact(this)" />';
			}
			//Inactive visits - currently allow and let synchrotron barf if unhappy.
			return '<input type="button" value="Use this visit" onclick="SynchwebShippingHandler.chooseLabContact(this)" />';
		},
		
		/**
		 * Triggers final submission of the shipment to the synchrotron (along with any preflight validation).
		 * @param {Element} btn The "Send shipment" button.
		 */
		chooseLabContact:function(btn){
			let visit=btn.closest("tr").rowData;
			let proposalElement=btn.closest("div.treeitem");
			let proposal=proposalElement.record;
			let labContacts=proposalElement.labContacts;
			if(undefined===labContacts){
				alert("Still waiting for list of lab contacts for this proposal. Try again in a moment.");
				return false;
			} else if(0===labContacts.length){
				alert("This proposal has no lab contacts. You need to define at least one lab contact in "+window.shipmentDestination.name+"'s ISPyB.");
				return false;
			} else {
				SynchwebShippingHandler.proposal=proposal;
				SynchwebShippingHandler.visit=visit;
				let box=document.getElementById("modalBox").querySelector(".boxbody");
				box.innerHTML='';
				box.table({
					headers:[ 'Name','Organisation',''],
					cellTemplates:[ '{{CARDNAME}}','{{LABNAME}}','<input type="button" value="Choose" onclick="SynchwebShippingHandler.setLabContact(this)" />' ],
					contentBefore:'Choose the lab contact for the shipment:'
				}, {'total':labContacts.length, 'rows':labContacts });
				ui.setModalBoxTitle("Choose the lab contact for this shipment");
			}
		},
		setLabContact:function(btn){
			SynchwebShippingHandler.labContact=btn.closest("tr").rowData;
			SynchwebShippingHandler.chooseSafetyLevel();
		},

		chooseSafetyLevel:function(){
			ui.setModalBoxTitle("Choose the safety level for this shipment");
			let box=document.getElementById("modalBox").querySelector(".boxbody");
			box.innerHTML='';
			let t=box.table({
				headers:[ 'Colour code','Risk level',''],
				cellTemplates:[ '{{code}}','{{risk}}','<input type="button" value="Choose" onclick="SynchwebShippingHandler.setSafetyLevel(this)" />' ],
				contentBefore:'Choose the safety level for the shipment:'
			}, {'total':3, 'rows':SynchwebShippingHandler.safetyLevels });
			t.select("tr").each(function(tr){
				if(!tr.rowData){ return; }
				tr.style.backgroundColor=tr.rowData.background;
				tr.querySelector("td").style.color=tr.rowData.color;
				tr.querySelector("td").style.fontWeight="bold";
			});
		},
		setSafetyLevel:function(btn){
			let tr=btn.closest("tr");
			SynchwebShippingHandler.safetyLevel=tr.rowData["code"];
			SynchwebShippingHandler.showFinalConfirmation();
		},
		
		showFinalConfirmation:function(){
			let box=document.getElementById("modalBox").querySelector(".boxbody");
			let proposal=SynchwebShippingHandler.proposal;
			let visit=SynchwebShippingHandler.visit;
			let labContact=SynchwebShippingHandler.labContact;
			box.innerHTML='';
			ui.setModalBoxTitle("Check the details before sending the shipment");
			let f=box.form({ action:'#', method:'post' });
			f.setStyle({ position:"absolute", top:"1%", left:"1%", width:"48%" });
			f.textField({ readonly:true, label:'Destination', value:window.shipmentDestination.name });
			f.textField({ readonly:true, label:'Proposal', value:proposal["PROPOSAL"] });
			f.textField({ readonly:true, label:'Visit', value:visit["VISIT"] });
			f.textField({ readonly:true, label:'Home lab contact', value:labContact["CARDNAME"] });
			
			SynchwebShippingHandler.safetyLevels.each(function(sl){
				if(sl.code.toLowerCase()!==SynchwebShippingHandler.safetyLevel.toLowerCase()){ return; }
				let ff=f.textField({ readonly:true, label:'Safety level', value:sl.code });
				ff.style.backgroundColor=sl.background;
			});

			let f2=box.form({ action:'#', method:'post' });
			f2.setStyle({ position:"absolute", top:"1%", right:"1%", width:"48%", marginTop:0, paddingTop:0 });
			let b=f2.formField({ readonly:true, label:'Review this information to make sure that it is correct, then click "Send shipment".', content:'<input type="button" value="Send shipment" onclick="SynchwebShippingHandler.sendShipment()" />' });
			b.setStyle({ "height":f.getHeight()+"px"});
			b.querySelector("input").setStyle({ position:"absolute", bottom:"0.5em", right:"0.5em"});

			box.innerHTML+='<div style="position:absolute;bottom:10%; left:1%; width:98%; text-align:center"><a href="#" onclick="ui.closeModalBox();return false">Cancel - don\'t send this shipment</a></div>';
		},
		
		sendShipment:function(){
			let visit=SynchwebShippingHandler.visit;
			if(!confirm("Really submit shipment to "+window.shipmentDestination.name+", visit "+visit["VISIT"]+"?")){ return false; }
			ui.setModalBoxTitle("Submitting your shipment to "+window.shipmentDestination.name+"...");
			document.getElementById("modalBox").querySelector(".boxbody").innerHTML="";
			ui.logToDialog('Beginning to send shipment...');
			SynchwebShippingHandler.getProteins();
		},
		
		getProteins:function(){
			let proposal=SynchwebShippingHandler.proposal;
			AjaxUtils.remoteAjax(
					SynchwebShippingHandler.baseUri+"sample/proteins?page=1&per_page=1000&all=1&prop="+proposal["PROPOSAL"],
					'get',
					'', 
					SynchwebShippingHandler.doPreflight,
					SynchwebShippingHandler.getProteins_onFailure,
					{ 'Authorization':'Bearer '+SynchwebShippingHandler.token }
			);			
		},
		getProteins_onFailure:function(transport){
			ui.setModalBoxTitle("Could not send shipment");
			document.getElementById("modalBox").querySelector(".boxbody").innerHTML='';
			if(!transport.responseJSON){
				ui.logToDialog('Could not get list of proteins from '+window.shipmentDestination.name+
				', so could not validate shipment before sending.<br/><br/>Please try again later.','error');
			} else if(transport.responseJSON.message){
				ui.logToDialog('Could not get list of proteins from '+window.shipmentDestination.name+
				', so could not validate shipment before sending.<br/>The remote server said:<br/><br/>'+
				transport.responseJSON.message+'<br/><br/>Please try again later.','error');
			} else if(0===parseInt(transport.responseJSON.total)){
				ui.logToDialog('Proposal  '+SynchwebShippingHandler.proposal["PROPOSAL"]+
				' has no proteins. Therefore none of your samples have a protein acronym that appears in the proposal, '+
				'and the shipment cannot be sent.<br/><br/>Create the proteins in the '+window.shipmentDestination.name+' ISPyB system.','error');
			}
		},
		
		doPreflight:function(transport){
			if(!transport.responseJSON || 0===parseInt(transport.responseJSON.total) || 400<=transport.responseJSON.status){
				return SynchwebShippingHandler.getProteins_onFailure(transport);
			}
			let box=document.getElementById("modalBox").querySelector(".boxbody");
			SynchwebShippingHandler.proteins=transport.responseJSON.data;
			let acronyms=[];
			transport.responseJSON.data.each(function(protein){
				acronyms.push(protein["ACRONYM"]);
			});
			let errors=Shipment.getShipmentErrors(acronyms);
			
			if(0<errors.length){
				box.innerHTML="Could not send shipment. Please fix the following:<br/><ul><li>"+errors.join("</li><li>")+"</ul>";
				return false;
			}
			let dewars=document.querySelectorAll(".containertab");

			dewars.forEach(function(tab){
				let dewarName=tab.dataset.containername;
				let nameFormat=new RegExp('^DLS-MX-\d\d\d\d$');
				if(!nameFormat.test(dewarName)){
					errors.push('Dewar name does not match Diamond barcode format DLS-MX-nnnn');
				}
				tab.classList.add("noRemoteDewar");
			});

			ui.logToDialog("Pre-flight validation complete. Creating shipment in ISPyB...");
			SynchwebShippingHandler.remoteErrors=[];
			SynchwebShippingHandler.createShipmentWithEmptyDewars();
		},

		remoteErrors:[],
		logRemoteError:function(err){
			SynchwebShippingHandler.remoteErrors.push(err);
		},
		showRemoteErrors:function(){
			ui.logToDialog('The following errors occurred:<br/><br/>','error');
			ui.logToDialog(SynchwebShippingHandler.remoteErrors.join('<br/>'));
		},

		/**
		 * Creates a shipment at the synchrotron, with the empty dewars attached.
		 * Dewars that are not registered at the synchrotron are automatically created.
		 * We only receive a shipment ID in return, so the next stage retrieves all dewars for the shipment.
		 */
		createShipmentWithEmptyDewars:function(){
			ui.logToDialog("Submitting basic shipment details");
			let safetyLevel=SynchwebShippingHandler.safetyLevel;
			let proposal=SynchwebShippingHandler.proposal["PROPOSAL"];
			let visitId=SynchwebShippingHandler.visit["SESSIONID"];
			let labContactId=SynchwebShippingHandler.labContact["LABCONTACTID"];
			let shipmentName=document.getElementById("shipmentdetails_form").querySelector("#name").value;
			
			let dewars=document.querySelectorAll(".containertab");
			let dewarBarcodes=[];
			dewars.forEach(function(tab){
				dewarBarcodes.push(tab.dataset.containername);
			});
			let numBarcodes=dewarBarcodes.length;
			dewarBarcodes='["'+dewarBarcodes.join('","')+'"]';

			let parameters='{"SHIPPINGNAME":"'+shipmentName+'","DEWARS":"'+numBarcodes+'","FCODES":'+dewarBarcodes;
			if(SynchwebShippingHandler.UNATTENDED===SynchwebShippingHandler.visit["VISIT"]){
				parameters+=',"FIRSTEXPERIMENTID":"","noexp":"on", ';
			} else {
				parameters+=',"FIRSTEXPERIMENTID":"'+visitId+'","noexp":null, ';
			}
			parameters+='"SAFETYLEVEL":"'+safetyLevel+'","COMMENT":"","SENDINGLABCONTACTID":"'+labContactId+'","RETURNLABCONTACTID":"'+labContactId+'", '+
				'"DELIVERYAGENT_SHIPPINGDATE":"","PHYSICALLOCATION":"","READYBYTIME":"","CLOSETIME":"","DELIVERYAGENT_DELIVERYDATE":"", '+
				'"DELIVERYAGENT_AGENTNAME":"","DELIVERYAGENT_AGENTCODE":"","prop":"'+proposal+'" '+
			'}';

			AjaxUtils.remoteAjax(
					SynchwebShippingHandler.baseUri+"shipment/shipments",
					'post',
					parameters,
					SynchwebShippingHandler.createShipmentWithEmptyDewars_onSuccess,
					SynchwebShippingHandler.createShipmentWithEmptyDewars_onFailure,
					{ 'Authorization':'Bearer '+SynchwebShippingHandler.token }
			);
		},
		createShipmentWithEmptyDewars_onSuccess:function(transport){
			if(transport.responseJSON && transport.responseJSON["SHIPPINGID"]){
				SynchwebShippingHandler.shipmentIdAtFacility=transport.responseJSON["SHIPPINGID"];
				window.setTimeout(SynchwebShippingHandler.getDewarObjectsForShipment, 50);
			} else {
				SynchwebShippingHandler.logRemoteError("No shipment ID returned from shipment create attempt.");
				return SynchwebShippingHandler.createShipmentWithEmptyDewars_onFailure(transport);
			}
		},
		createShipmentWithEmptyDewars_onFailure:function(){
			SynchwebShippingHandler.logRemoteError("Could not create shipment.");
			SynchwebShippingHandler.showRemoteErrors();
		},

		
		/**
		 * Retrieves dewar objects for this shipment. On success, we attempt to populate them with pucks.
		 * We created the shipment above, along with empty dewars (which may or mat not have been registered at the synchrotron beforehand). However,
		 * we only got a shipment ID in return and have no idea what the dewar IDs are. We need those for the next step. So here , we retrieve all
		 * dewar objects for this shipment, and we attach them to the relevant tab in our shipping UI.
		 */
		getDewarObjectsForShipment:function(){
			ui.logToDialog("Getting list of dewars in remote shipment...");
			let shipmentId=SynchwebShippingHandler.shipmentIdAtFacility;
			let proposalName=SynchwebShippingHandler.proposal["PROPOSAL"];
			AjaxUtils.remoteAjax(
					SynchwebShippingHandler.baseUri+"shipment/dewars/sid/"+shipmentId+"?prop="+proposalName,
					'get',
					'',
					SynchwebShippingHandler.getDewarObjectsForShipment_onSuccess,
					SynchwebShippingHandler.getDewarObjectsForShipment_onFailure,
					{ 'Authorization':'Bearer '+SynchwebShippingHandler.token }
			);
		},

		getDewarObjectsForShipment_onSuccess:function(transport){
			if(!transport.responseJSON || !transport.responseJSON.total || !transport.responseJSON.data){
				return SynchwebShippingHandler.getDewarObjectsForShipment_onFailure(transport);
			}
			ui.logToDialog("...succeeded.");
			transport.responseJSON.data.each(function(remoteDewar){
				let dewarTab=document.getElementById(remoteDewar["FACILITYCODE"]);
				if(!dewarTab){
					SynchwebShippingHandler.logRemoteError("Barcode mismatch - could not find local dewar tab for remote dewar "+remoteDewar["FACILITYCODE"]+".");
					return;
				}
				dewarTab.remoteDewar=remoteDewar;
				dewarTab.dataset.remoteid=remoteDewar["DEWARID"];
				dewarTab.classList.remove("noRemoteDewar");
			});
			let unFoundDewars=document.querySelectorAll(".noRemoteDewar");
			if(0!==unFoundDewars.length){
				SynchwebShippingHandler.logRemoteError("Not all dewars in local shipment are in the remote shipment.");
			}
			if(0!==SynchwebShippingHandler.remoteErrors.length){
				SynchwebShippingHandler.showRemoteErrors();
			} else {
				SynchwebShippingHandler.addPucksToDewars();
			}
		},

		getDewarObjectsForShipment_onFailure:function(){
			SynchwebShippingHandler.logRemoteError("Could not get list of dewars in shipment, or list was empty.");
			SynchwebShippingHandler.showRemoteErrors();
		},

		/**
		 * Populate all dewars with their child pucks, at the synchrotron.
		 */
		addPucksToDewars:function(){
			ui.logToDialog("Adding pucks to dewars...");
			document.querySelectorAll(".containertab").forEach(function(localDewarTab){
				SynchwebShippingHandler.addPucksToDewar(localDewarTab);
			});
		},
		
		/**
		 * Populate a dewar with its child pucks, at the synchrotron.
		 */
		addPucksToDewar:function(localDewarTab){
			ui.logToDialog("Adding pucks to dewar "+localDewarTab.id+"...");
			let tabBody=localDewarTab.nextElementSibling;
			let pucks=tabBody.querySelectorAll(".treeitem");
			pucks.forEach(function(p){
				let localPuck=p.record;
				p.id=p.record.name;
				SynchwebShippingHandler.addPuckToDewar(localPuck, localDewarTab);
			});
		},
		
		/**
		 * Add a puck to a dewar within the shipment, at the synchrotron.
		 */
		addPuckToDewar:function(localPuck, localDewarTab){
			ui.logToDialog("Adding puck "+localPuck.name+" to "+localDewarTab.id+"...");
			let remoteDewarId=localDewarTab.dataset.remoteid;
			let proposalName=SynchwebShippingHandler.proposal["PROPOSAL"];
			AjaxUtils.remoteAjax(
					SynchwebShippingHandler.baseUri+"shipment/containers",
					'post',
					'{ "NAME":"'+localPuck.name+'", "CONTAINERTYPE":"Puck", "DEWARID":"'+remoteDewarId+'","CAPACITY":"'+localPuck.positions+'","prop":"'+proposalName+'"  }',
					function(transport){ SynchwebShippingHandler.addPuckToDewar_onSuccess(transport, localPuck) },
					function(transport){ SynchwebShippingHandler.addPuckToDewar_onFailure(transport, localPuck) },
					{ 'Authorization':'Bearer '+SynchwebShippingHandler.token }
			);
		},
		addPuckToDewar_onSuccess:function(transport, localPuck){
			if(transport.responseJSON["msg"] && -1!==transport.responseJSON["msg"].indexOf("Duplicate entry")){
				ui.logToDialog("Puck "+localPuck.name+" already exists. Trying another way...");
				SynchwebShippingHandler.findExistingPuck(localPuck);
				return;
			}
			localPuck.remoteId=transport.responseJSON["CONTAINERID"];
			ui.logToDialog("...added "+localPuck.name+" to dewar.");
			SynchwebShippingHandler.addSamplesToPuck(localPuck);			
		},
		addPuckToDewar_onFailure:function(){
			SynchwebShippingHandler.logRemoteError("Could not add puck to dewar.");
			SynchwebShippingHandler.showRemoteErrors();			
		},
		
		/**
		 * If addPuckToDewar reported "duplicate entry" so puck exists. Get it, and continue.
		 */
		findExistingPuck:function(localPuck){
			ui.logToDialog("Finding existing puck "+localPuck.name+"...");
			let proposalName=SynchwebShippingHandler.proposal["PROPOSAL"];
			AjaxUtils.remoteAjax(
					SynchwebShippingHandler.baseUri+"shipment/containers?per_page=100000&page=1&prop="+proposalName,
					'get',
					'{}',
					function(transport){ SynchwebShippingHandler.findExistingPuck_onSuccess(transport, localPuck) },
					function(transport){ SynchwebShippingHandler.findExistingPuck_onFailure(transport, localPuck) },
					{ 'Authorization':'Bearer '+SynchwebShippingHandler.token }
			);
		},
		findExistingPuck_onSuccess:function(transport, localPuck){
			let puckBarcode=localPuck.name;
			let remotePuckId=null;
			transport.responseJSON.data.each(function(container){
				if(container["NAME"]===puckBarcode){
					remotePuckId=container["CONTAINERID"];
				}
			});
			if(!remotePuckId){
				SynchwebShippingHandler.logRemoteError("Could not retrieve list of existing containers.");
				SynchwebShippingHandler.showRemoteErrors();				
			}
			localPuck.remoteId=remotePuckId;
			let localDewarTab=document.getElementById(localPuck.name).closest(".tabbody").previous();
			SynchwebShippingHandler.addExistingPuckToDewar(localPuck, localDewarTab);
		},
		findExistingPuck_onFailure:function(){
			SynchwebShippingHandler.logRemoteError("Could not retrieve list of existing containers.");
			SynchwebShippingHandler.showRemoteErrors();				
		},

		
		/**
		 * Add an existing puck to a dewar within the shipment, at the synchrotron.
		 */
		addExistingPuckToDewar:function(localPuck, localDewarTab){
			ui.logToDialog("Adding existing puck "+localPuck.name+" to "+localDewarTab.id+"...");
			let remoteDewarId=localDewarTab.dataset.remoteid;
			let proposalName=SynchwebShippingHandler.proposal["PROPOSAL"];
			AjaxUtils.remoteAjax(
					SynchwebShippingHandler.baseUri+"shipment/containers/move?cid="+localPuck.remoteId+"&did="+remoteDewarId+"&prop="+proposalName,
					'get', //yes, really
					'{   }',
					function(transport){ SynchwebShippingHandler.addExistingPuckToDewar_onSuccess(transport, localPuck) },
					function(transport){ SynchwebShippingHandler.addExistingPuckToDewar_onFailure(transport, localPuck) },
					{ 'Authorization':'Bearer '+SynchwebShippingHandler.token }
			);
		},
		addExistingPuckToDewar_onSuccess:function(transport, localPuck){
			ui.logToDialog("...added "+localPuck.name+" to dewar.");
			SynchwebShippingHandler.addSamplesToPuck(localPuck);			
		},
		addExistingPuckToDewar_onFailure:function(){
			SynchwebShippingHandler.logRemoteError("Could not add puck to dewar.");
			SynchwebShippingHandler.showRemoteErrors();			
		},

		
		addSamplesToPuck:function(localPuck){
			let puckElement=document.getElementById(localPuck.name);
			let positions=puckElement.select("tr.datarow");
			positions.each(function(tr){
				if(!tr.rowData || 1===parseInt(tr.rowData.isEmpty)){ return; }
				tr.classList.add("unsavedPin");
			});
			positions.each(function(tr){
				if(!tr.rowData || (tr.rowData.isEmpty && 1===parseInt(tr.rowData.isEmpty))){ return; }
				SynchwebShippingHandler.addSampleToPuck(localPuck, tr);
			});
		},
		addSampleToPuck:function(localPuck, pinTr){
			let proposalName=SynchwebShippingHandler.proposal["PROPOSAL"];
			let pin=pinTr.rowData;
			let position=pin.position;
			let pinBarcode=pin.name;
			if(0===pinBarcode.indexOf("dummypin")){
				pinBarcode="";
			}
			let crystal=pin.childitems[1]; //0 is dummy
			let proteinId=null;
			SynchwebShippingHandler.proteins.each(function(protein){
				if(protein["ACRONYM"]===crystal.proteinacronym){
					proteinId=protein["PROTEINID"];
				}
			});
			if(null==proteinId){
				SynchwebShippingHandler.logRemoteError("Could not find protein ID for crystal.");
				SynchwebShippingHandler.showRemoteErrors();
				return false;
			}
			let sampleName=crystal.name.replace(" ","_"); //Don't send sample names with spaces
			let experimentType=crystal["diffractiontype"];
			if(!experimentType){ experimentType=""; }
			let spaceGroup=crystal.spacegroup;
			if(!spaceGroup){ spaceGroup=""; }
			let shippingComment=crystal.shippingcomment.split('"').join('');
			ui.logToDialog("Adding pin with barcode ["+pinBarcode+"] to puck "+localPuck.name+", position "+position+"...");
						
			AjaxUtils.remoteAjax(
					SynchwebShippingHandler.baseUri+"sample",
					'post',
					' [{"LOCATION":"'+position+'","PROTEINID":"'+proteinId+'","CRYSTALID":-1,"new":true,"NAME":"'+sampleName+'","CODE":"'+pinBarcode+'", '+
					' "COMMENTS":"'+shippingComment+'","SPACEGROUP":"'+spaceGroup+'","REQUIREDRESOLUTION":"","ANOMALOUSSCATTERER":"","CELL_A":"'+crystal.unitcella+'","CELL_B":"'+crystal.unitcellb+'", '+
					' "CELL_C":"'+crystal.unitcellc+'","CELL_ALPHA":"'+crystal.unitcellalpha+'","CELL_BETA":"'+crystal.unitcellbeta+'","CELL_GAMMA":"'+crystal.unitcellgamma+'","VOLUME":"","ABUNDANCE":"","SYMBOL":null, '+
					' "PACKINGFRACTION":"","EXPERIMENTALDENSITY":"","COMPOSITION":"","LOOPTYPE":"","DIMENSION1":"","DIMENSION2":"", '+
					' "DIMENSION3":"","SHAPE":"","components":[],"EXPERIMENTKINDNAME":"","isSelected":false,"STATUS":"", '+
					' "CENTRINGMETHOD":"","EXPERIMENTKIND":"'+experimentType+'","CONTAINERID":'+localPuck.remoteId+',"COMPONENTID":"","ANOM_NO":"","prop":"'+proposalName+'" '+
					' }]',
					function(transport){ SynchwebShippingHandler.addSampleToPuck_onSuccess(transport, pinTr) },
					function(transport){ SynchwebShippingHandler.addSampleToPuck_onFailure(transport, pinTr) },
					{ 'Authorization':'Bearer '+SynchwebShippingHandler.token }
			);
		},
		addSampleToPuck_onSuccess:function(transport,pinTr){
			/*
			 [{"prop":"mx4025","PROTEINID":"42242","CONTAINERID":138441,"LOCATION":"1","CODE":"HA00AV9785","NAME"
				:"LMTIM_9098A01d1c1","COMMENTS":"","SPACEGROUP":"P1","BLSAMPLEID":2185329}]
			 */
			let remoteCrystalId=transport.responseJSON[0]["BLSAMPLEID"];
			let diffractionRequestId=pinTr.rowData.childitems[1]["diffractionrequestid"];
			SynchwebShippingHandler.saveRemoteCrystalId(diffractionRequestId, remoteCrystalId, pinTr);
			SynchwebShippingHandler.addShippedNote(pinTr.rowData.childitems[1]);
		},
		addSampleToPuck_onFailure:function(transport,pinTr){
			ui.logToDialog("<strong>Adding sample "+pinTr.name+"to puck failed.</strong>","error");			
		},

		addShippedNote:function(localCrystal){
			//TODO Move this to server-side, shipment::update - eliminate duplication in future handlers
			let noteText='Crystal shipped to '+window.shipmentDestination.name;
			new Ajax.Request('/api/note',{
				method:'post',
				parameters:{
					csrfToken:csrfToken,
					parentid:localCrystal.id,
					text:noteText
				},
				onSuccess:function(transport){ if(successCallback){ successCallback(transport.responseJSON); } },
				onFailure:AjaxUtils.checkResponse
			});
		},
		
		saveRemoteCrystalId:function(diffractionRequestId, remoteCrystalId, pinTr){
			new Ajax.Request('/api/diffractionrequest/'+diffractionRequestId,{
				'method':'patch',
				'parameters':{
					'csrfToken':csrfToken,
					'shipmentid':data.id,
					'crystalidatremotefacility':remoteCrystalId,
					'crystalurlatremotefacility':SynchwebShippingHandler.getCrystalUrlAtFacility(window.shipmentDestination, remoteCrystalId)
				},
				onSuccess:function(){
					pinTr.classList.remove("unsavedPin");
					let unsavedPins=document.querySelectorAll(".unsavedPin");
					if(!unsavedPins || 0===unsavedPins.length){
						window.setTimeout(SynchwebShippingHandler.completeShipment, 100);
					}
				},
				onFailure:function(){
					pinTr.classList.remove("unsavedPin");
					let unsavedPins=document.querySelectorAll(".unsavedPin");
					if(!unsavedPins || 0===unsavedPins.length){
						window.setTimeout(SynchwebShippingHandler.completeShipment, 100);
					}
				},
			});
			
		},
		
		completeShipment:function(){
			ui.logToDialog("Shipment was submitted to "+window.shipmentDestination.name+".","success");			
			ui.logToDialog("Updating IceBear and creating shipment manifest...");
			let shippedDate=new Date().toISOString().split("T")[0];
			new Ajax.Request('/api/shipment/'+data.id,{
				'method':'patch',
				'parameters':{
					'csrfToken':csrfToken,
					'dateshipped':shippedDate,
					'idatremotefacility':SynchwebShippingHandler.shipmentIdAtFacility,
					'urlatremotefacility':SynchwebShippingHandler.getShipmentUrlAtFacility(window.shipmentDestination, SynchwebShippingHandler.shipmentIdAtFacility),
					'proposalname':SynchwebShippingHandler.proposal["PROPOSAL"],
					'sessionname':SynchwebShippingHandler.visit["VISIT"],
				},
				onSuccess:SynchwebShippingHandler.completeShipment_onSuccess,
				onFailure:SynchwebShippingHandler.completeShipment_onFailure,
			});
		},
		completeShipment_onSuccess:function(){
			ui.logToDialog("IceBear updated successfully.","success");
			ui.logToDialog("The page will reload in a moment. Please wait...");
			window.setTimeout(ui.forceReload, 2500);
		},
		completeShipment_onFailure:function(transport){
			ui.logToDialog("The shipment submission to "+window.shipmentDestination.name+" succeeded, but IceBear was not updated ","error");
			if(transport.responseJSON && transport.responseJSON.error){				
				ui.logToDialog(transport.responseJSON.error, "error");
			}			
			ui.logToDialog("The page will reload in a moment. Please wait...");
			window.setTimeout(ui.forceReload, 5000);
		},

	DatasetRetrieval: {

		begin: function(){
			ui.modalBox({'title':'Dataset retrieval', content:'...'});
			let baseUri=window.shipmentDestination["baseuri"];
			if(!baseUri){
				alert("No shipping API base URI set for "+window.shipmentDestination.name+".");
				return false;
			}
			SynchwebShippingHandler.baseUri=baseUri;
			SynchwebShippingHandler.DatasetRetrieval.getDatasetsForShipment();
		},

		getDatasetsForShipment:function(){
			Shipment.DatasetRetrieval.datasets=[];
			let mb=document.getElementById("modalBox").querySelector(".boxbody");
			mb.innerHTML="";
			if(!SynchwebShippingHandler.token){
				return SynchwebShippingHandler.showLoginForm(SynchwebShippingHandler.DatasetRetrieval.getDatasetsForShipment);
			}
			document.getElementById("modalBox").querySelector("h2").innerHTML="Dataset retrieval";
			ui.logToDialog("Getting datasets for shipment...");
			AjaxUtils.remoteAjax(
				SynchwebShippingHandler.baseUri+"dc?page=1&per_page=15000&total_pages=0&shipment="+data["idatremotefacility"]+"&t=fc&prop="+data["proposalname"],
				'get',
				'',
				function(transport){
					let datasets=transport.responseJSON[1];
					datasets.forEach(function(ds){
						if("data"===ds["TYPE"]){
							Shipment.DatasetRetrieval.datasets.push(ds);
						}
					});
					Shipment.DatasetRetrieval.renderProgressNumbers();
					window.setTimeout(function(){
						SynchwebShippingHandler.DatasetRetrieval.gatherBeamlineDetails();
						SynchwebShippingHandler.DatasetRetrieval.processFoundDatasets();
					}, 50);
				},
				function(transport){ alert(transport.responseText); },
				{ 'Authorization':'Bearer '+SynchwebShippingHandler.token }
			);
		},

		processFoundDatasets:function (){
			Shipment.DatasetRetrieval.datasets.forEach(function (ds){
				if("datacollection successful"!==ds["RUNSTATUS"].toLowerCase()){
					Shipment.DatasetRetrieval.incrementFailedCount('Dataset run status is not "DataCollection Successful"');
				} else if(!ds["BLSAMPLEID"]){
					Shipment.DatasetRetrieval.incrementFailedCount("Dataset does not contain a remote crystal ID (BLSAMPLEID)");
				} else {
//					new Ajax.Request("/api/diffractionrequest/shipmentid/"+data["id"]+"/crystalidatremotefacility/3335807",{
					new Ajax.Request("/api/diffractionrequest/shipmentid/"+data["id"]+"/crystalidatremotefacility/"+ds["BLSAMPLEID"],{
						method:"get",
						onSuccess:function(transport){
							if(1===transport.responseJSON["rows"].length){
								SynchwebShippingHandler.DatasetRetrieval.processFoundDataset(ds, transport.responseJSON["rows"][0]);
							} else {
								Shipment.DatasetRetrieval.incrementFailedCount("More than one IceBear crystal in this shipment with remote ID "+ds["BLSAMPLEID"]);
							}
						}, onFailure:function (transport){
							if(404===transport.status){
								Shipment.DatasetRetrieval.incrementFailedCount("No IceBear crystal in this shipment with remote ID "+ds["BLSAMPLEID"]);
							} else {
								Shipment.DatasetRetrieval.incrementFailedCount("Error getting IceBear crystal(s) with remote ID "+ds["BLSAMPLEID"]+" (HTTP "+transport.status+")");
							}
						}
					});
				}
			});
		},

		processFoundDataset:function (ds, diffractionRequest){
			let remoteDatasetId=ds["ID"];
			if(!remoteDatasetId){
				Shipment.DatasetRetrieval.incrementFailedCount("Dataset has no ID");
				return;
			}
			new Ajax.Request('/api/dataset/diffractionrequestid/'+diffractionRequest["id"]+'/remotedatasetid/'+remoteDatasetId, {
				method: 'get',
				onSuccess:function (transport){
					if(transport.responseJSON.rows && 1===transport.responseJSON.rows.length){
						let localDatasetId=transport.responseJSON.rows[0]["id"];
						SynchwebShippingHandler.DatasetRetrieval.createOrUpdateLocalDataset(ds, diffractionRequest, localDatasetId);
					} else {
						Shipment.DatasetRetrieval.incrementFailedCount("Got more than one local dataset, should not happen");
					}
				},
				onFailure:function (transport){
					if(404===transport.status){
						SynchwebShippingHandler.DatasetRetrieval.createOrUpdateLocalDataset(ds, diffractionRequest, null);
					} else {
						Shipment.DatasetRetrieval.incrementFailedCount("Error on checking for existing dataset, HTTP "+transport.status);
					}
				},
			});
		},

		createOrUpdateLocalDataset:function (ds, diffractionRequest, localDatasetId){
			let beamlineName=ds["BL"];
			if(!beamlineName){
				Shipment.DatasetRetrieval.incrementFailedCount("No beamline name for this dataset - undefined");
				return false;
			}
			if(undefined===Shipment.DatasetRetrieval.beamlineNameToId[beamlineName]){
				//Need the beamline ID
				ui.logToDialog("Waiting for beamline ID");
				window.setTimeout(function (){
					SynchwebShippingHandler.DatasetRetrieval.createOrUpdateLocalDataset(ds, diffractionRequest,localDatasetId);
				},500);
				return false;
			}
			let beamlineId=Shipment.DatasetRetrieval.beamlineNameToId[beamlineName];
			let method="post";
			let uri="/api/dataset/";
			if(localDatasetId){
				method="patch";
				uri+=localDatasetId;
			}
			let detectorManufacturer=ds["DETECTORMANUFACTURER"];
			let detectorModel=ds["DETECTORMODEL"];
			let detectorMode=ds["DETECTORMODE"];
			let detectorType=ds["DETECTORTYPE"];
			//As of March 2021, Diamond returns null, "" or "1", which are useless, and doesn't provide detector type at all.
			if(!detectorManufacturer || 2>detectorManufacturer.length){ detectorManufacturer=""; }
			if(!detectorModel || 2>detectorModel.length){ detectorModel=""; }
			if(!detectorMode || 2>detectorMode.length){ detectorMode=""; }
			if(!detectorType || 2>detectorType.length){ detectorType=""; }
			new Ajax.Request(uri, {
				method: method,
				parameters: {
					"csrfToken": csrfToken,
					"diffractionrequestid": diffractionRequest["id"],
					"crystalid": diffractionRequest["crystalid"],
					"beamlineid": beamlineId,
					"remotedatasetid": ds["ID"],
					"remotedatasetobject": JSON.stringify(ds),
					"datalocation": window.shipmentDestination.name + ": " + ds["DIRFULL"],
					"description": "Run "+ds["RUN"]+". "+ds["COMMENTS"],
					"detectormanufacturer": detectorManufacturer,
					"detectormodel": detectorModel,
					"detectortype": detectorType,
					"detectormode": detectorMode
				},
				onSuccess: function () {
					Shipment.DatasetRetrieval.incrementSucceededCount("Created/updated IceBear dataset OK");
				},
				onFailure: function () {
					Shipment.DatasetRetrieval.incrementFailedCount("Failed creating/updating IceBear dataset");
				}
			});
		},


		gatherBeamlineDetails:function () {
			let beamlines = [];
			let namesDone = [];
			Shipment.DatasetRetrieval.datasets.forEach(function (ds) {
				let beamlineName = ds["BL"];
				if (undefined !== beamlineName && -1 === namesDone.indexOf(beamlineName)) {
					let detectorManufacturer=ds["DETECTORMANUFACTURER"];
					let detectorModel=ds["DETECTORMODEL"];
					let detectorMode=ds["DETECTORMODE"];
					let detectorType=ds["DETECTORTYPE"];
					//As of March 2021, Diamond returns null, "" or "1", which are useless, and doesn't provide detector type at all.
					if(!detectorManufacturer || 2>detectorManufacturer.length){ detectorManufacturer=""; }
					if(!detectorModel || 2>detectorModel.length){ detectorModel=""; }
					if(!detectorMode || 2>detectorMode.length){ detectorMode=""; }
					if(!detectorType || 2>detectorType.length){ detectorType=""; }
					let beamline={
						'name': beamlineName,
						'shipmentdestinationid': data['shipmentdestinationid'],
						"detectormanufacturer": detectorManufacturer,
						"detectormodel": detectorModel,
						"detectortype": detectorType,
						"detectormode": detectorMode
					};
					beamlines.push(beamline);
					namesDone.push(beamlineName);
				}
			});
			Shipment.DatasetRetrieval.updateIceBearBeamlines(beamlines);
		},

	} //end DatasetRetrieval

};