window.IspybRestShippingHandler={
		
		/**
		 * The base URI of the EXI instance.
		 * Populated from the shipment destination's baseuri property.
		 */
		baseUri:null,

		/**
		 * When authenticating with EXI, a "site" parameter is sent along with username and password.
		 * This object is a mapping of partial URI to the site parameter value. It will break if the synchrotron's
		 * ISPyB URI is changed - but checks in authenticate() below will catch this.
		 */
		sites:{
			'maxiv.lu.se':'MAXIV',
			'esrf.fr':'ESRF',
			'cells.es':'ALBA'
		},

		/**
		 * The authentication token to be passed to the synchrotron with each request after authentication.
		 * Received from the synchrotron on authentication.
		 */
		token:null,

		/*
		 * These are set and used during shipment submission
		 */
		shipmentIdAtFacility:"",
		safetyLevel:null,
		labContact:null,
		proteins:null,
		proposal:null,
		proposalName:null,
		session:null,
		proteinAcronymsInShipment:[],
		shipmentErrors:[],
		missingAcronyms:[],
		acronymToRemoteProteinId:{},

		SAMPLE_COMMENT_MAXLENGTH:250,

		/**
		 * Returns the URL of this shipment at the remote synchrotron, or false if the shipment has not been submitted.
		 * Strips /api(/) from base API and appends /shipment/sid/[REMOTE ID].
		 */
		getShipmentUrlAtFacility:function(shipmentDestination, shipmentIdAtFacility){
			if(!shipmentIdAtFacility &&!data.idatremotefacility){
				return false;
			} else if(!shipmentIdAtFacility){
				shipmentIdAtFacility=data.idatremotefacility;
			}
			let baseUri=shipmentDestination["baseuri"].trim("/");

			if(!baseUri){ 
				alert("Cannot get shipment URL. No shipping API base URI set for "+shipmentDestination.name+".");
				return false;
			}
			//https://ispyb.maxiv.lu.se/ispyb/ispyb-ws/rest/
			baseUri=baseUri.replace("/ispyb/ispyb-ws/rest/","");
			baseUri=baseUri.replace("//ispyb.","//exi.");

			return baseUri+"/mx/index.html#/shipping/"+shipmentIdAtFacility+"/main";
		},
		
		/**
		 * TODO No sample ID in response from saving puck
		 * Returns the URL of this crystal at the remote synchrotron, or false if the crystal does not have a remote ID.
		 * Strips /api(/) from base API and appends /samples/sid/[REMOTE ID].
		 *
		 */
		getCrystalUrlAtFacility:function(shipmentDestination, crystalIdAtFacility){
			if(!crystalIdAtFacility || ""===crystalIdAtFacility){
				return false;
			}
			let baseUri=shipmentDestination["baseuri"];
			if(!baseUri){ 
				alert("Cannot get crystal URL. No shipping API base URI set for "+shipmentDestination.name+".");
				return false;
			}
			baseUri=baseUri.replace("/ispyb/ispyb-ws/rest/","");
			baseUri=baseUri.replace("//ispyb.","//exi.");
			return baseUri+"/mx/index.html#/mx/proposal/"+IspybRestShippingHandler.proposalName+"/datacollection/sample/"+crystalIdAtFacility+"/main";
		},

		isInited: false,

		init: function(afterInit){
			if(IspybRestShippingHandler.isInited){ return afterInit(); }
			if(!window.shipmentDestination){
				alert("No shipment destination set - cannot determine shipping API location.");
				return false;
			}
			let baseUri=window.shipmentDestination["baseuri"];
			if(!baseUri){
				alert("No shipping API base URI set for "+window.shipmentDestination.name+".");
				return false;
			}
			IspybRestShippingHandler.baseUri=baseUri.replace(/\/+$/g,"")+"/";
			//Check still logged into IceBear, reset session clock - don't want to fail at the end.
			IspybRestShippingHandler.setProteinAcronymsInShipment();
			new Ajax.Request("/api/homepagebrick", {
				"method":"get",
				"onSuccess":afterInit,
				"onFailure":function(transport){
					if(401===transport.status){
						//IceBear session timed out. Reload page to show login form.
						ui.forceReload();
					} else {
						ui.keepAlive();
						IspybRestShippingHandler.isInited=true;
						afterInit();
					}
				}
			});

		},

		/**
		 * Begins the process of authenticating, choosing a proposal/session, and submitting to 
		 * the synchrotron. 
		 * 
		 * It is assumed that local validation of the shipment has already been done! This includes 
		 * basic checks such as: at least one dewar; each dewar has at least one puck; all crystals
		 * have protein acronyms. Some validation can only be done at the synchrotron, but we should
		 * not be sending them shipments that fail these basic checks.
		 */
		begin:function(){
			if(data["remoteidatfacility"]){
				alert("Shipment has already been sent.");
				return false;
			}
			IspybRestShippingHandler.init(IspybRestShippingHandler.openShippingDialog);
		},

		/**
		 * Iterates through all dewars, pucks, pins and adds the protein acronyms to IspybRestShippingHandler.proteinAcronymsInShipment.
		 * Note that this does no validation, not even for non-existent acronyms.
		 */
		setProteinAcronymsInShipment:function(){
			let dewars = document.querySelectorAll(".containertab+.tabbody");
			dewars.forEach(function (d) {
				let pucks = d.querySelectorAll(".treeitem");
				pucks.forEach(function (p) {
					let slots = p.querySelectorAll("tr.datarow");
					slots.forEach(function (tr) {
						if(!tr.rowData || !tr.rowData.childitems || !tr.rowData.childitems.length){
							return;
						}
						let pinAcronym=tr.rowData.childitems[1].proteinacronym;
						if(-1===IspybRestShippingHandler.proteinAcronymsInShipment.indexOf(pinAcronym)){
							IspybRestShippingHandler.proteinAcronymsInShipment.push(pinAcronym);
						}
					});
				});
			});
		},

		openShippingDialog:function(){
			ui.modalBox({ title:"Send shipment to "+window.shipmentDestination.name, content:"Getting list of proposals from "+window.shipmentDestination.name+"..." });
			IspybRestShippingHandler.getProposals();
		},

		/**
		 * Renders the login form into the modal box.
		 */
		showLoginForm:function(afterAuthenticate){
			let mb=document.getElementById("modalBox");
			mb.querySelector(".boxbody").innerHTML="";
			ui.setModalBoxTitle("Authenticate at "+window.shipmentDestination.name);
			let f=mb.form({
				action:IspybRestShippingHandler.baseUri+"authenticate",
				method:"post",
			});
			f.style.maxWidth="800px";
			let h=f.formField({
				label:"Authenticate with your "+window.shipmentDestination.name+" credentials", content:'&nbsp;'
			});
			h.addClassName("radiohead");
			f.textField({
				name:'remoteusername',
				label:window.shipmentDestination.name+" username",
				value:""
			});
			f.passwordField({
				name:'remotepassword',
				label:window.shipmentDestination.name+" password",
				value:""
			});
			fieldValidations.remoteusername="required";
			fieldValidations.remotepassword="required";
	
			f.submitButton({ label:"Authenticate" });
			f.onsubmit=function(){ IspybRestShippingHandler.authenticate(afterAuthenticate); return false; };
			document.getElementById("remoteusername").focus();
		},
		
		/**
		 * Validates that both username and password are present, then submits them to the remote API.
		 * Expects a token in return.
		 */
		authenticate:function(afterAuthenticate){
			let frm=document.getElementById("modalBox").querySelector(".boxbody form");
			let isValid=true;
			frm.querySelectorAll("input").forEach(function(f){
				if(!validator.validate(f)){ isValid=false; }
			});
			if(!isValid){ return false; }
			let site=false;
			Object.keys(IspybRestShippingHandler.sites).forEach(function (key){
				if(-1!==IspybRestShippingHandler.baseUri.indexOf(key)){
					site=IspybRestShippingHandler.sites[key];
				}
			});
			if(!site){
				alert("No entry in sites lookup table for "+window.shipmentDestination.name+"\n\nCannot authenticate. See your administrator.");
				return false;
			}

			let username = frm.remoteusername.value;
			let password = frm.remotepassword.value;
			frm.querySelector("input[type=submit]").closest("label").classList.add("updating");
			AjaxUtils.remoteAjax(
					IspybRestShippingHandler.baseUri+"authenticate?site="+site,
					'post',
					'login='+username+'&password='+password,
					function(transport){
						IspybRestShippingHandler.authenticate_onSuccess(transport, afterAuthenticate);
					},
					function(transport){
						IspybRestShippingHandler.authenticate_onFailure(transport, afterAuthenticate);
					}
			);
			return false;
		},
		/**
		 * Success handler for remote synchrotron authentication. Calls getProposals().
		 *
		 * Typical success response:
		 *   {"token":"8e993500f813f8fdb5b6caa75cf23772936289bb","roles":["User"]}
		 *
		 * A bad login appears to return a Java error:
		 *   JBAS011843: Failed instantiate InitialContextFactory com.sun.jndi.ldap.LdapCtxFactory from classloader
		 *   ModuleClassLoader for Module "deployment.ispyb.ear.ispyb-ws.war:main" from Service Module Loader
		 *
		 * @param {XMLHttpRequest} transport The response object.
		 * @param {function} afterAuthenticate A function to call after successful authentication
		 */
		authenticate_onSuccess:function(transport,afterAuthenticate){
			if(!transport.responseJSON || !transport.responseJSON["token"]){
				if(transport.responseJSON && transport.responseJSON["error"]){
					IspybRestShippingHandler.authenticate_onFailure(transport);
				} else {
					IspybRestShippingHandler.showErrorAtAuthenticate("Could not log you in. Check your username and password.");
				}
			} else {
				IspybRestShippingHandler.token=transport.responseJSON["token"];
				window.setTimeout(afterAuthenticate,50);
			}
		},
		/**
		 * Failure handler for remote synchrotron authentication.
		 * @param transport
		 */
		authenticate_onFailure:function(transport){
			let msg="Could not log you in. Check your username and password.";
			if(transport.responseJSON && transport.responseJSON.error){
				msg=window.shipmentDestination.name+" said: "+transport.responseJSON.error;
			}
			IspybRestShippingHandler.showErrorAtAuthenticate(msg);
		},
		/**
		 * Renders the authentication error message below the form.
		 * @param {String} msg The error message.
		 */
		showErrorAtAuthenticate:function(msg){
			let frm=document.getElementById("modalBox").querySelector(".boxbody").down("form");
			frm.querySelector("input[type=submit]").closest("label").classList.remove("updating");
			let ae=document.getElementById("autherror");
			if(!ae){
				frm.innerHTML+='<div style="text-align:left" id="autherror" class="errorbar"></div>';
			}
			ae.innerHTML=msg;
		},


		/**
		 * Fetches a list of the user's proposals from the synchrotron.
		 * On failure, triggers authentication.
		 *
		 * Typical response:
		 * [{"Proposal_proposalCode":"MX","Proposal_proposalType":"MX","Proposal_personId":343,
		 * "Proposal_title":"ICEBEAR API test","Proposal_proposalId":121,"Proposal_proposalNumber":"20200573"}]
		 *
		 */
		getProposals:function(){
			if(!IspybRestShippingHandler.baseUri){ 
				alert("No base URI supplied. Call begin() first, with base URI."); 
				return false;
			}
			if(!IspybRestShippingHandler.token){ 
				IspybRestShippingHandler.showLoginForm(IspybRestShippingHandler.getProposals);
				return false;
			}
			AjaxUtils.remoteAjax(
					IspybRestShippingHandler.baseUri+IspybRestShippingHandler.token+'/proposal/list',
					'get',
					{},
					IspybRestShippingHandler.getProposals_onSuccess,
					IspybRestShippingHandler.getProposals_onFailure
			);
		},
		getProposals_onSuccess:function(transport){
			if(!transport.responseJSON || transport.responseJSON.error){
				return IspybRestShippingHandler.getProposals_onFailure(transport);
			}
			IspybRestShippingHandler.renderProposalList(transport.responseJSON);
		},
		getProposals_onFailure:function(transport){
			let msg="Error getting list of proposals";
			if(transport.responseJSON && transport.responseJSON.error){
				msg=window.shipmentDestination.name+" said: "+transport.responseJSON.error;
			}
			let mb=(document.getElementById("modalBox").querySelector(".boxbody"));
			ui.setModalBoxTitle("Could not retrieve list of proposals from "+window.shipmentDestination.name);
			mb.innerHTML=msg;
		},
		renderProposalList:function(response){
			let mb=document.getElementById("modalBox").querySelector(".boxbody");
			if(response.length===0){
				ui.setModalBoxTitle("No proposals at "+window.shipmentDestination.name);
				mb.innerHTML='No proposals found using your '+window.shipmentDestination.name+' credentials. Cannot submit shipment.';
				return false;
			}
			ui.setModalBoxTitle("Choose your proposal and session at "+window.shipmentDestination.name);
			mb.innerHTML='';
			response.forEach(function(p){
				//ESRF may return other kinds of proposal. MAX-IV will be all MX.
				if("mx"===p["Proposal_proposalCode"].toLowerCase()){
					let proposalName=p["Proposal_proposalCode"]+ p["Proposal_proposalNumber"];
					let proposalTreeItem=mb.treeItem({
						record:p,
						header:proposalName+": "+p["Proposal_title"],
						content:'Getting details for this proposal, please wait a moment...'
					});
					proposalTreeItem.querySelector(".treebody").innerHTML="";
					proposalTreeItem.dataset.proposalName=proposalName;
					proposalTreeItem.errors=[];
					IspybRestShippingHandler.getFullProposal(proposalTreeItem);
					IspybRestShippingHandler.getProteinsForProposal(proposalTreeItem);
					IspybRestShippingHandler.getSessionsForProposal(proposalTreeItem);
					IspybRestShippingHandler.renderProposalTreeItemContent(proposalTreeItem);
				}
			});
		},

		/**
		 * Waits for the full proposal, its proteins and sessions to be available, attempts to create any proteins in the
		 * proposal whose acronyms exist in the shipment but not the proposal, then renders the content of the
		 * proposal tree item. If any errors are present, the tree item will be highlighted.
		 * @param proposalTreeItem
		 * @returns {boolean}
		 */
		renderProposalTreeItemContent:function(proposalTreeItem){
			//If any of these are undefined, we are still waiting for AJAX responses. Wait and try again.
			if(undefined===proposalTreeItem.dataset.hasLabContactErrors || undefined===proposalTreeItem.dataset.hasProteinErrors || undefined===proposalTreeItem.dataset.hasSessionErrors ){
				window.setTimeout(function () {
					IspybRestShippingHandler.renderProposalTreeItemContent(proposalTreeItem);
				},500);
				return false;
			}
			let proposalTreeBody=proposalTreeItem.querySelector(".treebody");
			//If we got here, we can render the list of sessions on this proposal - which should have been filtered to
			// include only those ending in the future.
			proposalTreeBody.table({
				contentBefore:proposalTreeBody.innerHTML,
				headers:["Beamline","Start","End",""],
				cellTemplates:["{{beamLineName}}","{{BLSession_startDate}}","{{BLSession_endDate}}",
					'<input type="button" value="Use this session" onclick="IspybRestShippingHandler.chooseSession(this)" />']
			}, proposalTreeItem.sessions);
		},

		chooseSession:function(btn){
			IspybRestShippingHandler.missingAcronyms=btn.closest(".treeitem").missingAcronyms;
			IspybRestShippingHandler.session=btn.closest("tr").rowData;
			IspybRestShippingHandler.acronymToRemoteProteinId={};
			btn.closest(".treeitem").record["proteins"].forEach(function (protein) {
				IspybRestShippingHandler.acronymToRemoteProteinId[protein["acronym"]]=protein["proteinId"];
			});
			IspybRestShippingHandler.chooseLabContact(btn);
		},



		/**
		 * Fetches the full details of the proposal, including its lab contacts, any crystals, etc.
		 * If found, attaches this to the proposal tree item.
		 * @param proposalTreeItem The tree item element.
		 */
		getFullProposal:function(proposalTreeItem){
			let proposalName=proposalTreeItem.dataset.proposalName;
			AjaxUtils.remoteAjax(
				IspybRestShippingHandler.baseUri+IspybRestShippingHandler.token+"/proposal/"+proposalName+"/info/get",
				'get',
				'',
				function(transport){ IspybRestShippingHandler.getFullProposal_onSuccess(transport, proposalTreeItem) },
				function(transport){ IspybRestShippingHandler.getFullProposal_onFailure(transport, proposalTreeItem) }
			);
		},
		getFullProposal_onSuccess:function(transport,proposalTreeItem){
			//Response is not a proposal object, but an array containing it.
			proposalTreeItem.record=transport.responseJSON[0];
			if(!transport.responseJSON || !transport.responseJSON[0] || !transport.responseJSON[0]["labcontacts"] || 0===transport.responseJSON[0]["labcontacts"].length){
				ui.warningMessageBar("The proposal does not contain any lab contacts, so cannot be used for this shipment.", proposalTreeItem.querySelector(".treebody"));
				proposalTreeItem.dataset.hasLabContactErrors="1";
			} else {
				proposalTreeItem.labContacts=transport.responseJSON[0]["labcontacts"];
				proposalTreeItem.dataset.hasLabContactErrors="0";
			}
		},
		getFullProposal_onFailure:function(transport,proposalTreeItem){
			proposalTreeItem.errors.push("Could not fetch the full details of this proposal.");
			proposalTreeItem.dataset.hasLabContactErrors="1";
		},


		/**
		 * Fetches the proteins on the proposal.
		 * If found, attaches this to the proposal tree item.
		 * @param proposalTreeItem The tree item element.
		 */
		getProteinsForProposal:function(proposalTreeItem){
			let proposalName=proposalTreeItem.dataset.proposalName;
			AjaxUtils.remoteAjax(
				IspybRestShippingHandler.baseUri+IspybRestShippingHandler.token+"/proposal/"+proposalName+"/mx/protein/list",
				'get',
				'',
				function(transport){ IspybRestShippingHandler.getProteinsForProposal_onSuccess(transport, proposalTreeItem) },
				function(transport){ IspybRestShippingHandler.getProteinsForProposal_onFailure(transport, proposalTreeItem) }
			);
		},
		getProteinsForProposal_onSuccess:function(transport, proposalTreeItem){
			if(undefined===transport.responseJSON) {
				return IspybRestShippingHandler.getProteinsForProposal_onFailure(transport, proposalTreeItem);
			} else if(0===transport.responseJSON.length){
				//no proteins is a 200 OK with []
				ui.errorMessageBar("The proposal does not have any protein acronyms, so cannot be used for this shipment.", proposalTreeItem.querySelector(".treebody"));
				proposalTreeItem.errors.push("The proposal does not have any protein acronyms, so cannot be used for this shipment.");
				proposalTreeItem.dataset.hasProteinErrors="1";
				return;
			}
			let proposalAcronyms=[];
			transport.responseJSON.forEach(function (protein) {
				proposalAcronyms.push(protein["acronym"]);
			});
			let missingAcronyms=[];
			for(let i=0;i<IspybRestShippingHandler.proteinAcronymsInShipment.length;i++){
				let shipmentAcronym=IspybRestShippingHandler.proteinAcronymsInShipment[i];
				if(-1===proposalAcronyms.indexOf(shipmentAcronym)){
					missingAcronyms.push(shipmentAcronym);
				}
			}
			proposalTreeItem.missingAcronyms=missingAcronyms;
			proposalTreeItem.dataset.hasProteinErrors="0"; //Will set to 1 if we can't create the proteins
		},
		getProteinsForProposal_onFailure:function(transport, proposalTreeItem){
			ui.errorMessageBar("There was a problem. Could not retrieve the list of proteins for this proposal.", proposalTreeItem.querySelector(".treebody"));
			proposalTreeItem.errors.push("There was a problem. Could not retrieve the list of proteins for this proposal.");
			proposalTreeItem.dataset.hasProteinErrors="1";
		},

		/**
		 * Fetches the sessions for the proposal.
		 * If found, iterates through them and attaches those ending in the future to the proposal tree item.
		 * If no future sessions are found, this is an error.
		 * @param proposalTreeItem The tree item element.
		 */
		getSessionsForProposal:function(proposalTreeItem){
			let proposalName=proposalTreeItem.dataset.proposalName;
			AjaxUtils.remoteAjax(
				IspybRestShippingHandler.baseUri+IspybRestShippingHandler.token+"/proposal/"+proposalName+"/session/list",
				'get',
				'',
				function(transport){ IspybRestShippingHandler.getSessionsForProposal_onSuccess(transport, proposalTreeItem) },
				function(transport){ IspybRestShippingHandler.getSessionsForProposal_onFailure(transport, proposalTreeItem) }
			);
		},
		getSessionsForProposal_onSuccess:function(transport, proposalTreeItem){
			if(!transport.responseJSON){ return IspybRestShippingHandler.getProteinsForProposal_onFailure(transport, proposalTreeItem); }
			//sessions are returned as an array, with dates in "Jun 23, 2020 1:00:00 AM" format
			let futureSessions=[];
			let now=new Date();
			transport.responseJSON.forEach(function(session){
				//BLSession_endDate: "Jun 24, 2020 8:59:59 AM"
				let endDate=Date.parse(session['BLSession_endDate']);
				if(endDate>now){
					futureSessions.push(session);
				}
			});
			if(0===futureSessions.length){
				ui.errorMessageBar("There are no sessions on this proposal, or all of them have already ended.", proposalTreeItem.querySelector(".treebody"));
				proposalTreeItem.errors.push("There are no sessions on this proposal, or all of them have already ended.");
				proposalTreeItem.dataset.hasSessionErrors="1";
			} else {
				proposalTreeItem.sessions={ total:futureSessions.length, rows:futureSessions };
				proposalTreeItem.dataset.hasSessionErrors="0";
			}
		},
		getSessionsForProposal_onFailure:function(transport, proposalTreeItem){
			ui.errorMessageBar("There was a problem. Could not retrieve the list of sessions for this proposal.", proposalTreeItem.querySelector(".treebody"));
			proposalTreeItem.errors.push("There was a problem. Could not retrieve the list of sessions for this proposal.");
			proposalTreeItem.dataset.hasSessionErrors="1";
		},


		/**
		 * Handles the selection of the lab contact.
		 * @param {Element} btn The "Send shipment" button.
		 */
		chooseLabContact:function(btn){
			let session=btn.closest("tr").rowData;
			let proposalElement=btn.closest("div.treeitem");
			let proposal=proposalElement.record;
			let labContacts=proposalElement.labContacts;
			if(undefined===labContacts){
				//Shouldn't happen - not fetching by AJAX
				alert("Still waiting for list of lab contacts for this proposal. Try again in a moment.");
				return false;
			} else if(0===labContacts.length){
				//Shouldn't have got this far because we check this when rendering the proposal
				alert("This proposal has no lab contacts. You need to define at least one lab contact in "+window.shipmentDestination.name+"'s ISPyB.");
				return false;
			}

			labContacts.forEach(function(lc){
				lc.contactName=lc["personVO"]["familyName"]+", "+lc["personVO"]["givenName"];
				lc.contactLab=lc["personVO"]["laboratoryVO"]["name"];
			});

			IspybRestShippingHandler.proposal=proposal;
			IspybRestShippingHandler.proposalName=proposalElement.dataset.proposalName;
			IspybRestShippingHandler.session=session;
			let box=document.getElementById("modalBox").querySelector(".boxbody");
			box.innerHTML='';
			box.table({
				headers:[ 'Name','Organisation',''],
				cellTemplates:[ '{{contactName}}','{{contactLab}}','<input type="button" value="Choose" onclick="IspybRestShippingHandler.setLabContact(this)" />' ],
				contentBefore:'Choose the lab contact for the shipment:'
			}, {'total':labContacts.length, 'rows':labContacts });
			ui.setModalBoxTitle("Choose the lab contact for this shipment");
		},
		setLabContact:function(btn){
			IspybRestShippingHandler.labContact=btn.closest("tr").rowData;
			IspybRestShippingHandler.showFinalConfirmation();
		},

		showFinalConfirmation:function(){
			let box=document.getElementById("modalBox").querySelector(".boxbody");
			let labContact=IspybRestShippingHandler.labContact;
			box.innerHTML='';
			ui.setModalBoxTitle("Check the details before sending the shipment");
			let f=box.form({ action:'#', method:'post' });
			f.setStyle({ position:"absolute", top:"1%", left:"1%", width:"48%" });
			f.textField({ readonly:true, label:'Destination', value:window.shipmentDestination.name });
			f.textField({ readonly:true, label:'Proposal', value:IspybRestShippingHandler.proposalName });
			f.textField({ readonly:true, label:'Home lab contact', value:labContact["contactName"] });

			let f2=box.form({ action:'#', method:'post' });
			f2.setStyle({ position:"absolute", top:"1%", right:"1%", width:"48%", marginTop:0, paddingTop:0 });
			let b=f2.formField({ readonly:true, label:'Review this information to make sure that it is correct, then click "Send shipment".', content:'<input type="button" value="Send shipment" onclick="IspybRestShippingHandler.sendShipment()" />' });
			b.setStyle({ "height":f.getHeight()+"px"});
			b.down("input").setStyle({ position:"absolute", bottom:"0.5em", right:"0.5em"});

			box.innerHTML+='<div style="position:absolute;bottom:10%; left:1%; width:98%; text-align:center"><a href="#" onclick="ui.closeModalBox();return false">Cancel - don\'t send this shipment</a></div>';
		},
		
		sendShipment:function(){
			IspybRestShippingHandler.shipmentErrors=[];
			if(!confirm("Really submit shipment to "+window.shipmentDestination.name+"?")){ return false; }
			ui.setModalBoxTitle("Submitting your shipment to "+window.shipmentDestination.name+"...");
			document.getElementById("modalBox").querySelector(".boxbody").innerHTML="";
			ui.logToDialog('Beginning to send shipment...');
			IspybRestShippingHandler.markAllPinsUnsaved();
			IspybRestShippingHandler.watchForAllPinsSaved();
			IspybRestShippingHandler.doPreflight();
		},

		markAllPinsUnsaved:function(){
			document.querySelectorAll("tr.datarow").forEach(function(tr){
				if(tr.rowData && 1!==parseInt(tr.rowData.isEmpty)) {
					tr.classList.add("unsavedPin");
				}
			});

		},

		doPreflight:function(){
			let box=document.getElementById("modalBox").querySelector(".boxbody");
			let errors=Shipment.getShipmentErrors(); //verify nothing silly like an empty dewar

			if(0<errors.length){
				box.innerHTML="Could not send shipment. Please fix the following:<br/><ul><li>"+errors.join("</li><li>")+"</ul>";
				return false;
			}

			ui.logToDialog("Pre-flight validation complete. Creating shipment in ISPyB...");
			IspybRestShippingHandler.remoteErrors=[];
			IspybRestShippingHandler.attemptToCreateMissingProteins();
		},

		/**
		 * If the shipment contains a protein acronym that is not listed on the proposal, attempt to add it to the proposal.
		 * This should succeed at MAX-IV, which allows adding on the fly. It should fail at ESRF, where protein acronyms
		 * for all samples must exist on the proposal before the shipment can be accepted.
		 */
		attemptToCreateMissingProteins:function(){
			if(0===IspybRestShippingHandler.missingAcronyms.length){
				IspybRestShippingHandler.createEmptyShipment();
				return;
			}
			let proteinAcronym=IspybRestShippingHandler.missingAcronyms[0];
			AjaxUtils.remoteAjax(
				IspybRestShippingHandler.baseUri+IspybRestShippingHandler.token+"/proposal/"+IspybRestShippingHandler.proposalName+"/mx/protein/save",
				'post',
				{
					proteinId:0,
					name:proteinAcronym,
					acronym:proteinAcronym
				},
				IspybRestShippingHandler.attemptToCreateMissingProteins_onSuccess,
				IspybRestShippingHandler.attemptToCreateMissingProteins_onFailure
			);
		},
		attemptToCreateMissingProteins_onSuccess:function(transport){
			//Re-route failures that got here because of a 200 OK status code
			if(!transport.responseJSON || !transport.responseJSON.acronym){
				return IspybRestShippingHandler.attemptToCreateMissingProteins_onFailure(transport);
			}
			//Remove the newly created acronym from the missing list
			let index=IspybRestShippingHandler.missingAcronyms.indexOf(transport.responseJSON.acronym);
			if(-1!==index){
				IspybRestShippingHandler.missingAcronyms.splice(index, 1);
			}
			//Add acronym to acronym-remoteProteinId mapping, for use later
			IspybRestShippingHandler.acronymToRemoteProteinId[transport.responseJSON.acronym]=transport.responseJSON.proteinId;
			//And go round again, to create the next missing protein
			window.setTimeout(IspybRestShippingHandler.attemptToCreateMissingProteins, 50);
		},
		attemptToCreateMissingProteins_onFailure:function(){
			IspybRestShippingHandler.logRemoteError("Could not create missing protein in ISPyB.");
			IspybRestShippingHandler.logRemoteError("You will need to create any missing proteins in ISPyB, on the proposal you want to use.");
			IspybRestShippingHandler.showRemoteErrors();
		},


		remoteErrors:[],
		logRemoteError:function(err){
			IspybRestShippingHandler.remoteErrors.push(err);
		},
		showRemoteErrors:function(){
			ui.logToDialog('The following errors occurred:<br/><br/>','error');
			ui.logToDialog(IspybRestShippingHandler.remoteErrors.join('<br/>'));
		},

		/**
		 * Creates a shipment at the synchrotron.
		 */
		createEmptyShipment:function(){
			ui.logToDialog("Submitting basic shipment details");
			let labContactId=IspybRestShippingHandler["labContact"]["labContactId"];
			let shipmentName=document.getElementById("shipmentdetails_form").querySelector("#name").value;

			AjaxUtils.remoteAjax(
				IspybRestShippingHandler.baseUri+IspybRestShippingHandler.token+"/proposal/"+IspybRestShippingHandler.proposalName+"/shipping/save",
				'post',
				{
					"name": shipmentName,
					"sendingLabContactId": labContactId,
					"returnLabContactId": -1, //same as outbound
					"comments": "IceBear: "+document.location.href,
					returnCourier:"-1",
					courierAccount:"",
					billingReference:"",
					dewarAvgCustomsValue:"0",
					dewarAvgTransportValue:"0",
					sendersID:data.id,
					sendersURL:document.location.href,
					sessionId:IspybRestShippingHandler.session["sessionId"]
				},
				IspybRestShippingHandler.createEmptyShipment_onSuccess,
				IspybRestShippingHandler.createEmptyShipment_onFailure
			);
		},
		createEmptyShipment_onSuccess:function(transport){
			if(transport.responseJSON && transport.responseJSON["shippingId"]){
				IspybRestShippingHandler.shipmentIdAtFacility=transport.responseJSON["shippingId"];
				window.setTimeout(IspybRestShippingHandler.createDewars, 50);
			} else {
				IspybRestShippingHandler.logRemoteError("No shipment ID returned from shipment create attempt.");
				return IspybRestShippingHandler.createEmptyShipment_onFailure(transport);
			}
		},
		createEmptyShipment_onFailure:function(transport){
			IspybRestShippingHandler.logRemoteError("Could not create shipment.");
			if(transport.responseJSON && transport.responseJSON["error"]){
				IspybRestShippingHandler.logRemoteError(transport.responseJSON["error"]);
			} else {
				IspybRestShippingHandler.logRemoteError(transport.responseText);
			}
			IspybRestShippingHandler.showRemoteErrors();
		},

		createDewars:function(){
			ui.logToDialog("Creating dewars in remote shipment...");
			let dewarTabs=document.querySelectorAll(".containertab");
			dewarTabs.forEach(function (dt) {
				IspybRestShippingHandler.createDewar(dt.dataset.containername);
			});
		},
		createDewar:function(dewarName){
			if(!dewarName || ""===dewarName){
				IspybRestShippingHandler.logRemoteError("Could not create shipment.");
				IspybRestShippingHandler.showRemoteErrors();
			}
			ui.logToDialog("Creating dewar "+dewarName+" in remote shipment...");
			AjaxUtils.remoteAjax(IspybRestShippingHandler.baseUri+IspybRestShippingHandler.token+"/proposal/"+
										IspybRestShippingHandler.proposalName+"/shipping/"+
										IspybRestShippingHandler.shipmentIdAtFacility+"/dewar/save",
				"post",
				{
					code:dewarName,
					transportValue:"",
					storageLocation:"N/A",
					comments:"",
					shippingId: IspybRestShippingHandler.shipmentIdAtFacility
				},
				function(transport){ IspybRestShippingHandler.createDewar_onSuccess(transport, dewarName); },
				IspybRestShippingHandler.createDewar_onFailure
			);
		},
		createDewar_onSuccess:function(transport, dewarName){
			// Returned JSON should represent the entire shipment
			// {
			//  "shippingId":604,
			//  ...
			// 	"dewarVOs":[
			// 		{
			// 			"dewarId":2218,"code":"DEWAR1","comments":"","storageLocation":"N/A","dewarStatus":null,
			// 			"timeStamp":null,"isStorageDewar":null,"barCode":"MAXIV02218","customsValue":null,
			// 			"transportValue":0,"trackingNumberToSynchrotron":null,"trackingNumberFromSynchrotron":null,
			// 			"facilityCode":null,"type":"Dewar","isReimbursed":false,
			// 			"containerVOs":[
			//
			// 			],
			// 			"sessionVO":null
			// 		}
			// 	],
			// 	"sessions":[]
			// 	}
			if(!transport.responseJSON || !transport.responseJSON["dewarVOs"] || 0===transport.responseJSON["dewarVOs"].length){
				return IspybRestShippingHandler.createDewar_onFailure(dewarName);
			}
			let dewarCreated=false;
			transport.responseJSON["dewarVOs"].forEach(function(remoteDewar){
				if(remoteDewar["code"]===dewarName){
					let dewarTab=document.getElementById(dewarName);
					dewarTab.remoteDewar=remoteDewar;
					dewarTab.dataset.remoteid=remoteDewar["dewarId"];
					dewarTab.removeClassName("noRemoteDewar");
					dewarCreated=true;
				}
			});
			if(!dewarCreated){
				return IspybRestShippingHandler.createDewar_onFailure(dewarName);
			}
			ui.logToDialog("...added "+dewarName+" to shipment.");
			window.setTimeout(function(){ IspybRestShippingHandler.addPucksToDewar(dewarName) }, 50);
		},
		createDewar_onFailure:function(dewarName){
			IspybRestShippingHandler.logRemoteError("Could not create dewar "+dewarName+" in remote shipment.");
			IspybRestShippingHandler.showRemoteErrors();
		},

		/**
		 * Populate a dewar with its child pucks, at the synchrotron.
		 */
		addPucksToDewar:function(dewarName){
			ui.logToDialog("Adding pucks to dewar "+dewarName+"...");
			let tabBody=document.getElementById(dewarName+"_body");
			let pucks=tabBody.querySelectorAll(".treeitem");
			pucks.forEach(function(p){
				let localPuck=p.record;
				p.id=p.record.name;
				IspybRestShippingHandler.addPuckToDewar(localPuck);
			});
		},
		
		/**
		 * Add a puck to a dewar within the shipment, at the synchrotron.
		 */
		addPuckToDewar:function(localPuck){
			let localDewarTab=document.getElementById(localPuck["name"]).closest(".tabbody").previousElementSibling;
			let dewarName=localDewarTab.id;
			ui.logToDialog("Adding puck "+localPuck.name+" to "+dewarName+"...");
			let remoteDewarId=localDewarTab.dataset.remoteid;
			AjaxUtils.remoteAjax(
				IspybRestShippingHandler.baseUri+IspybRestShippingHandler.token+"/proposal/"+IspybRestShippingHandler.proposalName+
						"/shipping/"+IspybRestShippingHandler.shipmentIdAtFacility+"/dewar/"+remoteDewarId+
						"/containerType/"+localPuck["containertypename"]+"/capacity/"+localPuck["positions"]+"/container/add",
				'get',
				{},
				function(transport){ IspybRestShippingHandler.addPuckToDewar_onSuccess(transport, localPuck) },
				function(transport){ IspybRestShippingHandler.addPuckToDewar_onFailure(transport, localPuck) },
			);
		},
		addPuckToDewar_onSuccess:function(transport, localPuck){
			if(!transport.responseJSON || !transport.responseJSON["containerId"]){
				return IspybRestShippingHandler.addPuckToDewar_onFailure();
			}
			ui.logToDialog("...added "+localPuck.name+" to dewar.");
			localPuck.remoteId=transport.responseJSON["containerId"];
			IspybRestShippingHandler.addSamplesToPuck(localPuck);			
		},
		addPuckToDewar_onFailure:function(){
			IspybRestShippingHandler.logRemoteError("Could not add puck to dewar.");
			IspybRestShippingHandler.showRemoteErrors();
		},

		addSamplesToPuck:function(localPuck){
			ui.logToDialog("Adding samples to puck "+localPuck.name+"...");
			let puckElement=document.getElementById(localPuck.name);
			let remotePuckId=localPuck.remoteId;
			let remoteDewarId=puckElement.closest(".tabbody").previousElementSibling["remoteDewar"]["dewarId"];
			let positions=puckElement.querySelectorAll("tr.datarow");

			let remotePuck= {
				"containerId": remotePuckId, "code": localPuck["name"],
				"containerType": localPuck["containertypename"], "capacity": localPuck["positions"],
				"beamlineLocation": IspybRestShippingHandler.session["beamLineName"], "sampleChangerLocation": null,
				"containerStatus": null, "barcode": null,
				"sampleVOs":[]
			};

			positions.forEach(function(tr){
				if(tr.rowData && 1!==parseInt(tr.rowData.isEmpty)) {
					let pin = tr.rowData;
					let proteinName=pin["proteinname"];
					let proteinAcronym=pin["proteinacronym"];
					let remoteProteinId=IspybRestShippingHandler.acronymToRemoteProteinId[proteinAcronym];
					let position=pin.position;
					let pinBarcode=pin.name;
					if(0===pinBarcode.indexOf("dummypin")){
						pinBarcode="";
					}
					let crystal=pin.childitems[1]; //0 is dummy
					let sampleName=crystal.name.replace(" ","_"); //Don't send sample names with spaces
					let shippingComment=crystal.shippingcomment.split('"').join('');

					let joiner=" -IceBear: ";
					let localUrl=window.document.location.protocol+"//"+window.document.location.hostname+"/crystal/"+crystal.id;
					if(shippingComment.length+joiner.length+localUrl.length <= IspybRestShippingHandler.SAMPLE_COMMENT_MAXLENGTH){
						shippingComment+=localUrl;
					}

					let remoteSample={
						"name":sampleName,"BLSample_code":pinBarcode,"location":position,"comments":shippingComment,
						"crystalVO":{
							"proteinVO":{
								"proteinId":remoteProteinId,"name":proteinName,"acronym":proteinAcronym,"safetyLevel":null,"molecularMass":null,
								"proteinType":null,"sequence":null,"personId":null,"timeStamp":null,
								"isCreatedBySampleSheet":null,"externalId":null
							},
							"spaceGroup":"P1","cellA":0,"cellB":0,"cellC":0,"cellAlpha":0,"cellBeta":0,"cellGamma":0
						},
						"diffractionPlanVO":{
							"radiationSensitivity":null,"requiredCompleteness":null,"requiredMultiplicity":null,
							"requiredResolution":null, "observedResolution":null,"preferredBeamDiameter":null,
							"numberOfPositions":null,"experimentKind":"Default"
						},
						"sendersID":crystal.id,
						"sendersURL":localUrl
					};
					if(""!==crystal.spacegroup){ remoteSample["crystalVO"]["spaceGroup"]=crystal.spacegroup.replace("_",""); }
					if(""!==crystal.unitcella){ remoteSample["crystalVO"]["cellA"]=1*crystal.unitcella; }
					if(""!==crystal.unitcellb){ remoteSample["crystalVO"]["cellB"]=1*crystal.unitcellb; }
					if(""!==crystal.unitcellc){ remoteSample["crystalVO"]["cellC"]=1*crystal.unitcellc; }
					if(""!==crystal.unitcellalpha){ remoteSample["crystalVO"]["cellAlpha"]=1*crystal.unitcellalpha; }
					if(""!==crystal.unitcellbeta) { remoteSample["crystalVO"]["cellBeta"] =1*crystal.unitcellbeta;  }
					if(""!==crystal.unitcellgamma){ remoteSample["crystalVO"]["cellGamma"]=1*crystal.unitcellgamma; }

					remotePuck.sampleVOs.push(remoteSample);
				}
			});
			AjaxUtils.remoteAjax(IspybRestShippingHandler.baseUri+IspybRestShippingHandler.token+"/proposal/"+
									IspybRestShippingHandler.proposalName+"/shipping/"+IspybRestShippingHandler.shipmentIdAtFacility+
									"/dewar/"+remoteDewarId+"/puck/"+remotePuckId+"/save",
				"post",
				{
					puck:JSON.stringify(remotePuck)
				},
				IspybRestShippingHandler.addSamplesToPuck_onSuccess,
				IspybRestShippingHandler.addSamplesToPuck_onFailure
			);
		},
		addSamplesToPuck_onSuccess:function(transport){
			if(!transport.responseJSON || !transport.responseJSON["containerId"]){
				return IspybRestShippingHandler.addSamplesToPuck_onFailure(transport);
			}
			let puckElement=document.getElementById(transport.responseJSON["code"]);
			transport.responseJSON["sampleVOs"].forEach(function(remoteSample){
				//IceBear puck table may have empty rows. ISPyB puck response doesn't.
				//So we can't just match on array positions and have to iterate.
				let positions=puckElement.querySelectorAll("tr.unsavedPin");
				positions.forEach(function (tr) {
					if(tr.rowData && 1!==parseInt(tr.rowData.isEmpty) && tr.rowData.samplename===remoteSample["name"]) {
						let pin = tr.rowData;
						let diffractionRequestId=pin.childitems[1]["diffractionrequestid"];
						IspybRestShippingHandler.addShippedNote(pin.childitems[1]);

						let remoteCrystalId=remoteSample["blSampleId"];
						//Or do we want the crystalVO ID?
						//let remoteCrystalId=remoteSample["crystalVO"]["crystalId"];
						if(remoteCrystalId){
							IspybRestShippingHandler.saveRemoteCrystalId(diffractionRequestId, remoteCrystalId, tr);
						} else {
							tr.classList.remove("unsavedPin");
						}
					}
				});
			});


		},
		addSamplesToPuck_onFailure:function(){
			//TODO Puck name
			ui.logToDialog("<strong>Adding samples to puck failed.</strong>","error");
		},

		addShippedNote:function(localCrystal){
			//TODO Move this to server-side, shipment::update - eliminate duplication in future handlers
			let noteText='Crystal shipped to '+window.shipmentDestination.name;
			new Ajax.Request('/api/note',{
				method:'post',
				parameters:{
					csrfToken:csrfToken,
					parentid:localCrystal.id,
					text:noteText
				},
				onSuccess:function(transport){ if(successCallback){ successCallback(transport.responseJSON); } },
				onFailure:AjaxUtils.checkResponse
			});
		},
		
		saveRemoteCrystalId:function(diffractionRequestId, remoteCrystalId, pinTr){
			new Ajax.Request('/api/diffractionrequest/'+diffractionRequestId,{
				'method':'patch',
				'parameters':{
					'csrfToken':csrfToken,
					'shipmentid':data.id,
					'crystalidatremotefacility':remoteCrystalId,
					'crystalurlatremotefacility':IspybRestShippingHandler.getCrystalUrlAtFacility(window.shipmentDestination, remoteCrystalId)
				},
				onSuccess:function(){
					pinTr.classList.remove("unsavedPin");
				},
				onFailure:function(){
					pinTr.classList.remove("unsavedPin");
				},
			});
			
		},

		watchForAllPinsSaved:function(){
			if(document.querySelector(".unsavedPin")){
				window.setTimeout(IspybRestShippingHandler.watchForAllPinsSaved, 500);
				return;
			}
			window.setTimeout(IspybRestShippingHandler.completeShipment,50);
		},
		
		completeShipment:function(){
			ui.logToDialog("Shipment was submitted to "+window.shipmentDestination.name+".","success");			
			ui.logToDialog("Updating IceBear and creating shipment manifest...");
			let shippedDate=new Date().toISOString().split("T")[0];
			new Ajax.Request('/api/shipment/'+data.id,{
				'method':'patch',
				'parameters':{
					'csrfToken':csrfToken,
					'dateshipped':shippedDate,
					'idatremotefacility':IspybRestShippingHandler.shipmentIdAtFacility,
					'urlatremotefacility':IspybRestShippingHandler.getShipmentUrlAtFacility(window.shipmentDestination, IspybRestShippingHandler.shipmentIdAtFacility),
					'proposalname':IspybRestShippingHandler.proposalName,
				},
				onSuccess:IspybRestShippingHandler.completeShipment_onSuccess,
				onFailure:IspybRestShippingHandler.completeShipment_onFailure,
			});
		},
		completeShipment_onSuccess:function(){
			ui.logToDialog("IceBear updated successfully.","success");
			ui.logToDialog("The page will reload in a moment. Please wait...");
			window.setTimeout(ui.forceReload, 2500);
		},
		completeShipment_onFailure:function(transport){
			ui.logToDialog("The shipment submission to "+window.shipmentDestination.name+" succeeded, but IceBear was not updated ","error");
			if(transport.responseJSON && transport.responseJSON.error){				
				ui.logToDialog(transport.responseJSON.error, "error");
			}			
			ui.logToDialog("The page will reload in a moment. Please wait...");
			window.setTimeout(ui.forceReload, 5000);
		},

	DatasetRetrieval:{

		datasets:[],

		begin:function (){
			IspybRestShippingHandler.init(IspybRestShippingHandler.DatasetRetrieval.openRetrievalDialog);
		},

		openRetrievalDialog:function (){
			ui.modalBox({ title:"Getting datasets from "+window.shipmentDestination.name, content:"Getting datasets from "+window.shipmentDestination.name+"..." });
			IspybRestShippingHandler.DatasetRetrieval.getDatasets();
		},

		getDatasets:function (){
			Shipment.DatasetRetrieval.datasets=[];
			Shipment.DatasetRetrieval.beamlineNameToId={};
			ui.setModalBoxTitle("Shipment datasets");
			if(!IspybRestShippingHandler.baseUri){
				alert("No base URI supplied. Call begin() first, with base URI.");
				return false;
			}
			if(!IspybRestShippingHandler.token){
				IspybRestShippingHandler.showLoginForm(IspybRestShippingHandler.DatasetRetrieval.getDatasets);
				return false;
			}
			let mb=document.getElementById("modalBox").querySelector(".boxbody");
			if(!mb){ alert('Call begin() not getDatasets()'); return false; }
			mb.innerHTML="Getting datasets from "+window.shipmentDestination.name+"...";

			let proteinAcronyms=[];
			let crystalRows=document.querySelectorAll(".containertab+.tabbody tr.datarow");
			crystalRows.forEach(function (tr){
				let proteinAcronym=tr["rowData"]["proteinacronym"];
				if(proteinAcronym && -1===proteinAcronyms.indexOf(proteinAcronym)){
					proteinAcronyms.push(tr["rowData"]["proteinacronym"]);
				}
			});
			Shipment.DatasetRetrieval.renderProgressNumbers();
			IspybRestShippingHandler.DatasetRetrieval._doGetDatasets(proteinAcronyms);
		},

		_doGetDatasets:function (proteinAcronyms){
			let acronym=proteinAcronyms.pop();
			if(!acronym){
				//We already tried getting datasets for all protein acronyms.
				IspybRestShippingHandler.DatasetRetrieval.processFoundDatasets();
				return false;
			}
			ui.logToDialog("Checking for datasets with protein acronym "+acronym);
			AjaxUtils.remoteAjax(
				IspybRestShippingHandler.baseUri+IspybRestShippingHandler.token+"/proposal/"+
				data.proposalname+"/mx/datacollection/protein_acronym/"+acronym+"/list",
				'get',
				{},
				function (transport){
					let foundDatasets=transport.responseJSON;
					if(!foundDatasets || 0===foundDatasets.length){
						ui.logToDialog("No datasets found with acronym "+acronym, "error");
					} else {
						let count=0;
						foundDatasets.forEach(function (ds){
							if(ds["BLSample_blSampleId"] && 1*ds["Shipping_shippingId"]===1*data["idatremotefacility"]){
								Shipment.DatasetRetrieval.datasets.push(ds);
								Shipment.DatasetRetrieval.setFoundCount();
								count++;
							}
						});
						if(0===count){
							ui.logToDialog("No datasets found for this shipment with acronym "+acronym, "error");
						} else {
							ui.logToDialog(""+count+" dataset(s) found for this shipment with acronym "+acronym, "success");
						}
					}
					IspybRestShippingHandler.DatasetRetrieval._doGetDatasets(proteinAcronyms);
				},
				function (transport){
					if(404===transport.status){
						ui.logToDialog("No datasets found with acronym "+acronym, "error");
					} else {
						ui.logToDialog("Could not get datasets for acronym "+acronym+" (HTTP "+transport.status+")", "error");
					}
					IspybRestShippingHandler.DatasetRetrieval._doGetDatasets(proteinAcronyms);
				}
			);
		},

		processFoundDatasets:function (){
			ui.logToDialog("Checked for datasets against all protein acronyms in the shipment.");
			let numDatasets=IspybRestShippingHandler.DatasetRetrieval.datasets.length;
			if(0===numDatasets){
				ui.logToDialog("No datasets found for this shipment.");
			} else {
				ui.logToDialog(numDatasets+" dataset(s) found for this shipment.");
				IspybRestShippingHandler.DatasetRetrieval.gatherBeamlineDetails();
				//foreach, checkDataset(ds)
				IspybRestShippingHandler.DatasetRetrieval.datasets.forEach(function (ds){
					IspybRestShippingHandler.DatasetRetrieval.checkDataset(ds);
				});
			}
		},

		checkDataset:function (ds){
			let sampleId=ds['BLSample_blSampleId'];
			new Ajax.Request("/api/diffractionrequest/shipmentid/"+data.id+"/crystalidatremotefacility/"+sampleId,{
				method:'get',
				onSuccess:function (transport){ IspybRestShippingHandler.DatasetRetrieval.checkDataset_foundDiffractionRequest(transport, ds) },
				onFailure:function (transport){ IspybRestShippingHandler.DatasetRetrieval.checkDataset_noDiffractionRequest(transport, ds) },
			});
		},

		checkDataset_noDiffractionRequest:function (transport, ds) {
			Shipment.DatasetRetrieval.incrementFailedCount("No IceBear crystal in shipment with "+window.shipmentDestination.name+" ID "+ds['BLSample_blSampleId']);
		},

		checkDataset_foundDiffractionRequest:function (transport, ds) {
			if(1!==transport.responseJSON.rows.length){
				Shipment.DatasetRetrieval.incrementFailedCount("More than one IceBear crystal in shipment with "+window.shipmentDestination.name+" ID "+ds['BLSample_blSampleId']);
				return false;
			}
			let diffractionRequest=transport.responseJSON.rows[0];
			let remoteDatasetId=ds["DataCollection_dataCollectionId"];
			ui.logToDialog("Exactly one IceBear crystal in shipment with "+window.shipmentDestination.name+" ID "+ds['BLSample_blSampleId']);
			//Get diffractionrequestid from lookup table made above
			//Check whether dataset already exists
			new Ajax.Request('/api/dataset/diffractionrequestid/'+diffractionRequest["id"]+'/remotedatasetid/'+remoteDatasetId,{
				method:'get',
				onSuccess:function (transport){
					//create and attach dataset
					if(1!==transport.responseJSON.rows.length){
						//shouldn't be more than one!
						Shipment.DatasetRetrieval.incrementFailedCount("Error on checking for existing local dataset, multiple returned");
					}
					let localDatasetId=transport.responseJSON.rows[0]["id"];
					IspybRestShippingHandler.DatasetRetrieval.attachOrUpdateDataset(ds, diffractionRequest, localDatasetId);
				}, onFailure:function (transport) {
					//if 404, create, otherwise it's an error
					if(404!==transport.status){
						Shipment.DatasetRetrieval.incrementFailedCount("Error on checking for existing local dataset, HTTP "+transport.status);
						return;
					}
					ui.logToDialog("Found no local dataset,creating...");
					IspybRestShippingHandler.DatasetRetrieval.attachOrUpdateDataset(ds, diffractionRequest, null);
				}
			});

		},

		attachOrUpdateDataset(ds, diffractionRequest, localDatasetId){
			let beamlineName=ds["Container_beamlineLocation"];
			if(undefined===beamlineName){
				Shipment.DatasetRetrieval.incrementFailedCount("No beamline name for this dataset - undefined");
				return false;
			}
			if(undefined===Shipment.DatasetRetrieval.beamlineNameToId[beamlineName]){
				//Need the beamline ID
				ui.logToDialog("Waiting for beamline ID");
				window.setTimeout(function (){
					IspybRestShippingHandler.DatasetRetrieval.attachOrUpdateDataset(ds, diffractionRequest,localDatasetId);
				},500);
				return false;
			}
			let beamlineId=Shipment.DatasetRetrieval.beamlineNameToId[beamlineName];
			let method="post";
			let uri="/api/dataset";
			if(localDatasetId){
				method="patch";
				uri+=localDatasetId;
			}
			new Ajax.Request(uri,{
				method:method,
				parameters:{
					"csrfToken":csrfToken,
					"diffractionrequestid":diffractionRequest["id"],
					"crystalid":diffractionRequest["crystalid"],
					"beamlineid":beamlineId,
					"remotedatasetid":ds["DataCollection_dataCollectionId"],
					"remotedatasetobject":JSON.stringify(ds),
					"datalocation":window.shipmentDestination.name+": "+ds["DataCollection_imageDirectory"],
					"description":"Retrieved from "+window.shipmentDestination.name+", run "+ds["DataCollection_dataCollectionNumber"],
					"detectormanufacturer":ds["Detector_detectorManufacturer"],
					"detectormodel":ds["Detector_detectorModel"],
					"detectortype":ds["Detector_detectorType"],
					"detectormode": ds["Detector_detectorMode"]
				},
				onSuccess:function (){
					Shipment.DatasetRetrieval.incrementSucceededCount("Created/updated IceBear dataset OK");
				},
				onFailure:function (){
					Shipment.DatasetRetrieval.incrementFailedCount("Failed creating/updating IceBear dataset");
				}
			});
		},

		gatherBeamlineDetails:function () {
			let beamlines = [];
			let namesDone = [];
			Shipment.DatasetRetrieval.datasets.forEach(function (ds) {
				let beamlineName = ds["Container_beamlineLocation"];
				if (undefined !== beamlineName && -1 === namesDone.indexOf(beamlineName)) {
					beamlines.push({
						'name': beamlineName,
						'detectormanufacturer': ds["Detector_detectorManufacturer"],
						'detectormodel': ds["Detector_detectorModel"],
						'detectortype': ds["Detector_detectorType"],
						'shipmentdestinationid': data['shipmentdestinationid']
					});
					namesDone.push(beamlineName);
				}
			});
			Shipment.DatasetRetrieval.updateIceBearBeamlines(beamlines);
		},

	} //end DatasetRetrieval

}; //end IspybRestShippingHandler