
let Crystal={
		
		diffractionTypes:[
			{ label:'OSC', value:'OSC' },
			{ label:'SAD', value:'SAD' },
			{ label:'MAD', value:'MAD' },
		],
		
		/**
		 * Crystal space groups, by category.
		 * Format: 
		 * Underscores are purely to indicate digit grouping boundaries. Some numbers are written as subscripts. Where 
		 * two digits are together, the second should be rendered as a subscript, e.g. P_42_2_2 rendered to the user as
		 * P4<sub>2</sub>22 and saved to the database as P4222.
		 */
		spaceGroups:[
			{
				'category':'Triclinic', 
				'groups':['Unknown','P_1']
			},
			{
				'category':'Monoclinic',
				'groups':[
					'P_2','P_21','C_2'
				]
			},
			{
				'category':'Orthorhombic',
				'groups':[
					'P_2_2_2','P_2_2_21','P_21_21_2','P_21_21_21','C_2_2_21','C_2_2_2','F_2_2_2','I_2_2_2','I_21_21_21'
				]
			},
			{
				'category':'Tetragonal',
				'groups':[
					'P_4','P_41','P_42','P_43','I_4','I_41','P_4_2_2','P_4_21_2','P_41_2_2',
					'P_41_21_2','P_42_2_2','P_42_21_2','P_43_2_2','P_43_21_2','I_4_2_2','I_41_2_2'
				]
			},
			{
				'category':'Trigonal',
				'groups':[
					'P_3','P_31','P_32','R_3','H_3','P_3_1_2','P_3_2_1','P_31_1_2','P_31_2_1',
					'P_32_1_2','P_32_2_1','R_3_2','H_3_2'
				]
			},
			{
				'category':'Hexagonal',
				'groups':[
					'P_6','P_61','P_65','P_62','P_64','P_63','P_6_2_2',
					'P_61_2_2','P_65_2_2','P_62_2_2','P_64_2_2','P_63_2_2'
				]
			},
			{
				'category':'Cubic',
				'groups':[
					'P_2_3','F_2_3','I_2_3','P_21_3','I_21_3','P_4_3_2','P_42_3_2','F_4_3_2',
					'F_41_3_2','I_4_3_2','P_43_3_2','P_41_3_2','I_41_3_2'
			 	]
			}
		],
		
		getThumbnailLink: function(crystal){
			return '<a href="'+Crystal.getDropViewerUrl(crystal)+'"><img src="/dropimagethumb/'+crystal.dropimageid+'/" style="max-height:5em;margin:0.25em 0"  alt=""/></a>';
		},

		getTextLink: function(crystal){
			let dropName=Plate.getWellNameFromObject(crystal);
			dropName=dropName.replace("."," drop ");
			return '<a href="/crystal/'+crystal.id+'">'+dropName+' crystal '+crystal.numberindrop+'</a>';
		},
		getDropViewerUrl: function(crystal){
			let dropName=Plate.getWellNameFromObject(crystal);
			return '/imagingsession/'+crystal["imagingsessionid"]+'/#'+dropName+'c'+crystal.numberindrop;
		},

		spaceGroupField:function(crystal,parentElement){
			let val=crystal.spacegroup;
			let displayValue="Unknown";
			if(""!==val){
				Crystal.spaceGroups.forEach(function(cat){
					cat.groups.forEach(function(grp){
						if(val===grp.replace(/_/g,"")){
							displayValue=grp.replace(/(\d)(\d)/g,"$1<sub>$2</sub>").replace(/_/g,'');
						}
					});
				});
			}
			let content='<input data-apiurl="/api/crystal/'+crystal.id+'" type="hidden" name="spacegroup" value="'+val+'" /><span class="sgvalue">'+displayValue+'</span>';
			content+=' <button onclick="Crystal.showSpaceGroupPicker(this);return false">Change...</button>';
			let ff=ui.formField({
				'label':'Space group',
				'content':content
			}, parentElement);
			ff.down("input").oldvalue=val;
			
		},
		
		showSpaceGroupPicker:function(btn){
			let box;
			let mb=$("modalBox");
			if(mb){
				mb.oldTitle=mb.down("h2").innerHTML;
				mb.down("h2").innerHTML="Choose space group";
				let modalBody=mb.down(".boxbody");
				box=modalBody.clone(false);
				box.id="sgoverlay";
				mb.down("h2").insert({ after:box });
				modalBody.style.display="none";
				box.style.display="block";
			} else {
				box=ui.modalBox({ title:"Choose a space group", content:"" });
			}
			let lbl=btn.up("label");
			lbl.down("input").oldvalue=lbl.down("input").value;
			box.field=lbl;
			Crystal.spaceGroups.each(function(cat){
				let buttons='<h4>'+cat.category+'</h4>';
				cat.groups.forEach(function(grp){
					let btnVal=grp.replace(/_/g,"");
					let btnLbl=grp.replace(/(\d)(\d)/g,"$1<sub>$2</sub>").replace(/_/g,'');
					buttons+='<button onclick="Crystal.setSpaceGroupField(this)" style="line-height:2em; margin:0.25em" value="'+btnVal+'">'+btnLbl+'</button>';
				});
				box.innerHTML+=buttons;
			});
			
		},
		
		setSpaceGroupField:function(btn){
			let mb=$("modalBox");
			let overlay=$("sgoverlay");
			let field=btn.up(".boxbody").field;
			field.down(".sgvalue").innerHTML=btn.innerHTML;
			let val=btn.value;
			if("Unknown"===val){ val=""; }
			field.down("input").value=val;
			ui.updateFormField(field.down("input"));
			if(overlay){
				let box=overlay.up(".box");
				overlay.remove();
				box.down(".boxbody").style.display="block";
				if(mb.oldTitle){ mb.down("h2").innerHTML=mb.oldTitle; }
			} else {
				ui.closeModalBox();
			}
		},
		
		writeDetailsFormFields:function(xtal,canUpdate, parentElement){
			let xtalUrl='/api/crystal/'+xtal.id;
			let diffreq=null;
			xtal.diffractionrequests.each(function(dr){
				if(""===dr.shipmentid){
					diffreq=dr;
				}
			});

			
			parentElement.innerHTML=parentElement.innerHTML+"";

			Crystal.spaceGroupField(xtal, parentElement);
			
			//Unit cell
			let tbl='<table style="width:100%">';
			tbl+='<tr><td id="uca">a: </td><td id="ucb">b: </td><td id="ucc">c: </td></tr>';
			tbl+='<tr><td id="ucalpha">&alpha;: </td><td id="ucbeta">&beta;: </td><td id="ucgamma">&gamma;: </td></tr>';
			tbl+='</table>';
			ui.formField({label:'Unit cell dimensions', content:tbl }, parentElement);

			parentElement.innerHTML+='<h4>Information for next synchrotron trip</h4>';

		
			//Sample name
			ui.formField({
				label:'Sample name for synchrotron',
				content:'<span id="snamectrl" style="white-space:nowrap">'+xtal.prefix+ (""===xtal["suffix"] ? "" : "_"+xtal["suffix"]) +'</span>'
			}, parentElement);
			if(canUpdate){
				let sampleNameControl=$("snamectrl");
				sampleNameControl.innerHTML=xtal.prefix+'<span id="nameseparator">_</span>';
				ui.textBox({ apiUrl:xtalUrl, name:'suffix', value:xtal["suffix"] }, sampleNameControl );
				$("suffix").observe("keyup",function(){
					if(''===$("suffix").value){
						$("nameseparator").style.visibility="hidden";
					} else {
						$("nameseparator").style.visibility="visible";
					}
				});
			}

			if(!diffreq){ return; }
			let drUrl='/api/diffractionrequest/'+diffreq.id;

			//Shipping comment
			ui.formField({
				label:'Shipping comment',
				content:'<span id="commctrl">'+ (""===diffreq["comment"] ? "-" : diffreq["comment"]) +'</span>'
			}, parentElement);
			if(canUpdate){
				let ctrl=$("commctrl");
				ctrl.innerHTML='';
				ui.textBox({ apiUrl:drUrl, name:'comment', value:diffreq["comment"] }, ctrl );
				ctrl.down("input").style.width="95%";
			}

			//Diffraction type, resolutions
			ui.formField({ label:'Diffraction type', content:'<span id="dtype"></span>'}, parentElement);
			ui.formField({ label:'Observed resolution (&#8491;)', content:'<span id="obsres"></span>'}, parentElement);
			ui.formField({ label:'Required resolution: (&#8491;)', content:'<span id="reqres"></span>'}, parentElement);
			ui.formField({ label:'Minimum resolution: (&#8491;)', content:'<span id="minres"></span>'}, parentElement);
			ui.dropdownElement({ apiUrl:drUrl, name:'diffractiontype', value:diffreq["diffractiontype"], options:Crystal.diffractionTypes }, $("dtype") );

			let obs=$("obsres");
			let req=$("reqres");
			let min=$("minres");
			ui.textBox({ apiUrl:drUrl, name:'observedresolution', value:diffreq["observedresolution"] }, obs );
			ui.textBox({ apiUrl:drUrl, name:'requiredresolution', value:diffreq["requiredresolution"] }, req );
			ui.textBox({ apiUrl:drUrl, name:'minimumresolution', value:diffreq["minimumresolution"] }, min );
			obs.down("input").style.width="4em";
			req.down("input").style.width="4em";
			min.down("input").style.width="4em";
			
			ui.textBox({ apiUrl:xtalUrl, name:'unitcella',     value:xtal.unitcella },     $("uca")     );
			ui.textBox({ apiUrl:xtalUrl, name:'unitcellb',     value:xtal.unitcellb },     $("ucb")     );
			ui.textBox({ apiUrl:xtalUrl, name:'unitcellc',     value:xtal.unitcellc },     $("ucc")     );
			ui.textBox({ apiUrl:xtalUrl, name:'unitcellalpha', value:xtal.unitcellalpha }, $("ucalpha") );
			ui.textBox({ apiUrl:xtalUrl, name:'unitcellbeta',  value:xtal.unitcellbeta },  $("ucbeta")  );
			ui.textBox({ apiUrl:xtalUrl, name:'unitcellgamma', value:xtal.unitcellgamma }, $("ucgamma") );
		},

		/**
		 * Deletes the crystal on the server. 
		 * @param {object} crystal The crystal to be deleted
		 * @param {function|null} afterSuccess A function to be called if deletion succeeds. The response JSON will be passed as a parameter.
		 * @param {function|null} afterFailure A function to be called if deletion fails.
		 */
		delete: function(crystal, afterSuccess, afterFailure){
			if(1*crystal.isFished){
				alert("Crystal cannot be deleted because it has been fished.");
				if(afterFailure){ return afterFailure(); }
				return false;
			}
			new Ajax.Request('/api/crystal/'+crystal.id, {
				method:"delete",
				postBody:"csrfToken="+csrfToken,
				onSuccess:function(transport){ 
					if(afterSuccess){
						afterSuccess(transport.responseJSON);
					}
				},
				onFailure:function(transport){ 
					AjaxUtils.checkResponse(transport);
					if(afterFailure){
						afterFailure(transport);
					}
				},
			});
		},

		renderProteinTab:function(crystal, tabSet) {
			let proteinTab = tabSet.tab({"label": "Protein", "id": "protein"}).next();
			Crystal.refreshProteinTab(crystal, proteinTab);
		},

		refreshProteinTab:function(crystal, proteinTab){
			proteinTab.innerHTML="";
			if(!crystal["constructname"] || ""===crystal["constructname"]){
				proteinTab.innerHTML='Protein and construct not set';
			} else {
				let f=proteinTab.form({});
				f.formField({ readOnly:true, label:"Project", content:'<a href="/project/'+crystal["projectid"]+'">'+crystal["projectname"]+'</a>' });
				f.formField({ readOnly:true, label:"Protein", content:crystal["proteinname"]+' (acronym: '+crystal["proteinacronym"]+')' });
				f.formField({ readOnly:true, label:"Construct", content:crystal["constructname"] });
				Crystal.getPdbDepositionsForProteinTab(crystal, proteinTab);
			}
			return proteinTab;
		},

		getPdbDepositionsForProteinTab:function(crystal, proteinTab){
			new Ajax.Request('/api/crystal/'+crystal.id+'/pdbdeposition',{
				method:'get',
				onSuccess:function (transport) {
					data["pdbdepositions"]=transport.responseJSON.rows;
					let frm=proteinTab.querySelector("form");
					Crystal.renderAssociatedPdbCodes(transport, frm);
				},
				onFailure:function (transport) {
					if(404===transport.status){
						data["pdbdepositions"]=[];
						let frm=proteinTab.querySelector("form");
						return Crystal.renderAssociatedPdbCodes(transport, frm);
					}
					AjaxUtils.checkResponse(transport);
				},

			});
		},

		renderAssociatedPdbCodes:function(transport, frm){
			frm.innerHTML+='<label><h3>Deposited structures</h3></label>';
			let content='&nbsp;';
			let wrote=false;
			if(canManagePdbDepositions){
				content='<img src="/images/icons/btn_no.gif" alt="Dissociate" title="Unlink this structure from the crystal" onclick="Crystal.dissociatePdbCode(this)"/>';
			}
			if(transport.responseJSON && transport.responseJSON.rows){
				transport.responseJSON.rows.forEach(function (row) {
					let f=frm.formField({
						label:'<a href="/pdbdeposition/'+row.id+'">'+row.name+'</a>',
						content:content
					});
					f.dataset.pdbdepositioncrystalid=row.id;
					f.dataset.pdbcode=row.name;
					wrote=true;
				});
			}
			if(canManagePdbDepositions){
				let f=frm.formField({
					label:'',
					content:'Link this crystal to a deposited structure: <input style="width:8em" type="text" name="pdbcode" placeholder="PDB code"/> <input type="button" value="Link"/>'
				});
				fieldValidations["pdbcode"]=["required","pdbcode"];
				f.querySelector('[name=pdbcode]').addEventListener("keydown", Crystal.handlePdbCodeFieldsEvent);
				f.querySelector('[type=button]').addEventListener("click", Crystal.handlePdbCodeFieldsEvent);
				wrote=true;
			}

			if(!wrote){
				frm.innerHTML+='<label><span class="label">(none)</span>&nbsp;</label>';
			}
		},

		handlePdbCodeFieldsEvent:function(evt){
			if('Enter'===evt.key || 'click'===evt.type){
				evt.preventDefault();
				let field=evt.target.closest('label').querySelector('[name=pdbcode]');
				if(validator.validate(field)){
					let pdbCode=field.value;
					new Ajax.Request('/api/pdbdeposition/name/'+pdbCode,{
						method:'get',
						onSuccess:function (transport) {
							Crystal.associatePdbCodeWithCrystal(transport.responseJSON.rows[0].id, data.id, field);
						},
						onFailure:function(transport){
							if(404===transport.status){
								return Crystal.createAndAssociatePdbCode(pdbCode, data.id, field);
							}
							AjaxUtils.checkResponse(transport);
						}
					});
				}
			}
			return true;
		},

		associatePdbCodeWithCrystal:function(pdbDepositionId, crystalId, field){
			let lbl;
			if(field){
				lbl=field.closest("label");
				lbl.classList.add("updating");
			}
			new Ajax.Request('/api/pdbdepositioncrystal/', {
				method:'post',
				parameters:{
					csrfToken:csrfToken,
					pdbdepositionid:pdbDepositionId,
					projectid:data.projectid,
					crystalid:crystalId
				},
				onSuccess:function () {
					Crystal.refreshProteinTab(data,lbl.up(".tabbody"));
				},
				onFailure:function (transport) {
					if(lbl){ lbl.classList.remove("updating"); }
					if(400===transport.status && transport.responseText.indexOf("may already exist")!==false){
						alert("That PDB entry is already associated with this crystal");
					} else {
						AjaxUtils.checkResponse(transport);
					}
				}
			});
		},

		dissociatePdbCode:function(elem){
			let lbl=elem.closest("label");
			if(!confirm('Really unlink this crystal from PDB code '+lbl.dataset.pdbcode+'?\n\n(This won\'t delete IceBear\'s record of the PDB deposition.)')){
				return false;
			}
			lbl.classList.add("updating");
			new Ajax.Request("/api/pdbdepositioncrystal/"+lbl.dataset.pdbdepositioncrystalid,{
				method:"delete",
				postBody:"csrfToken="+csrfToken,
				onSuccess:function(){
					Crystal.refreshProteinTab(data,lbl.up(".tabbody"));
				},
				onFailure:function(transport){
					AjaxUtils.checkResponse(transport);
					lbl.classList.remove("updating");
				},
			})
		},

		createAndAssociatePdbCode:function(pdbCode, crystalId, field){
			if(field){
				if(pdbCode!==field.value){
					alert("PDB code does not match field value. This is a bug in IceBear.");
					return false;
				}
				field.closest("label").classList.add("updating");
			}
			new Ajax.Request('/api/pdbdeposition',{
				method:'post',
				parameters:{
					csrfToken:csrfToken,
					projectid:data.projectid,
					name:pdbCode
				},
				onSuccess:function (transport) {
					let pdbId=transport.responseJSON['created']["id"];
					return Crystal.associatePdbCodeWithCrystal(pdbId, crystalId, field);
				},
				onFailure:AjaxUtils.checkResponse
			})
		},

		renderDataCollectionsTab:function(crystal, tabSet){
			let dcTab=tabSet.tab({ "label":"Datasets", "id":"datacollections" }).nextElementSibling;
			new Ajax.Request('/api/crystal/'+crystal.id+'/diffractionrequest?sortby=id&sortdescending=yes', {
				method:'get',
				onFailure:function(){
					dcTab.innerHTML='No shipments containing this crystal';
				},
				onSuccess:function(transport){
					if(1*transport.responseJSON.total===1 && !(1*transport.responseJSON.rows[0].shipmentid)){
						//We got the only the current "pending" diffraction request id, which has not been shipped
						//Render a "No shipments" box, then render a box for no-shipment datasets
						let lbl=document.createElement("label");
						lbl.innerHTML='No shipments containing this crystal';
						lbl.style.textAlign="left";
						dcTab.appendChild(lbl);
						lbl=document.createElement("label");
						lbl.innerHTML+='<div class="shipment_datasets"><hr/>Getting datasets...</div>';
						lbl.style.textAlign="left";
						dcTab.appendChild(lbl);
						window.setTimeout(function(){ Crystal.getDatasetsForLabelElement(lbl) },50);
						return false;
					}
					transport.responseJSON.rows.each(function(dr){
						if(!(1*dr.shipmentid)){ return; }

						let lbl=document.createElement("label");
						lbl.id="shipment"+dr.shipmentid;
						lbl.dataset.shipmentId=dr.shipmentid;
						lbl.dataset.diffractionRequestId=dr.id;
						lbl.innerHTML="Getting shipment details...";
						lbl.style.textAlign="left";
						dcTab.appendChild(lbl);

						//Get the shipment for this diffraction request
						new Ajax.Request('/api/shipment/'+dr.shipmentid,{
							method:'get',
							onSuccess:function(transport){
								lbl.shipment=transport.responseJSON;
								//Get the shipment destination for this shipment
								lbl.dataset.shipmentDestinationId=lbl.shipment["shipmentdestinationid"];
								new Ajax.Request('/api/shipmentdestination/'+transport.responseJSON["shipmentdestinationid"],{
									method:'get',
									onSuccess:function(transport){
										lbl.shipmentDestination=transport.responseJSON;
										let out='<strong>Shipment:</strong> <a href="/shipment/'+lbl.shipment.id+'">'+lbl.shipment.name+'</a>';
										out+=' to <a href="/shipmentdestination/'+lbl.shipmentDestination.id+'">'+lbl.shipmentDestination.name+'</a>';
										out+=', shipped '+ui.friendlyDate(lbl.shipment.dateshipped);
										if(""!==lbl.shipment.datereturned){
											out+=', returned '+ui.friendlyDate(lbl.shipment.datereturned);
										}
										if(""!==lbl.shipment.proposalname && ""!==lbl.shipment.sessionname) {
											out+='<br/>Proposal: ' + lbl.shipment.proposalname + ', session: ' + lbl.shipment.sessionname;
										} else if(""!==lbl.shipment.proposalname) {
											out+='<br/>Proposal: ' + lbl.shipment.proposalname;
										} else if(""!==lbl.shipment.sessionname){
											out+='<br/>Session: '+lbl.shipment.sessionname;
										}
										out+=' &bull; ';
										out+='<a target="_blank" href="'+lbl.shipment.urlatremotefacility+'">View shipment at '+lbl.shipmentDestination.name+'</a>';
										if(""!==dr.crystalurlatremotefacility){
											out+=' &bull; ';
											out+='<a target="_blank" href="'+dr.crystalurlatremotefacility+'">View crystal at '+lbl.shipmentDestination.name+'</a>';
										}
										//Get the datasets for this shipment
										out+='<div class="shipment_datasets"><hr/>Getting datasets...</div>';
										lbl.innerHTML=out;
										window.setTimeout(function(){ Crystal.getDatasetsForLabelElement(lbl) },50);
									},
									onFailure:function(){
										//Couldn't retrieve the shipment destination. Something went badly wrong.
										lbl.innerHTML='Could not get shipment details';
									},
								});
							},
							onFailure:function(){
								lbl.innerHTML='Could not get shipment details';
							},
						});
					});
					let lbl=document.createElement("label");
					lbl.innerHTML+='<div class="shipment_datasets"><hr/>Getting datasets...</div>';
					lbl.style.textAlign="left";
					dcTab.appendChild(lbl);
					window.setTimeout(function(){ Crystal.getDatasetsForLabelElement(lbl) },50);
				}
			});
			return dcTab;
		},

		getDatasets:function() {
			let dcBody=document.getElementById("datacollections_body");
			if(!dcBody){ return true; } //If false or void, dataset view will not close
			let shipments=dcBody.querySelectorAll("label");
			shipments.forEach(function (lbl){
				Crystal.getDatasetsForLabelElement(lbl);
			});
			return true; //needed because it's used as a callback on dataset view close
		},

		getDatasetsForLabelElement:function (lbl){
			let crystalId=data.id;
			let diffractionRequestId=nullValue; //Database null string, set in header.
			if(lbl.dataset.diffractionRequestId){
				diffractionRequestId=lbl.dataset.diffractionRequestId;
			}
			new Ajax.Request('/api/dataset/crystalid/'+crystalId+'/diffractionrequestid/'+diffractionRequestId, {
				method:'get',
				onFailure:function (transport){
					let div=lbl.querySelector(".shipment_datasets");
					if(404===transport.status){
						lbl.datasets=[];
						Crystal.renderDatasets(lbl);
					} else {
						div.innerHTML="Could not get datasets";
					}
				},
				onSuccess:function (transport){
					lbl.datasets=transport.responseJSON["rows"];
					Crystal.renderDatasets(lbl);
				}
			});
			return true; //needed because it's used as a callback on dataset view close
		},

		renderDatasets:function (lbl){
			let headerText="Datasets from this crystal, with no associated shipment";
			if(lbl.dataset.diffractionRequestId){
				headerText="Datasets associated with this crystal and shipment";
			}
			let datasets=lbl.datasets;
			let cellTemplates=[ [Dataset.getLinkAndDescriptionHTML, 'datalocation'] ];
			let headers=[headerText];
			let control='<input type="button" value="View..." onclick="Dataset.beginEdit(this)" />';
			if(canEdit){
				cellTemplates=[ [Dataset.getLinkAndDescriptionHTML, 'datalocation'],'{{control}}'];
				headers.push('<input type="button" style="margin-left:0.5em" value="Add dataset..." onclick="Dataset.beginCreate(this)" />');
				control='<input type="button" style="margin-left:0.5em" value="Delete" onclick="Dataset.delete(this)" />'+control;
			}
			datasets.forEach(function(ds){
				ds['control']=control;
			});
			if(!datasets || !datasets.length){
				datasets=[ {datalocation:"None", control:""} ];
			}
			ui.table({
					headers:headers,
					cellTemplates: cellTemplates
				},
				datasets,
				lbl.querySelector("div")
			);
		},

};

let Dataset={

	/**
	 * Generates a tab with a list of datasets, filtered by the supplied property key and value.
	 * For example, for datasets pertaining to project 123.
	 *
	 * @param tabSet The parent tab set element
	 * @param propertyKey The property to filter on, e.g., "projectid"
	 * @param propertyValue The required value of the property, e.g., 123
	 */
	listTab:function (tabSet, propertyKey, propertyValue){
		let control='<input type="button" value="View..." onclick="Dataset.beginEdit(this)" />';
		let headers=[];
		let cellTemplates=[];
		if("pdbdeposition"!==data["objecttype"]){
			headers=['PDB code'];
			cellTemplates=['<a href="/pdbdeposition/{{pdbdepositionid}}">{{pdbcode}}</a>'];
		}
		headers.push('Crystal','Data location','Description','');
			cellTemplates.push(
			'<a href="/crystal/{{crystalid}}">{{crystalname}}</a>',
			[Dataset.makeDoiAndUrlClickable,'datalocation'],
			'{{description}}',
			control
		);
		tabSet.tab({
			'label':'Datasets',
			'id':'datasets',
			'url':'/api/dataset/'+propertyKey+'/'+propertyValue+'/',
			headers:headers,
			cellTemplates:cellTemplates,
			contentBefore: 'Add datasets by associating them with the relevant crystal'
		});

	},

	beginCreate:function(addButton){
		let mb=ui.modalBox({
			'title':'Add a dataset',
			'content':''
		});
		let diffractionRequestId=addButton.closest("label").dataset.diffractionRequestId || nullValue;
		let shipmentDestinationId=addButton.closest("label").dataset.shipmentDestinationId || nullValue;
		Dataset.writeForm(mb,diffractionRequestId,shipmentDestinationId);
	},

	beginEdit:function(editButton){
		let ds=editButton.closest("tr").rowData;

		// If this isn't a crystal page it's read-only and there is no master list of PDB entries for the read-only
		// SELECT field to choose from. We need to bodge the single PDB entry associated with the record to be viewed
		// into data.pdbdepositions if there is one. This needs to work for multiple views of multiple datasets with
		// different PDB codes.
		if(parseInt(ds.pdbdepositionid) && "crystal"!==data["objecttype"]){
			if (!data.pdbdepositions) {
				data.pdbdepositions = [];
			}
			let hasPdbAlready=false;
			data.pdbdepositions.forEach(function(pdb){
				if(pdb["name"]===ds["pdbcode"]){ hasPdbAlready=true; }
			});
			if(!hasPdbAlready){
				data.pdbdepositions.push({ "id":ds['pdbdepositionid'], "name":ds["pdbcode"]} );
			}
		}

		let ts=ui.modalTabSet({
			onclose:Crystal.getDatasets
		});
		let detailsTab=ts.tab({label:"Dataset"}).nextElementSibling;
		let lbl=editButton.closest("label");
		let shipmentDestinationId=nullValue;
		if(lbl){
			shipmentDestinationId=lbl.dataset.shipmentDestinationId || nullValue;
		}
		Dataset.writeForm(detailsTab, ds["diffractionrequestid"], shipmentDestinationId, ds);
		ts.notesTab(ds.id);
	},



	/**
	 * Deletes the dataset, from a button on the crystal page.
	 */
	delete:function (deleteButton){
		if(!confirm("Really delete the dataset?")){ return false; }
		let tr=deleteButton.closest("tr");
		tr.classList.add("updating");
		new Ajax.Request("/api/dataset/"+tr.rowData["id"],{
			method:"delete",
			parameters:{
				csrfToken:csrfToken
			},
			onFailure:AjaxUtils.checkResponse,
			onSuccess:Crystal.getDatasets()
		});
	},

	/**
	 * Changes URLs and DOIs in the supplied dataset description into clickable links.
	 * @param datasetOrText An Object representing a dataset, or a string representing its location
	 * @returns string The dataset description
	 */
	makeDoiAndUrlClickable:function (datasetOrText) {
		let ds=datasetOrText;
		if(datasetOrText["datalocation"]){
			ds=datasetOrText["datalocation"];
		}
 		if(ds.indexOf("<input")!==0){
			ds=ds.replace(new RegExp("(http[s]?:\/\/doi.org\/)","gi"), '');
 			ds=ds.replace(new RegExp("(http[s]?:\/\/[^\\s]+)","gi"), '<a target="_blank" href="$1">$1</a>'); //FIRST make URLs clickable
			ds=ds.replace(new RegExp("(?:doi:)?(10\.[0-9]{4,9}/[^\s]+)","gi"), '<a target="_blank" href="https://www.doi.org/$1">$1</a>');
		}
		return ds;
	},

	/**
	 * Generates the HTML for the basic dataset view on the crystal page, with a link and a brief description.
	 * @param obj Object The dataset
	 * @returns The HTML for the table cell
	 */
	getLinkAndDescriptionHTML:function (obj){
		//NB Check for existence of all properties of obj used in here. They may not be present in the default "None" object.
		let ds=Dataset.makeDoiAndUrlClickable(obj); //Uses datalocation
		if(obj["description"] && ""!==obj["description"]){
			ds+="<br/>"+obj["description"];
		}
		return ds;
	},

	writeForm:function(parentElement, diffractionRequestId, shipmentDestinationId, datasetObject) {
		parentElement.innerHTML="Just a moment...";
		let url="/api/beamline/";
		if(nullValue!==shipmentDestinationId){
			url+='shipmentdestinationid/'+shipmentDestinationId+'/';
		}
		url+="?pagenum=1&pagesize=10000";
		new Ajax.Request(url, {
			method: "get",
			onSuccess: function (transport) {
				Dataset._doWriteForm(parentElement, diffractionRequestId, datasetObject, transport.responseJSON);
			},
			onFailure:function (transport){
				if(404===transport.status){
					Dataset._doWriteForm(parentElement, diffractionRequestId, datasetObject, []);
					return;
				}
				AjaxUtils.checkResponse(transport);
			}
		});
	},
	_doWriteForm:function(parentElement, diffractionRequestId, datasetObject, beamlines){
		parentElement.innerHTML="";
		let isCreate=!datasetObject;
		let readOnly=!(canEdit && "crystal"===data["objecttype"]) //Only allow edit on crystal page
		let action = isCreate ? "/api/dataset/" : "/api/dataset/"+datasetObject['id'];
		let method = isCreate ? "post" : "patch";
		let dataLocation= isCreate ? "" : datasetObject['datalocation'];
		let description = isCreate ? "" : datasetObject['description'];
		let beamlineid = isCreate ? "" : datasetObject['beamlineid'];
		let detectormanufacturer = isCreate ? "" : datasetObject['detectormanufacturer'];
		let detectormodel = isCreate ? "" : datasetObject['detectormodel'];
		let detectortype = isCreate ? "" : datasetObject['detectortype'];
		let detectormode = isCreate ? "" : datasetObject['detectormode'];
		let pdbdepositionid = isCreate ? "" : datasetObject['pdbdepositionid'];

		let f=parentElement.form({
			action:action,
			method:method,
			autosubmit: !isCreate,
			readonly:readOnly
		});

		let beamlineOptions=[{ "label":"Choose...","value":nullValue, "id":"", "detectormanufacturer":"", "detectormodel":"", "detectortype":"" }];
		if(beamlines){
			if(beamlines["rows"]){
				beamlines=beamlines["rows"];
			}
			beamlines.forEach(function (bl){
				beamlineOptions.push({
					"label":bl["shipmentdestinationname"]+": "+bl.name,
					"value":bl["id"]
				})
			});
		}

		f.hiddenField("diffractionrequestid",diffractionRequestId);
		f.hiddenField("projectid",data.projectid);
		f.hiddenField("crystalid",data.id); //Only valid on crystal page, but readonly everywhere else so we can get away with this
		let lf=f.textField({
			name:'datalocation',
			label:'Data location',
			helpText: 'A DOI, URL, file path, or other location where the raw data is found',
			value: readOnly ? Dataset.makeDoiAndUrlClickable(dataLocation) : dataLocation,
			validation:{
				required:true
			}
		});
		let df=f.textField({
			name:'description',
			label:'Description',
			helpText:'Brief details of the dataset',
			value:description
		});
		if(!readOnly){
			lf.querySelector("input").style.width="70%";
			df.querySelector("input").style.width="70%";
		}

		//TODO won't work in PDB page. Use the PDB record ("data")
		if(data["pdbdepositions"] && 0!==data["pdbdepositions"].length){
			let pdbs=[{ "label":"(None)", "value":nullValue }];
			data["pdbdepositions"].forEach(function(pdb){
				pdbs.push({ "label":pdb["name"], "value":pdb["id"] });
			});
			f.dropdown({
				name:"pdbdepositionid",
				label:"Associated PDB deposition",
				showOptionLabelOnReadOnly:true,
				value:pdbdepositionid,
				options:pdbs
			})
		} else {
			f.formField({
				label:"Associated PDB deposition",
				content:"(Add a PDB deposition to the crystal, then associate it with the dataset)"
			});
		}

		/*
		 * Beamline dropdown. On choosing a beamline, we update the detector manufacturer, model and type from the
		 * IceBear record.
		 */
		if(1!==beamlineOptions.length){
			let dd=f.dropdown({
				name:"beamlineid",
				label:"Beamline",
				value:beamlineid,
				options:beamlineOptions,
				showOptionLabelOnReadOnly:true
			})
			if(!readOnly){
				ui.addSuppliedHelpText(dd,"Choosing the beamline will automatically update the detector manufacturer, model and type");
				dd=dd.querySelector("select");
				beamlines.forEach(function (b){
					if(b.hasOwnProperty("id")){
						let opt=dd.querySelector('option[value="'+b.id+'"]');
						opt["beamline"]=b;
					}
				});
				dd.onchange=function (evt){
					let sel=evt.target;
					let opt=sel.options[sel.selectedIndex];
					let frm=sel.closest("form");
					let fields=["detectormanufacturer","detectormodel","detectortype"];
					fields.forEach(function (f){
						let formField=frm.querySelector("[name="+f+"]");
						formField.value= (opt.beamline) ? opt.beamline[f] : "";
						window.setTimeout(function() {
							formField.focus();
							formField.blur();
						},100);
					});
				}
			}
		}

		f.textField({
			name:'detectormanufacturer',
			label:'Detector manufacturer',
			value:detectormanufacturer
		});

		f.textField({
			name:'detectormodel',
			label:'Detector model',
			value:detectormodel
		});

		f.textField({
			name:'detectortype',
			label:'Detector type',
			value:detectortype
		});

		f.textField({
			name:'detectormode',
			label:'Detector mode',
			value:detectormode
		});

		if(isCreate){
			f.buttonField({
				label:"Create dataset",
				onclick:Dataset.doCreate
			})
		}

	},

	doCreate:function (evt){
		let btn=evt.target;
		let frm=btn.closest("form");
		let dl=frm.querySelector("[name=datalocation]");
		dl.value=dl.value.trim();
		if(""===dl.value){
			dl.closest("label").classList.add("invalidfield");
			return false;
		} else {
			dl.closest("label").classList.remove("invalidfield");
		}
		let parameters={ "csrfToken":csrfToken };
		frm.querySelectorAll("input,select").forEach(function(field){
			if(field.name){
				parameters[field.name]=encodeURIComponent(field.value);
			}
		});
		btn.closest("label").classList.add("updating");
		new Ajax.Request("/api/dataset/",{
			method:'post',
			parameters:parameters,
			onSuccess:function (){
				Crystal.getDatasets();
				window.setTimeout(ui.closeModalBox,500);
			},
			onFailure:function (transport){
				AjaxUtils.checkResponse(transport);
			}
		});
	}

};

let PlateWidget={
		
		/**
		 * Renders the HTML for a plate widget and attaches it to the parent.
		 * @param details A JSON object containing at least the following:
		 * 					wellDrops An array of welldrop objects for this plate
		 * 					plateType A platetype object describing the plate geometry
		 *                Optional parameters:
		 *                  id The HTML ID of the widget element. Internal element IDs are prepended with this ID.
		 *                  dropPickerPosition:left Float the drop picker to the left, instead of on top.
		 *                  dropRenderer: A function taking the individual drop div, applied to all non-empty drop divs.
		 *                  dropOnMouseOver,
		 *                  dropOnMouseOut, 
		 *                  dropOnClick: Functions taking the EVENT (not the drop div element). event.target is the element.
		 *                               Applied to all non-empty drop divs.
		 * @param parent The parent HTML element
		 * @returns {Element|boolean} The plate widget element, or false if cannot be rendered due to missing drops or plate type
		 */
		render:function(details, parent){
			
			let drops=details.wellDrops;
			let plateType=details.plateType;
			if(!drops || !plateType){
				alert("Cannot render plate widget - plate type and/or well drops not provided");
				return false;
			}
			//wrapper div
			let w=document.createElement("div");
			w.plateType=plateType;
			w.details=details;
			w.classList.add("platewidget");
			if(details.id){ w.id=details.id; }
			
			//drop mapping/picker
				let dp=document.createElement("table");
				dp.classList.add("pw_droppicker");
				let dm=plateType["dropmapping"];
				let rows=dm.split(",");
				rows.each(function(r){
					let tr=document.createElement("tr");
					for (let i = 0; i < r.length; i++) {
						let td = document.createElement("td");
						let cellContent = r[i];
						td.innerHTML = cellContent;
						if (parseInt(cellContent)) {
							td.addClassName("pw_drop pw_empty drop" + cellContent);
							td.dataset.dropnumber = cellContent;
						} else if ("R" === cellContent) {
							td.classList.add("pw_reservoir");
						}
						tr.appendChild(td);
					}
					dp.appendChild(tr);
				});
				w.appendChild(dp);
				if("left"===details.dropPickerPosition){ dp.style.float="left"; }
				if(plateType["subs"]<=1){
					dp.style.display="none";
				}

				//plate table
				let t=document.createElement("table");
				t.classList.add("pw_plate");
					//Header row
					let tr=document.createElement("tr");
					let th=document.createElement("th");
					tr.appendChild(th);
					for(let c=1;c<=plateType.cols;c++){
						th=document.createElement("th");
						th.innerHTML=c+"";
						tr.appendChild(th);
					}
					t.appendChild(tr);

					//One row per plate type row
					for(let r=1;r<=plateType.rows;r++){
						tr=document.createElement("tr");
						tr.classList.add("row"+r);
						th=document.createElement("th");
						th.innerHTML=Plate.rowLabels[r];
						tr.appendChild(th);
						for(let c=1;c<=plateType.cols;c++){
							let td=document.createElement("td");
							let wellNum=c+((r-1)*plateType.cols);
							td.addClassName("pw_well pw_empty row"+r+" col"+c+" well"+wellNum);
							td.dataset.row=""+r;
							td.dataset.col=""+c;
							td.dataset.wellNumber=""+wellNum;
							td.dataset.isInBottomHalf=(r>plateType["rows"]/2) ? "1" : "0";
							td.dataset.isInRightHalf=(c>plateType["cols"]/2) ? "1" : "0";
							if(details.id){ td.id=details.id+"_"+r+"_"+c; }
							for(let d=1; d<=plateType["subs"]; d++){
								let dd=document.createElement("div");
								dd.addClassName("pw_drop pw_empty row"+r+" col"+c+" drop"+d);
								dd.dataset.row=""+r;
								dd.dataset.col=""+c;
								dd.dataset.dropnumber=""+d;
								if(details.id){ dd.id=details.id+"_"+r+"_"+c+"_"+d; }
								td.appendChild(dd);
							}	
							tr.appendChild(td);
						}
						t.appendChild(tr);
					}
				w.appendChild(t);
			if("absolute"!==parent.style.position){ parent.relativize(); }
			parent.appendChild(w);

			//iterate through drops and attach, removing class name "empty"
			drops.each(function(d){
				let cell=t.down("tr.row"+d.row).down("td.col"+d.col);
				let dropDiv=cell.down(".drop"+d.dropnumber);
				dropDiv.wellDrop=d;
				dropDiv.dataset.welldropid=d.id;
				dropDiv.classList.remove("pw_empty");
				cell.classList.remove("pw_empty");
				if(dp){
					dp.down("td.drop"+d.dropnumber).classList.remove("pw_empty");
				}
			});
			
			if(dp){
				dp.select("td.pw_drop").each(function(td){
					if(!td.hasClassName("pw_empty")){
						td.observe("click",PlateWidget.dropPickerSetDrop);
					}
				});
			}
			
			w.setCurrentDrop=function(rowNumber,colNumber,dropNumber){
				return PlateWidget.setCurrentDrop(w,rowNumber,colNumber,dropNumber);
			};

			let currentRow=1;
			let currentCol=1;
			let currentDrop=1;
			if(details["currentDropIndex"] && details["currentDropIndex"]<=details.wellDrops.length){
				let wd=details.wellDrops[details["currentDropIndex"]];
				currentRow=wd.row;
				currentCol=wd.col;
				currentDrop=wd.dropnumber;
			}
			w.setCurrentDrop(currentRow,currentCol,currentDrop);

			window.setTimeout(function(){PlateWidget.afterRender(w)},50);
			return $(w);
		},
		/**
		 * called by PlateWidget.render. Do not call directly.
		 * @param plateWidget
		 */
		afterRender:function(plateWidget){
			let p=plateWidget.down(".pw_plate");
			p.absolutize();
			let offset=ui.positionedOffset(p);
			let newHeight=plateWidget.getHeight()-offset.top;
			let newWidth=plateWidget.getWidth()-offset.left;
			let cellHeight=newHeight/(1+(1*plateWidget.plateType.rows));
			let cellWidth=newWidth/(1+(1*plateWidget.plateType.cols));
			let cellSize=Math.min(cellHeight,cellWidth);
			let firstWell=p.down("td");
			let cellBorder=parseInt(window.getComputedStyle(firstWell).borderLeftWidth);
			cellSize-=cellBorder;
			p.select("th").each(function(c){
				c.style.height=cellSize+"px";
				c.style.lineHeight=cellSize+"px";
				c.style.width=cellSize+"px";
				c.style.minWidth=cellSize+"px";
			});
			if(plateWidget.down("table.pw_droppicker")){
				plateWidget.down("table.pw_droppicker").style.marginTop=cellSize+"px";
			}
			window.setTimeout(function(){PlateWidget.attachEventsAndRenderDrops(plateWidget)},10);
		},
		/**
		 * called by PlateWidget.afterRender. Do not call directly.
		 * @param plateWidget
		 */
		attachEventsAndRenderDrops:function(plateWidget){
			let details=plateWidget.details;
			if(!details){ return; }
			plateWidget.select("div.pw_drop").each(function(d){
				if(d.hasClassName("pw_empty")){ return; /*from this iteration*/ }
				d.observe("click",PlateWidget.dropDivSetDrop);
				if(details.dropRenderer){ details.dropRenderer(d); }
				if(details.dropOnClick){ d.observe("click",details.dropOnClick); }
				if(details.dropOnMouseover){ d.observe("mouseover",details.dropOnMouseover); }
				if(details.dropOnMouseout){ d.observe("mouseout",details.dropOnMouseout); }
			});
			if(!!window.MSInputMethodContext && !!document["documentMode"]/*IE11*/){
				//because IE is special
				let firstTd=plateWidget.down("table.pw_plate td");
				let borderWidth=parseInt(firstTd.getStyle("borderLeftWidth"));
				let h=(firstTd.getHeight()-(2*borderWidth))+"px";
				let w=(firstTd.getWidth()-(2*borderWidth))+"px";
				plateWidget.select("div.pw_drop").each(function(d){
					d.setStyle({ width:w, height:h })
				});
			}
		},
		
		
		dropPickerSetDrop:function(evt){
			let td=evt.target.closest("td");
			if(td.hasClassName("pw_current")){ return false; }
			let newDropNumber=td.dataset.dropnumber;
			let w=td.up(".platewidget");
			w.setCurrentDrop(null,null,newDropNumber);
		},
		
		dropDivSetDrop:function(evt){
			let d=evt.target;
			if(d.up(".pw_drop")){ d=d.up(".pw_drop"); }
			d.dataset.foobar="hello";
			PlateWidget.setCurrentDrop(d.up(".platewidget"),d.dataset.row, d.dataset.col, null);
		},
		
		setCurrentDrop:function(plateWidget,rowNumber,colNumber,dropNumber){
			if(null!=rowNumber){ plateWidget.currentRow=rowNumber; }
			if(null!=colNumber){ plateWidget.currentCol=colNumber; }
			if(null!=dropNumber){ 
				plateWidget.currentDrop=dropNumber; 
				if(plateWidget.down(".pw_droppicker")){
					plateWidget.down(".pw_droppicker").select("td").each(function(td){
						td.classList.remove("pw_current");
						if(td.hasClassName("drop"+dropNumber)){
							td.classList.add("pw_current");
						}
					});
					plateWidget.down(".pw_plate").select("div.pw_drop").each(function(dd){
						dd.classList.remove("pw_current");
						if(1*dd.dataset.dropnumber===1*dropNumber){
							dd.classList.add("pw_current");
						}
					});
				}
			}
			//set current on plate cell
			let plateTable=plateWidget.down(".pw_plate");
			let currentWell=plateTable.down("td.pw_current");
			if(currentWell){ currentWell.classList.remove("pw_current"); }
			currentWell=plateTable.down("tr.row"+plateWidget.currentRow).down("td.col"+plateWidget.currentCol);
			currentWell.classList.add("pw_current");
			return currentWell.down("div.drop"+plateWidget.currentDrop);
		}

};

let Plate={

		/**
		 * Renders the HTML for a plate widget and attaches it to the parent.
		 * @see PlateWidget.render for documentation. This is an alias for that.
		 */
		plateWidget:function(details, parent){
			return PlateWidget.render(details, parent);
		},
		
		//array is zero-based, plate row numbers are 1-based, so empty item first.
		rowLabels:['','A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R','S','T'],
		
		constructs:null,
		plateType:null,
		
		/*
		 * Selected coordinate ranges for setting a construct. These defaults do all drops in the entire plate.
		 * These are 1-based, ie A1 drop 1 is selectionDrop, selectionTop and selectionLeft all 1.
		 */
		selectionDrop:-1,
		selectionTop:-1,
		selectionBottom:10000,
		selectionLeft:-1,
		selectionRight:10000,

		getWellName:function(rowNumber, colNumber, dropNumber){
			let col=colNumber+"";
			if(colNumber<10){ col="0"+colNumber; }
			if(!dropNumber){ return Plate.rowLabels[rowNumber]+ col; }
			return Plate.rowLabels[rowNumber]+ col +"."+dropNumber;
		},
		
		getWellNameFromObject:function(obj){
			return Plate.getWellName(obj.row, obj.col, obj.dropnumber);
		},
		
		/*
		 * Update isn't destroyed. On success, reload the page.
		 * @return bool false if action was cancelled at confirm(), otherwise true. Note that true ONLY
		 *         means the request was fired, and does NOT imply that it was successful.
		 */
		setIsDestroyed:function(plateId, isDestroyed){
			let msg;
			let newValue;
			if(!isDestroyed){
				msg='Change plate status to "Not destroyed"?';
				newValue="0000-00-00";
			} else {
				msg='Mark plate as destroyed?';
				newValue=new Date().toISOString().split('T')[0];
			}
			if(!confirm(msg)){ return false; }
			new Ajax.Request("/api/plate/"+plateId,{
				"method":"patch",
				"parameters":{ csrfToken:csrfToken, "datedestroyed":newValue },
				onSuccess:function(transport){
					if(304===transport.status){
						alert("Plate has already been destroyed");
					}
					ui.forceReload();
				},
				onFailure:AjaxUtils.checkResponse
			});
			return true; //Request was made. DOES NOT mean success.
		},
		
		
		/*
	 	 * PLATE INSPECTIONS tab
	 	 */
		
		getManualImagers: function(){
			if(!document.manualImagers){
				new Ajax.Request('/api/imager/manualimaging/1?sortby=name&pagesize=100:pagenumber=1',{
					method:'get',
					onFailure:function(){
						document.manualImagers=null;
					},
					onSuccess:function(transport){
						document.manualImagers=transport.responseJSON.rows;
					},
				});
			}
		},
		
		beginCreateInspection: function(){
			if(!canCreateInspections){
				alert("Either manual creation of plate inspections is not enabled in IceBear, or you do not have permission to create them on this plate.");
				return false;
			}
			if(!document.manualImagers){
				alert("No imagers suitable for manual imaging are defined in IceBear.");
				return false;
			}
			let mb=ui.modalBox({
				id:'inspectiondetails',
				title:"New plate inspection on plate "+data.name
			});
			Plate.renderSetInspectionDetailsForm(mb);
		},
		
		beginEditInspection: function(btn){
			if(!canCreateInspections){
				alert("Either manual editing of plate inspections is not enabled in IceBear, or you do not have permission to edit them on this plate.");
				return false;
			}
			let inspection=btn.up("tr").rowData;
			if(!1*inspection["manualimaging"]){
				alert("Either manual editing of plate inspections is not enabled in IceBear, or this inspection was done on an automatic imager and cannot be edited.");
				return false;
			}
			Plate.editInspection(inspection);
		},
		
		editInspection:function(inspection){
			let ts=ui.modalTabSet();
			ts.imagingsession=inspection;
			ts.tab({
				id:'inspectiondetails',
				label:'Basic details'
			});
			let imagesTab=ts.tab({
				id:'inspectionimages',
				label:'Images'
			});
			new Ajax.Request('/api/imagingsession/'+inspection.id+'/dropimage?pagesize=10000&pagenum=1',{
				method:'get',
				onFailure:function(transport){
					if(404===transport.status){
						imagesTab.images=[];
					} else {
						return AjaxUtils.checkResponse(transport);
					}
				},
				onSuccess:function(transport){
					imagesTab.images=transport.responseJSON.rows;
				}
			});
			Plate.renderSetInspectionDetailsForm($('inspectiondetails_body'),inspection);
			window.setTimeout(function(){
				Plate.renderSetInspectionImagesForm(inspection);
				$("inspectionimages").click();
			},50);
		},
		
		renderSetInspectionDetailsForm: function(parent,inspection){
			
			let imagerId=0;
			let imagedTime='';
			let formAction='/api/imagingsession';
			let formMethod="post";
			if(inspection){
				imagerId=inspection["imagerid"];
				imagedTime=inspection["imageddatetime"];
				formAction+="/"+inspection.id;
				formMethod="patch";
			}
			let imagerOptions = [];
			document.manualImagers.each(function(im){
				if(im["manualimaging"]){
					imagerOptions.push({ label:im["name"]+" ("+im["temperature"]+"C)", value:im["id"] });
				}
			});

			let f=parent.form({
				method:formMethod,
				action:formAction
			});
			
			f.hiddenField('plateid',data.id);
			f.hiddenField('name','');
			f.hiddenField('manufacturerdatabaseid',0);
			
			f.datePicker({
				label:'Inspection date and time',
				name:'imageddatetime',
				value:imagedTime,
				showTime:true,
				minuteStep:5
			});
			f.dropdown({
				label:'Imaging device',
				name:'imagerid',
				value:imagerId,
				options:imagerOptions
			});
			f.formField({
				label:'Plate',
				content:data.name
			});
			
			if(!inspection){
				f.createButton({
					beforeSubmit:function(){
						f.down("input[name=name]").value=data.name+" "+new Date().getTime();
					},
					afterSuccess:function(responseJSON){
						Plate.editInspection(responseJSON.created);
						$("inspections").refresh();
					}
				});
			} else {
				f.select("label").each(function(lbl){
					lbl.afterUpdate=Plate.afterInspectionUpdate;
				});
			}
		},
		
		renderSetInspectionImagesForm: function(inspection){
			let ii=$("inspectionimages");
			if(!inspection){
				alert('Inspection ID not provided');
				return false;
			}
			if(undefined===ii.images){
				window.setTimeout(function(){Plate.renderSetInspectionImagesForm(inspection)}, 50);
				return false;
			}
			let w=Plate.plateWidget({
				plateType:data.plateType,
				wellDrops:data.welldrops,
				dropPickerPosition:"left",
				dropRenderer:Plate.renderImageEditorDrop,
				dropOnClick:Plate.onImageEditorDropClick,
				dropOnMouseover:Plate.onImageEditorDropMouseover,
				dropOnMouseout:Plate.onImageEditorDropMouseout
			},$("inspectionimages_body"));
			if(0!==ii.images.length){
				ii.images.each(function(im){
					let div=w.down("tr.row"+im.row).down("td.col"+im.col).down("div.drop"+im.dropnumber);
					div.image=im;
				});
			}
			w.querySelector("td.pw_drop").addEventListener("click",Plate.closeContextMenu);
		},

		renderImageEditorDrop:function(dropDiv){
			dropDiv.innerHTML="";
			let im=dropDiv.image;
			if(im){
				let img=document.createElement("img");
				let imageAspectRatio=im["pixelwidth"]/im["pixelheight"];
				let containerAspectRatio=dropDiv.getWidth()/dropDiv.getHeight();
				if(imageAspectRatio>=containerAspectRatio){
					img.style.width="100%";
				} else {
					dropDiv.style.textAlign="center";
					img.style.height="100%";
				}
				img.src="/dropimagethumb/"+im.id;
				dropDiv.appendChild(img);
			}
			let overlay=document.createElement("label");
			overlay.classList.add("pw_overlay");
			overlay.setStyle({
				position:"absolute",
				top:0, left:0, right:0, bottom:0,
				backgroundSize:"50% 50%", backgroundPosition:"center center", backgroundRepeat:"no-repeat",
				textAlign:"center",
				display:"block",
				opacity:0.0
			});
			dropDiv.appendChild(overlay);
			dropDiv.addEventListener("contextmenu",Plate.onImageEditorContextMenu);
			if(im){
				overlay.style.backgroundImage='url(/images/icons/'+skin["bodyicontheme"]+'/no.gif)';
				overlay.title="Click to remove this image. Right-click to set its scale.";
			} else {
				overlay.style.backgroundImage='url(/images/icons/'+skin["bodyicontheme"]+'/btn_plus.gif)';
				overlay.title="Add an image";
				let f=document.createElement("input");
				f.type="file";
				f.observe("change",Plate.startDropImageAdd);
				overlay.appendChild(f);
				overlay.htmlFor=f;
				f.setStyle({ position:"absolute",top:0,left:0,height:"100%",width:"100%", opacity:"0.0" });
			}
		},		

		onImageEditorContextMenu:function (evt){
			evt.preventDefault();
			Plate.closeContextMenu(evt);
			PlateWidget.dropPickerSetDrop(evt);
			let dropDiv=evt.target.closest(".pw_drop");
			let tb=document.getElementById("inspectionimages_body");
			let f=ui.form({"id":"dropcontext"});
			f.dropDiv=dropDiv;
			dropDiv.click();
			f.style.width="20em";
			f.style.zIndex="999999";
			f.style.padding="0.5em";
			f.style.border=getComputedStyle(tb)["border"];
			f.style.background=getComputedStyle(tb)["background"];
			if(dropDiv.image){
				f.dataset.imageId=dropDiv.image.id;
				let scale=parseFloat(dropDiv.image.micronsperpixelx);
				if(0>=scale){ scale=""; }
				f.textField({ "label":"Scale (&#181;m/pixel)","value":scale+"" })
				f.buttonField({ "label":"Delete image", "id":"deletebtn", onclick:Plate.handleContextMenuDelete }).style.textAlign="left";
				f.addEventListener("submit",Plate.handleContextMenuSubmit);
			} else {
				f.buttonField({ "label":"Add image", "id":"addbtn", onclick:Plate.handleContextMenuAdd }).style.textAlign="left";
			}
			tb.appendChild(f);
			f.style.position="absolute";
			if("1"===dropDiv.closest("td").dataset.isInBottomHalf){
				f.style.top=(ui.cumulativeOffset(dropDiv).top - ui.cumulativeOffset(tb).top - f.offsetHeight)+"px";
			} else {
				f.style.top=(ui.cumulativeOffset(dropDiv).top - ui.cumulativeOffset(tb).top + dropDiv.offsetHeight)+"px";
			}
			if("1"===dropDiv.closest("td").dataset.isInRightHalf){
				f.style.left=(dropDiv.offsetWidth+ui.cumulativeOffset(dropDiv).left- ui.cumulativeOffset(tb).left- f.offsetWidth)+"px";
			} else {
				f.style.left=(ui.cumulativeOffset(dropDiv).left- ui.cumulativeOffset(tb).left)+"px";
			}
			tb.addEventListener("click",Plate.closeContextMenu);
		},

		handleContextMenuSubmit:function(evt){
			evt.preventDefault();
			Plate.startDropImageSetScale(evt);
		},
		handleContextMenuDelete:function(evt){
			evt.preventDefault();
			Plate.startDropImageRemove(evt.target.closest("form").dropDiv);
		},
		handleContextMenuAdd:function(evt){
			evt.preventDefault();
			evt.target.closest("form").dropDiv.querySelector("input[type=file]").click();
		},

		closeContextMenu:function(evt){
			let clicked=(evt) ? evt.target : null;
			let tb=document.getElementById("inspectionimages_body");
			let dc=document.getElementById("dropcontext");
			if(dc && (!clicked || !clicked.closest("#dropcontext"))){ dc.remove(); }
			tb.removeEventListener("click",Plate.closeContextMenu);
		},

		onImageEditorDropClick:function(evt){
			let dropDiv=evt.target;
			if(!dropDiv.hasClassName(".pw_drop")){ dropDiv=dropDiv.up(".pw_drop"); }
			if(dropDiv.image){
				Plate.startDropImageRemove(dropDiv);
			} else {
				//Handled by file input and its onchange event
			}
			return true;
		},
		onImageEditorDropMouseover:function(evt){
			let dropDiv=evt.target;
			if(dropDiv.tagName.toLowerCase()==="input"){ dropDiv=dropDiv.up(".pw_drop"); }
			if(!dropDiv.hasClassName("pw_overlay")){ dropDiv=dropDiv.down(".pw_overlay"); }
			dropDiv.style.opacity="0.6";
			if(dropDiv.image){
				//show delete icon
			} else {
				//show add icon
			}
		},
		onImageEditorDropMouseout:function(evt){
			let dropDiv=evt.target;
			if(dropDiv.tagName.toLowerCase()==="input"){ dropDiv=dropDiv.up(".pw_drop"); }
			if(!dropDiv.hasClassName("pw_overlay")){ dropDiv=dropDiv.down(".pw_overlay"); }
			dropDiv.style.opacity=0;
		},

		startDropImageSetScale:function (evt){
			let frm=evt.target;
			let field=frm.querySelector("input[type=text]");
			let newScale=field.value.trim();
			let lbl=field.closest("label");
			let dropDiv=frm.dropDiv;
			let imageId=frm.dataset.imageId;
			if(""===newScale){
				newScale=-1;
			} else if(!parseFloat(newScale)){
				lbl.classList.add("invalidfield");
				return false;
			} else if(0>=parseFloat(newScale)){
				newScale=-1;
			}
			lbl.classList.add("updating");
			new Ajax.Request("/api/dropimage/"+imageId,{
				method:"patch",
				parameters:{
					csrfToken:csrfToken,
					micronsperpixelx:newScale,
					micronsperpixely:newScale,
				},
				onSuccess:function (transport){
					dropDiv.image=transport.responseJSON["updated"];
					Plate.closeContextMenu();
				},onFailure:function (){
					Plate.closeContextMenu();
					alert("Could not set scale information");
				}
			});
		},

		startDropImageAdd:function(evt){
			Plate.closeContextMenu();
			let fileField=evt.target; //the file input
			fileField.disabled="disabled";
			let dropDiv=fileField.closest(".pw_drop");
			dropDiv.classList.add("updating");
			dropDiv.querySelector("label").style.background="none"; //Don't show (+) on mouseover
			let formData=new FormData();
			formData.append('dropimage', fileField.files[0], fileField.files[0].name);
			formData.append('csrfToken', encodeURIComponent(csrfToken));
			formData.append('imagingsessionid', encodeURIComponent($("modalTabSet").imagingsession.id));
			formData.append('welldropid', encodeURIComponent(dropDiv.dataset.welldropid));
			let xhr=new XMLHttpRequest();
			let method="post";
	        xhr.open(method, "/api/dropimage", true);
	        xhr.onload=function(){
	        	try{
	        		xhr.responseJSON=JSON.parse(xhr.responseText);
	        	} catch(ex) {
	        		alert("Could not understand reply from server:\n\n"+xhr.responseText);
	        		return false;
	        	}
	            if(200===xhr.status || 201===xhr.status){
	                if(xhr.responseJSON.created){
	                	dropDiv.image=xhr.responseJSON.created;
	                }
	            } else if(401===xhr.status){
	            	ui.handleSessionExpired();
	            } else if(xhr.responseJSON.error){
	            	alert("Image upload failed. The server said:\n\n"+xhr.responseJSON.error);
	            } else {
	            	alert("Image upload failed. The server said:\n\n"+xhr.responseText);
	            }
	            dropDiv.classList.remove("updating");
	            Plate.renderImageEditorDrop(dropDiv);
	        };
			xhr.send(formData);
			
		},
		
		startDropImageRemove:function(dropDiv){
			if(!dropDiv.image){ alert("No image attached to that drop"); return false; }
			let dropName=Plate.rowLabels[dropDiv.dataset.row];
			if(10>(1*dropDiv.dataset.col)){ dropName+="0"; }
			dropName+=dropDiv.dataset.col+" drop "+dropDiv.dataset.dropnumber;
			if(!confirm("Really remove image from "+dropName+"?")){
				return false;
			}
			dropDiv.down("label").style.background="none"; //Don't show (x) on mouseover
			dropDiv.classList.add("updating");
			new Ajax.Request('/api/dropimage/'+dropDiv.image.id,{
				method:'delete',
				postBody:'csrfToken='+csrfToken,
				onSuccess:function(transport){ Plate.dropImageRemove_onSuccess(transport,dropDiv); },
				onFailure:function(transport){ Plate.dropImageRemove_onFailure(transport,dropDiv); },
			});
		},
		dropImageRemove_onSuccess:function(transport,dropDiv){
			AjaxUtils.checkResponse(transport);
			if(transport.responseJSON["deleted"]){ dropDiv.image=null; }
            Plate.renderImageEditorDrop(dropDiv);
			dropDiv.classList.remove("updating");
			Plate.closeContextMenu();
		},
		dropImageRemove_onFailure:function(transport,dropDiv){
			AjaxUtils.checkResponse(transport);
            Plate.renderImageEditorDrop(dropDiv);
			dropDiv.classList.remove("updating");
			Plate.closeContextMenu();
		},
		
		afterInspectionUpdate:function(){
			$("inspections").refresh();
		},
		
		/*
	 	 * PROTEIN tab
	 	 */

		/**
		 * Renders the "Protein" tab.
		 * Generates a well map enabling different proteins to be set in different drop positions, i.e., one protein in ALL drop 1s,
		 * another protein in ALL drop 2s, etc. 
		 * ASSUMPTIONS: All plates will have the same protein in all drop 1s, the same protein in all drop 2s, etc. This assumption
		 * holds true as long as both (1) there is no UI for setting more complex arrangements, and (2) no API caller does it either.
		 */
		renderProteinTab:function(){
			let tb=$("proteins_body");
			tb.innerHTML='';
			if(null==Plate.constructs || !data.plateType || !data.welldrops){
				window.setTimeout(Plate.renderProteinTab,50);
				return;
			}
			let dropMap=PlateType.getDropMapElement(data.plateType,tb);
			dropMap.style.height="100%";
			dropMap.style.width=dropMap.getHeight()+"px";
			let drops=dropMap.select(".dropmapwell");
			drops.each(function(d){
				
				d.style.overflow="auto";
				let wellDrop, proteinAmount, proteinUnit, buffer;
				//Find first drop with this drop number
				for(let i=0;i<data.welldrops.length;i++){
					let wd=data.welldrops[i];
					if(0===wd.dropnumber-1*d.dataset.dropnumber){
						wellDrop=wd;
						proteinAmount=wd.proteinconcentrationamount;
						proteinUnit=wd.proteinconcentrationunit;
						buffer=wd.proteinbuffer;
						break;
					}
				}

				if(undefined===wellDrop){ return; }

				let f=ui.form({
					action:'/api/plate/'+data.id,
					method:'patch'
				},d);
				f.dataset.constructid=wellDrop.constructid;
				f.selectionTop=1;
				f.selectionLeft=1;
				f.selectionRight=1*data.plateType.cols;
				f.selectionBottom=1*data.plateType.rows;
				f.selectionDrop=1*d.dataset.dropnumber;
				Plate.renderDropConstructFormFields(f, wellDrop);
			});
			window.setTimeout(function(){
				let margins=20;
				dropMap.style.width=($("proteins_body").getWidth()-margins)+"px";
				let screensTab=$("screens");
				if(screensTab){
					screensTab.refresh();
				}
			},100);

			if(!canEdit){ return; }

			//"Copy from other drop"
			drops.each(function(drop){
				let toDrop=1*drop.dataset.dropnumber;
				if(0!==1*(drop.down("form").dataset.constructid)){
					//Drops in this position already have a construct
					return; //from this iteration
				}
				let fromDrops=[];
				drops.each(function(otherDrop){
					let fromDrop=1*otherDrop.dataset.dropnumber;
					if(fromDrop===toDrop){
						//Can't copy onto ourselves
						return; //from this iteration
					} else if(0===1*(otherDrop.down("form").dataset.constructid)){
						//other drop has no construct, nothing to copy from
						return; //from this iteration
					}
					fromDrops.push(fromDrop);
				});
				if(0<fromDrops.length){
					drop.down("form").innerHTML+='<label class="radiohead"><span class="label">Copy from other drop positions:</span>&nbsp;</label>';
					fromDrops.each(function(fromDrop){
						let field=drop.down("form").buttonField({
							label:'Copy from position '+fromDrop,
							
						});
						field.innerHTML+='&nbsp;';
						field.down("input").onclick=function(){ Plate.copyProteinDetailsAcrossDropPositions(fromDrop, toDrop); };
					});
				}
			});
			
		},

		/**
		 * Renders the form for protein/construct choice and, if construct is set, protein buffer and concentration.
		 * 
		 * ASSUMPTION: This is used in the context of a well drop map, where we can pass in a single drop and assume that
		 * all drops being affected by this form have the same protein/construct, buffer, and concentration.
		 * 
		 * @param f The HTML form element
		 * @param wellDrop One of the well drops affected by this form. See assumption above.
		 */
		renderDropConstructFormFields:function(f, wellDrop){
			let proteinBuffer=wellDrop.proteinbuffer;
			let concAmount=wellDrop.proteinconcentrationamount;
			let concUnit=wellDrop.proteinconcentrationunit;
			f.innerHTML='';
			let out='';
			let constructId=1*f.dataset.constructid;
			f.dataset.proteinconcentrationamount=concAmount;
			f.dataset.proteinconcentrationunit=concUnit;
			f.dataset.proteinbuffer=proteinBuffer;
			if(!constructId){
				
				//"choose construct..."
				out+='<label style="text-align:left">'+
				'<strong>No construct chosen</strong>';
				if(canEdit){
					out+='<input type="button" value="Choose..."  onclick="Plate.beginSetProtein(this);" />';
				}
				out+='</label>';
				f.innerHTML=out;

			} else {
				
				let construct=null;
				Plate.constructs.each(function(c){
					if(1*c.id===constructId){ construct=c; }
				});
				if(!construct){ 
					alert("Drop "+wellDrop.dropnumber+" has a protein/construct not found in Plate.constructs. This shouldn't happen.");
					return; 
				}
				
				out+='<label style="text-align:left">'+
					'<strong>Protein:</strong> '+construct.proteinacronym+' - '+construct["proteinname"]+'<hr/>'+
					'<strong>Construct:</strong> '+construct.name+' - '+construct.description+'<br><a href="#" onclick="Plate.showSequencesForConstruct(this);return false;">Show sequence...</a>';
				if(canEdit){
					out+='<input type="button" value="Change..."  onclick="Plate.beginSetProtein(this);" />';
				}
				out+='<div class="shim">&nbsp;</div></label>';
				//protein buffer
				f.innerHTML=out;
				bufferBox='<textarea name="proteinbuffer" id="proteinbuffer">'+proteinBuffer+'</textarea>';
				if(!canEdit){
					bufferBox=proteinBuffer;
				}
				let buff=f.formField({
					label:'Protein buffer',
					content:bufferBox,
					value:proteinBuffer,
				});
				if(canEdit){
					buff.down("textarea").observe("blur",Plate.updateProteinConcentrationOrBuffer);
					buff.down("textarea").observe("keyup",Plate.updateProteinConcentrationOrBuffer);
					buff.down("textarea").observe("change",Plate.updateProteinConcentrationOrBuffer);
				}
				//protein concentration
				let concContent=concAmount+concUnit.replace("u","&#181;");
				if(canEdit){
					concContent='<input type="text" id="proteinconcentrationamount" name="proteinconcentrationamount" value="'+concAmount+'" style="width:3em"/>';
					concContent+='<select id="proteinconcentrationunit" name="proteinconcentrationunit">';
					concContent+='<option value="mg/mL" '+( 'mg/mL'===concUnit ?  'selected="selected"' : '' )+'>mg/mL</option>';
					concContent+='<option value="ug/uL" '+( 'ug/uL'===concUnit ?  'selected="selected"' : '' )+'>&#181;g/&#181;L</option>';
					concContent+='</select>';
				}
				let concField=f.formField({
					label:'Protein concentration',
					content:concContent
				});
				if(canEdit){
					fieldValidations["proteinconcentrationamount"]=["required","float"];
					buff.observe("keyup",Plate.updateProteinConcentrationOrBuffer);
					concField.down('[name=proteinconcentrationamount]').observe("blur",Plate.updateProteinConcentrationOrBuffer);
					concField.down('[name=proteinconcentrationamount]').observe("keyup",Plate.updateProteinConcentrationOrBuffer);
					concField.down('[name=proteinconcentrationamount]').observe("change",Plate.updateProteinConcentrationOrBuffer);
					concField.down('[name=proteinconcentrationunit]').observe("blur",Plate.updateProteinConcentrationOrBuffer);
					concField.down('[name=proteinconcentrationunit]').observe("keyup",Plate.updateProteinConcentrationOrBuffer);
					concField.down('[name=proteinconcentrationunit]').observe("change",Plate.updateProteinConcentrationOrBuffer);
				}
			}
			Plate.setProteinTabWarning(f);
		},

		setProteinTabWarning:function(frm){
			let tbl=frm.up("table");
			let tb=tbl.up(".tabbody");
			let hasAllConstructs=true;
			let hasAllBufferInfo=true;
			tbl.select("td.dropmapwell form").each(function (f) {
				if(!f.dataset.constructid){
					hasAllConstructs=false;
				} else if(""===f.dataset.proteinbuffer || ""===f.dataset.proteinconcentrationunit || ""===f.dataset.proteinconcentrationamount){
					hasAllBufferInfo=false;
				}
			});
			if(!hasAllConstructs){
				tb.warn("Protein and construct are not specified for all drops");
			} else if(!hasAllBufferInfo){
				tb.warn("Protein buffer and concentration are not specified for all drops");
			} else {
				tb.unwarn();
			}
		},

		showSequencesForConstruct:function(elem){
			let frm=elem.up("form");
			if(!frm){ return false; }
			let constructId=frm.dataset.constructid;
			if(!constructId){ return false; }
			new Ajax.Request("/api/construct/"+constructId+"/sequence",{
				method:"get",
				onSuccess:Plate.showSequencesForConstruct_onSuccess,
				onFailure:Plate.showSequencesForConstruct_onFailure,
			});
		},

		showSequencesForConstruct_onSuccess:function(transport){
			let mb=ui.modalBox({
				title:"Sequences"
			});
			transport.responseJSON.rows.each(function(seq){
				let f=mb.form({
					readonly:true
				});
				f.style.marginBottom="1em";
				f.insert({bottom:'<label><h3>'+seq.name+'</h3></label>'});
				if(""!==seq.dnasequence){
					f.formField({
						label:"DNA sequence",
						content:'<div style="float:right;text-align:left;width:80%;max-width:80%">'+seq.dnasequence+'</div><div style="clear: both"></div>'
					})
				}
				if(""!==seq.proteinsequence){
					f.formField({
						label:"Protein sequence",
						content:'<div style="float:right;text-align:left;width:80%;max-width:80%">'+seq.proteinsequence+'</div><div style="clear: both"></div>'
					})
				}
			});
		},

		showSequencesForConstruct_onFailure:function(transport){
			if(404===transport.status){
				alert("No sequence information was found.");
			} else {
				AjaxUtils.checkResponse(transport);
			}
		},

	 	/**
	 	 * Begins the process of setting a protein.
	 	 * @param button The button pressed to start this process
	 	 */
		beginSetProtein: function(button){
			if("Default Project"===data["projectname"]){
				new Ajax.Request('/api/project/?pagesize=1000',{
					method:'get',
					onSuccess:function(transport){
						if(!AjaxUtils.checkResponse(transport)){ return false; }
						Plate.showProjectList(transport.responseJSON.rows, button);
					},
					onFailure:function(transport){
						AjaxUtils.checkResponse(transport);
					},
				});
			} else {
				Plate.showProjectList([{ name:data.projectname, id:data.projectid, description:'Plate is in project "'+data.projectname+'". You can choose from proteins in that project' }], button);
			}
		},
		
		/**
		 * Renders a list of projects as tree items in a modal box. Opening a project shows its proteins.
		 * @param projects The projects to render. Each must have name, id and description. The object must have a "rows" key, containing the array of projects.
	 	 * @param button The button pressed to start this process
		 */
		showProjectList: function(projects, button){
			let mb=ui.modalBox({
				'title':'Choose the construct',
				'content':''
			});
			mb.launcherForm=button.up("form");
			Plate.doAfter=Plate.afterSetConstruct;
			projects.each(function(p){
				if(1*p["isarchived"] || 1*p["issystem"]){ return; } //from this iteration
				mb.treeItem({
					record:p,
					id:"project"+p.id,
					header:p.name+": "+p.description+"",
					updater:Plate.showProteinsForProject
				});
			});
			if(1===mb.select(".treeitem").length){
				mb.select(".treehead")[0].click();
			}		
		},

		/**
		 * Within a tree list of projects, shows the proteins for the selected project.
		 * @param elem The clicked element (representing a project).
		 */
		showProteinsForProject: function(elem){
			if(elem.up(".treeitem")){ elem=elem.up(".treeitem"); }
			let proj=elem.record;
			new Ajax.Request('/api/project/'+proj.id+'/protein?pagesize=1000',{
				method:'get',
				onSuccess:function(transport){
					if(!AjaxUtils.checkResponse(transport)){ return false; }
					elem.down(".treebody").innerHTML='';
					transport.responseJSON.rows.each(function(p){
						elem.treeItem({
							record:p,
							id:"protein"+p.id,
							header:p.name+" ("+p.proteinacronym+"): "+p.description+"",
							updater:Plate.showConstructsForProtein
						});
					});
				},
				onFailure:function(transport){
					if(404===transport.status){
						elem.down(".treebody").innerHTML='<label style="text-align:left">No proteins in this project</label>';
						return;
					}
					AjaxUtils.checkResponse(transport);
				},
			});
		},

		/**
		 * Within a tree list of projects and proteins, shows the constructs for the selected protein.
		 * @param elem The clicked element (representing a protein).
		 */
		showConstructsForProtein: function(elem){
			if(elem.up(".treeitem")){ elem=elem.up(".treeitem"); }
			let protein=elem.record;
			new Ajax.Request('/api/protein/'+protein.id+'/construct?pagesize=1000',{
				method:'get',
				onSuccess:function(transport){
					if(!AjaxUtils.checkResponse(transport)){ return false; }
					elem.down(".treebody").innerHTML='';
					transport.responseJSON.rows.each(function(c){
						let lbl=document.createElement("label");
						lbl.className="treeitem";
						lbl.record=c;
						lbl.innerHTML='<input type="button" value="Use this construct" onclick="Plate.setConstruct(this)" style="cursor:pointer" /> '+c.name+": "+c.description;
						lbl.style.textAlign="left";
						lbl.style.padding="0.5em 0.25em";
						elem.down(".treebody").appendChild(lbl);
					});
				},
				onFailure:function(transport){
					if(404===transport.status){
						elem.down(".treebody").innerHTML="No constructs for this protein";
						return;
					}
					AjaxUtils.checkResponse(transport);
				},
			});
		},

		/**
		 * Sets the chosen construct on the relevant protein drops within the plate.
		 * @param btn The clicked button.
		 */
		setConstruct: function(btn){
			let construct=btn.up(".treeitem").record;
			let mb=$("modalBox");
			let frm=mb.launcherForm || mb.down(".boxbody").launcherForm;
			btn.up(".treeitem").classList.add("updating");
			new Ajax.Request("/api/plate/"+data.id, {
				method:'patch',
				parameters:{
					csrfToken:csrfToken,
					constructid:construct.id,
					selectionDrop:frm.selectionDrop,
					selectionTop:frm.selectionTop,
					selectionBottom:frm.selectionBottom,
					selectionLeft:frm.selectionLeft,
					selectionRight:frm.selectionRight,
				},
				onSuccess:Plate.reloadPageAfterUpdate,
				onFailure:function(transport){ 
					AjaxUtils.checkResponse(transport);
				},
			});
		},

		reloadPageAfterUpdate:function(transport){
			AjaxUtils.checkResponse(transport);
			//Reload the page with a unique timestamp, to bypass caching
			document.location.replace('/plate/'+data.id+'?ts=?'+Date.now()+"#"+$("grid").down(".tabset").down(".current").id);
		},
		
		getWellDrops:function(){
			new Ajax.Request("/api/plate/"+data.id+"/welldrop",{
				method:'get',
				onSuccess:Plate.getWellDrops_onSuccess,
				onFailure:Plate.getWellDrops_onFailure,
			});
		},
		getWellDrops_onSuccess:function(transport){
			if(!AjaxUtils.checkResponse(transport)){ return false; }
			data.welldrops=transport.responseJSON.rows;
		},
		getWellDrops_onFailure:function(transport){ 
			return AjaxUtils.checkResponse(transport);
		},

		/**
		 * Copies protein/construct ID, protein buffer, and protein concentration from one drop position to another.
		 * For example, ALL drops in position 1 (A01.1, A02.1, ...H12.1) have the same protein, etc.; after copying
		 * to drop 2, ALL drops in position 2 (A01.2, A02.2, ...H12.2) also have this protein, etc.
		 * @param fromDrop The drop position number to copy from
		 * @param toDrop The drop position number to copy to
		 */
		copyProteinDetailsAcrossDropPositions:function(fromDrop, toDrop){
			let fromForm=null;
			let toForm=null;
			$("proteins_body").select("td.dropmapwell").each(function(d){
				if(1*d.dataset.dropnumber===1*fromDrop){
					fromForm=d.down("form");
				} else if(1*d.dataset.dropnumber===1*toDrop){
					toForm=d.down("form");
				} 
			});
			if(!fromForm || !toForm){
				alert("Could not find source and destination drops");
				return false;
			}
			let constructId=fromForm.dataset.constructid;
			let buffer=fromForm.down("[name=proteinbuffer]").value.trim();
			let concAmount=fromForm.down("[name=proteinconcentrationamount]").value.trim();
			let concUnit=fromForm.down("[name=proteinconcentrationunit]").value.trim();
			toForm.up("td").classList.add("updating");
			toForm.dataset.constructid=constructId;
			let postBody="csrfToken="+csrfToken+
				"&selectionTop="+toForm.selectionTop+
				"&selectionBottom="+toForm.selectionBottom+
				"&selectionLeft="+toForm.selectionLeft+
				"&selectionRight="+toForm.selectionRight+
				"&selectionDrop="+toForm.selectionDrop+
				"&constructid="+constructId;
			if(""!==buffer){ postBody+="&proteinbuffer="+buffer; }
			if(""!==concAmount){ postBody+="&proteinconcentrationamount="+concAmount; }
			if(""!==concUnit){ postBody+="&proteinconcentrationunit="+concUnit; }
			new Ajax.Request('/api/plate/'+data.id, {
				method:'patch',
				postBody:postBody,
				onSuccess:Plate.reloadPageAfterUpdate,
				onFailure:function(transport){ Plate.copyProteinDetailsAcrossDropPositions_onFailure(transport, f); }
			});
		},
		copyProteinDetailsAcrossDropPositions_onFailure:function(transport,toForm){
			toForm.up("td").classList.remove("updating");
			AjaxUtils.checkResponse(transport);
		},
		
		updateProteinConcentrationOrBuffer:function(evt){
			let field=evt.target;
			window.clearTimeout(submitTimer);
			if(!validator.validate(field)){ return false; }
			if(field.up("label,td")){ field.up("label,td").classList.remove("invalidfield"); }
			let delay=1500;
			if("blur"===evt.type){ delay=10; }
			window.submitTimer=setTimeout(function(){ Plate.doUpdateProteinConcentrationOrBuffer(field) },delay);
		},
		doUpdateProteinConcentrationOrBuffer:function(field){
			if(field.up(".updating")){
				field.style.border="1px solid red";
				return false;
			}
			field.up("label").classList.add("updating");
			let f=field.up("form");
			new Ajax.Request('/api/plate/'+data.id, {
				method:'patch',
				postBody:"csrfToken="+csrfToken+
					"&selectionTop="+f.selectionTop+
					"&selectionBottom="+f.selectionBottom+
					"&selectionLeft="+f.selectionLeft+
					"&selectionRight="+f.selectionRight+
					"&selectionDrop="+f.selectionDrop+
					"&"+field.name+"="+field.value, //Defaults on server to all drops in all wells
				onSuccess:function(transport){ Plate.updateProteinConcentrationOrBuffer_onSuccess(transport, f); },
				onFailure:function(transport){ Plate.updateProteinConcentrationOrBuffer_onFailure(transport, f); }
			});
		},
		updateProteinConcentrationOrBuffer_onSuccess:function(transport, frm){
			if(!AjaxUtils.checkResponse(transport)){ return false; }
			if(!transport.responseJSON || !transport.responseJSON.updated || !transport.responseJSON.updated.updateddrops){
				return AjaxUtils.checkResponse(transport);
			}
			let updated=transport.responseJSON.updated.updateddrops[0];
			Plate.updateWellDropsByDropNumberAfterChange(updated.dropnumber,{
				'proteinbuffer':updated.proteinbuffer,
				'proteinconcentrationamount':updated.proteinconcentrationamount,
				'proteinconcentrationunit':updated.proteinconcentrationunit
			});
			frm.down('[name=proteinbuffer]').up("label").classList.remove("updating");
			frm.down('[name=proteinconcentrationamount]').up("label").classList.remove("updating");
			Plate.setProteinTabWarning(f);
		},
		updateProteinConcentrationOrBuffer_onFailure:function(transport){
			return AjaxUtils.checkResponse(transport);
		},

		/**
		 * Updates any data.welldrops with the specified drop number, setting the new values supplied.
		 * @param dropNum The drop position number, or -1 for all drops
		 * @param newValues Key-value pairs to set on each of the chosen drops
		 */
		updateWellDropsByDropNumberAfterChange:function(dropNum, newValues){
			dropNum=1*dropNum;
			data.welldrops.each(function(drop){
				if(parseInt(drop.dropnumber)!==dropNum && -1!==dropNum){ return; }
				Object.keys(newValues).each(function(k){
					drop[k]=newValues[k];
				});
			});
		},
		
		getConstructs: function(){
			new Ajax.Request("/api/plate/"+data.id+'/construct', {
				method:'get',
				onSuccess:function(transport){ 
					AjaxUtils.checkResponse(transport);
					Plate.constructs=transport.responseJSON.rows;
				},
				onFailure:function(transport){ 
					if(404===transport.status){
						Plate.constructs=[];
						return;
					}
					Ajax.checkResponse(transport); 
				},
			});
		},
		
	  	/*
	  	 * SCREEN tab
	  	 */
		renderScreenTab: function(){
			let tab=$("screens_body");
			tab.unwarn();
			tab.dataset.apiurl='/api/screen/'+data.screenid+'/screencondition';
			if('Default Project'===data["projectname"]){
				if(canEdit){
					tab.innerHTML='No screen has been chosen for this plate. Set the protein first';
				} else {
					tab.innerHTML='No screen has been chosen for this plate.';
				}
				tab.warn("No screen information has been provided");
				return false;
			}
			if(""===data.screenid){
				tab.warn("No screen information has been provided");
				tab.innerHTML='';
				if(canEdit){
					let ss=document.createElement("div");
					ss.setStyle({ cssFloat:"left", width:"49%" });
					let sf=tab.form({ action:'/api/screen/', method:'post', id:'standardscreenform' });
					sf.hiddenField("name","Optimization "+data.name);
					sf.hiddenField("projectid", data.projectid);
					sf.hiddenField("rows", "");
					sf.hiddenField("cols", "");
					sf.formField({ label:'<h3>Choose a standard screen</h3>',content:'&nbsp;' });
					sf.formField({ label:'',content:'' });
					sf.formField({ label:'Standard screen',content:'<input type="button" value="Choose..." onclick="Plate.beginSetScreen(Plate.afterSetScreen)"/>' });
					ss.appendChild(sf);
					tab.appendChild(ss);

					let os=document.createElement("div");
					let of=tab.form({ action:'/api/screen/', method:'post', id:'optimizationscreenform' });
					of.setStyle({ cssFloat:"right", width:"49%", marginBottom:"1.5em" });
					of.formField({ label:'<h3>Upload an optimization screen file</h3>',content:'&nbsp;' });
					of.formField({ label:'',content:'<input type="file" name="file" onchange="Plate.uploadScreen()" id="createscreenbtn"/>' });
					of.hiddenField("name", "Optimization "+data.name);
					of.hiddenField("projectid", data.projectid);
					of.hiddenField("rows", "");
					of.hiddenField("cols", "");
					os.appendChild(of);
					tab.appendChild(os);

					tab.innerHTML+='<div class="tabhelp">'+
					'<p style="clear:both">You can either (i) use a standard screen or (ii) upload details of an optimization screen. In both cases, the screen will be attached to your plate as a file.</p>'+
					'<h3 style="">Standard screens:</h3>'+
					'<p>If your plate uses a standard screen, simply click the <strong>Choose...</strong> button above and choose your screen from the list.</p>'+
					'<p>To add more standard screens to IceBear, talk to your administrator.</p>'+
					'<h3 style="clear:both;margin-top:1.5em">Optimization screens:</h3>'+
					'<p>You can upload a file describing your optimization screen. Select your file in "Upload an optimization screen file" above, and it will be uploaded and attached to your plate automatically.</p>'+
					'<p>IceBear understands the following screen file formats:</p>'+
					'<ul style="padding-left:1.5em">'+
					'<li><strong>Two-column CSV:</strong> A simple file with well number and description. You can use this <a href="/resources/WellAndCondition.csv">Template CSV file</a>;</li>'+
					'<li><strong>Mimer:</strong> The output from Mimer*, saved as CSV (not Excel format);</li>'+
					'<li><strong>Rock Maker XML:</strong> A screen XML file in Rock Maker XML format.</li>'+
					'<li><strong>Tecan CrysScreen:</strong> The Microsoft Word document generated by the Tecan robot can be uploaded directly into IceBear; or</li>'+
					'<li><strong>Rigaku CrystalTrak:</strong> A plate or screen XML file generated by CrystalTrak. If IceBear is configured to import from your Rigaku imagers, these files should be detected automatically; you shouldn\'t need to upload them here.</li>'+
					'</ul>'+
					'<p>You can upload other files, but IceBear won\'t understand them. Instead, the file will be attached and each condition will read "See screen definition file."</p>'+
					'<hr style="clear:both;margin-top:1.5em;margin-bottom:1.5em"/>'+
					'<p><span style="font-size:80%">* Mimer: an automated spreadsheet-based crystallization screening system. Brodersen DE, Andersen GR, Andersen CB<br/>'+
						'<span style="color:transparent">* </span>Acta Cryst F, 2013 (69) pp815-820. <a target="_blank" href="https://www.ncbi.nlm.nih.gov/pubmed/?term=PMID%3A+23832216">PubMed PMID:23832216</a></span></p>'+
					'</div>';
				}
			} else {
				new Ajax.Request('/api/screen/'+data.screenid, {
					method:'get',
					onSuccess:function(transport){
						if(!AjaxUtils.checkResponse(transport)){ return false; }
						new Ajax.Request(tab.dataset.apiurl,{
							method:'get',
							onSuccess:function(transport){
								if(!AjaxUtils.checkResponse(transport)){ return false; }
								let screen=transport.responseJSON.screen;
								data.screen=screen;
								data.screen.conditions=transport.responseJSON.rows;
								let contentBefore='<div style="text-align:right">'+
									'<span style="float:left"><a href="/screen/'+screen.id+'">'+screen.name+'</a>';
									if(1*screen["isstandard"]){
										contentBefore+=' (Standard screen)';
									} else {
										contentBefore+=' (Optimization screen)';
									}
									contentBefore+='</span>';
									if(canEdit){
										contentBefore+='<input type="button" value="Remove screen from plate" onclick="Plate.unsetScreen()" />';
									}
									contentBefore+='</div>';
								let headers=['Well number','Description'];
								let cellTemplates=['{{wellnumber}}','{{description}}'];
								$("screens_body").table({
									'headers':headers,
									'cellTemplates':cellTemplates,
									'contentBefore':contentBefore
								}, transport.responseJSON);
							},
							onFailure:AjaxUtils.checkResponse
						});
					}, 
					onFailure:AjaxUtils.checkResponse
				});			
			}
			window.setTimeout(function(){
				let f=$("files");
				if(f){
					f.refresh();
				}
			},100);
		},
	  	 
		beginSetScreen: function(handler){
			new Ajax.Request('/api/screen/isstandard/1?pagesize=25',{
				method:'get',
				onSuccess:function(transport){
					if(!AjaxUtils.checkResponse(transport)){ return false; }
					Plate.showScreenList(transport.responseJSON, handler);
				},
				onFailure:function(transport){
					AjaxUtils.checkResponse(transport);
				},
			});

		},
		showScreenList: function(response, handler){
			let mb=ui.modalBox({
				'title':'Choose the screen',
				'content':'Just a moment...'
			});
			Plate.doAfter=handler;
			mb.dataset.pagesize="25";
			mb.table({
				'headers':['Name','Manufacturer',''],
				'cellTemplates':['{{name}}','{{manufacturer}}','<input type="button" value="Choose" onclick="Plate.chooseStandardScreen(this)">']
			}, response);		
		},
		chooseStandardScreen: function(btn){
			btn.up("tr").classList.add("updating");
			let screen=btn.up("tr").rowData;
			Plate.setScreen(screen.id);
		},
		uploadScreen: function(){
			let frm=$("optimizationscreenform");
			frm.rows.value=data.rows;
			frm.cols.value=data.cols;
			if(0===frm.file.files.length){ alert('Choose a screen description file to upload'); return false; }
			frm.file.options={ afterSuccess:Plate.setUploadedScreen };
			ui.submitForm(frm.file);
		},
		setUploadedScreen: function(responseJSON){
			let screen=responseJSON.created;
			Plate.setScreen(screen.id);
		},
		
		setScreen: function(screenId){
			new Ajax.Request('/api/plate/'+data.id,{
				method:'patch',
				parameters:{
					csrfToken:csrfToken,
					screenid:screenId
				},
				onSuccess:function(transport){
					if(!AjaxUtils.checkResponse(transport)){ return false; }
					Plate.afterSetScreen(transport);
				},
				onFailure:function(transport){
					AjaxUtils.checkResponse(transport);
				},
			});
		},
		afterSetScreen: function(transport){
			data=transport.responseJSON["updated"];
			$("files").relatedObjectIds=data["screenid"];
			ui.closeModalBox();
			window.setTimeout(function(){
				Plate.renderScreenTab();
				$("files_body").refresh();
			}, 50);
		},

		unsetScreen: function(){
			let msg;
			let isStandard=1*data.screen["isstandard"];
			if(isStandard){ 
				msg='Unset standard screen "'+data.screen.name+'"?\n\nThis will not delete the screen.'
			} else {
				//split because "Delete ... from" looks like SQL to IDE
				msg="Delete the optimization screen ";
				msg+="from this plate?\n\nThe screen and its definition files will be deleted."
			}
			if(!confirm(msg)){ return false; }
			new Ajax.Request('/api/plate/'+data.id,{
				method:'patch',
				parameters:{
					csrfToken:csrfToken,
					screenid:"NULL"
				},
				onSuccess:function(transport){
					if(!AjaxUtils.checkResponse(transport)){ return false; }
					Plate.afterUnsetScreen(transport);
				},
				onFailure:function(transport){
					AjaxUtils.checkResponse(transport);
				},
			});
		},
		afterUnsetScreen: function(){
			let isStandard=1*data.screen["isstandard"];
			$("files").relatedObjectIds=null;
			if(!isStandard){
				new Ajax.Request('/api/screen/'+data.screen.id,{
					method:'delete',
					parameters:{
						csrfToken:csrfToken,
					},
					onFailure:function(transport){
						AjaxUtils.checkResponse(transport);
					},
				});
			}
			data.screenid="";
			data.screen=null;
			window.setTimeout(function(){
				Plate.renderScreenTab();
				$("files_body").refresh();
			}, 50);
		},

	 	/*
	 	 * DROP CONDITIONS tab
	 	 */
		renderDropConditionsTab:function(){
			/*
			 * First implementation, assumes same across entire plate
			 */
			let db=$("drops_body");
			if(!data.plateType || !data.welldrops){
				db.innerHTML="Just a moment...";
				window.setTimeout(Plate.renderDropConditionsTab,100);
				return;
			}
			let units=['uL','nL'];
			let unitsOptions='';
			units.forEach(function(u){
				unitsOptions+='<option value="'+u+'">'+u.replace("u","&#181;")+'</option>';
			});

			db.innerHTML="";
			let dropMap=PlateType.getDropMapElement(data.plateType, db);
			dropMap.style.height="100%";
			dropMap.style.width=dropMap.getHeight()+"px";
			let drops=dropMap.select(".dropmapwell");
			drops.each(function(drop){
				drop.style.textAlign="center";
				let dropNumber=parseInt(drop.dataset.dropnumber);
				let proteinAmount="";
				let proteinUnit="uL";
				let wellAmount="";
				let wellUnit="uL";

				for(let i=0;i<data.welldrops.length;i++){
					let wd=data.welldrops[i];
					if(parseInt(wd.dropnumber)===dropNumber){
						proteinAmount=wd["proteinsolutionamount"];
						proteinUnit=wd["proteinsolutionunit"];
						wellAmount=wd["wellsolutionamount"];
						wellUnit=wd["wellsolutionunit"];
						break;
					}
				}
				
				if(canEdit){
					let out="<h3>Well solution:</h3>";
					out+='<div style="display:inline-block;width:6.25em;border:1px solid #666;border-radius:0.25em">';
					out+='<input type="text" style="width:2em;height:1.5em" name="wellsolutionamount" value="'+wellAmount+'" />';
					out+='<select style="border:none;width:3em;height:1.8em" name="wellsolutionunit">'+unitsOptions.replace('"'+wellUnit+'"','"'+wellUnit+'" selected="selected"')+'</select>';
					out+='</div>';
					out+='<hr style="margin:0.5em"/>';
					out+='<h3>Protein solution:</h3>';
					out+='<div style="display:inline-block;width:6.25em;border:1px solid #666;border-radius:0.25em">';
					out+='<input type="text" style="width:2em;height:1.5em" name="proteinsolutionamount" value="'+proteinAmount+'" />';
					out+='<select style="border:none;width:3em;height:1.8em" name="proteinsolutionunit">'+unitsOptions.replace('"'+proteinUnit+'"','"'+proteinUnit+'" selected="selected"')+'</select>';
					out+='</div>';
					drop.innerHTML+=out;
					drop.select("input,select").each(function(elem){
						elem.observe("keyup",Plate.updateWellSolutions);
						elem.observe("change",Plate.updateWellSolutions);
						elem.observe("blur",Plate.updateWellSolutions);
					});
				} else {
					drop.innerHTML+='<h3>Protein solution:</h3>';
					let proteinContent=proteinAmount+proteinUnit.replace("u","&#181;");
					if(""===proteinAmount || ""===proteinUnit){ proteinContent="(Not set)"; }
					drop.innerHTML+=proteinContent;
					drop.innerHTML+='<hr style="margin:0.5em"/>';
					drop.innerHTML+="<h3>Well solution:</h3>";
					let wellContent=wellAmount+wellUnit.replace("u","&#181;");
					if(""===wellAmount || ""===wellUnit){ wellContent="(Not set)"; }
					drop.innerHTML+=wellContent;
				}
			});
			Plate.setDropConditionsTabWarningStatus();
		},

		/**
		 * Historically, saving in the "Drop Conditions" tab has been unreliable. This is believed to have been fixed,
		 * but this function reinforces the existing believed-good save-on-change mechanism with a periodic check for
		 * fields that should have saved and haven't been. Belt-and-braces approach.
		 */
		periodicallyUpdateWellSolutions:function(){
			window.setTimeout(Plate.periodicallyUpdateWellSolutions,100);
			if(!window["updateTimer"]) {
				//Not waiting to update a field
				let taintedFields = document.querySelectorAll("#drops_body input[data-tainted]");
				taintedFields.forEach(function (field) {
					if (!field.dataset.forceUpdated) {
						field.dataset.forceUpdated = "1";
						Plate.doUpdateWellSolutions(field);
					}
				});
			}
		},

		updateWellSolutions:function(evt){
			window.clearTimeout(submitTimer);
			let field=evt.target;
			field.dataset.tainted="1";
			let drop=field.up("td");
			if(field.up("label,td")){ field.up("label,td").classList.remove("invalidfield"); }
			drop.select("input,select").each(function(elem){
				if(!validator.validate(elem)){
					drop.classList.add("invalidfield");
					return false;
				}
			});
			let delay=1500;
			if("blur"===evt.type){ delay=50; }
			window.submitTimer=setTimeout(function(){ Plate.doUpdateWellSolutions(field) },delay);
		},
		doUpdateWellSolutions:function(field){
			/*
			 * First implementation, assumes solutions are same for all drops in same position for whole plate
			 */
			let drop=field.up("td");
			if(drop.hasClassName("updating")){
				return false;
			}
			drop.classList.remove("invalidfield");
			drop.classList.add("updating");
			let parameters={ csrfToken:csrfToken};
			parameters[field.name]=field.value;
			parameters['selectionDrop']=drop.dataset.dropnumber;
			new Ajax.Request("/api/plate/"+data.id,{
				method:"patch",
				parameters:parameters,
				onSuccess:function(transport){
					field.dataset.tainted=null;
					field.dataset.forceUpdated=null;
					Plate.updateWellSolutions_onSuccess(transport);
				},
				onFailure:function(transport){
					field.dataset.tainted=null;
					field.dataset.forceUpdated=null;
					Plate.updateWellSolutions_onFailure(transport);
				}
			});
		},
		updateWellSolutions_onSuccess:function(transport){
			/*
			 * First implementation, assumes solutions are same for all drops in same position for whole plate
			 */
			if(!AjaxUtils.checkResponse(transport)){ return false; }
			let updatedDrops=transport.responseJSON["updated"]["updateddrops"];
			if(!updatedDrops || 0===updatedDrops.length){ return false; }
			let firstUpdatedDrop = updatedDrops[0];
			let dropCells=$("drops_body").select(".dropmapwell");
			dropCells.each(function(dropCell){
				if(parseInt(dropCell.dataset.dropnumber)===parseInt(firstUpdatedDrop.dropnumber)){
					['proteinsolutionamount','proteinsolutionunit','wellsolutionamount','wellsolutionunit'].each(function(field){
						dropCell.down('[name='+field+']').value=firstUpdatedDrop[field];
					});
					dropCell.classList.remove("updating");
				}
			});
			Plate.setDropConditionsTabWarningStatus();
		},
		updateWellSolutions_onFailure:function(transport){
			AjaxUtils.checkResponse(transport);
		},

		setDropConditionsTabWarningStatus:function (){
			let tab=document.getElementById("drops_body");
			if(!tab){ return false; }
			let inputs=tab.querySelectorAll("input[type=text]");
			tab.unwarn();
			inputs.forEach(function (inp){
				if(""===inp.value.trim()){
					tab.warn("Conditions not set for all drop positions");
				}
			});
		},
		
		getPlateType:function(){
			new Ajax.Request('/api/platetype/'+data["platetypeid"],{
				method:"get",
				onSuccess:Plate.getPlateType_onSuccess,
				onFailure:Plate.getPlateType_onFailure,
			});
		},
		getPlateType_onSuccess:function(transport){
			if(!AjaxUtils.checkResponse(transport)){ return false; }
			data.plateType=transport.responseJSON;
		},
		getPlateType_onFailure:function(transport){
			AjaxUtils.checkResponse(transport);
		},

		
		/*
		 * INSPECTIONS tab of plate view
		 */
		writeInspectionLink: function(obj,field){
			return ImagingSession.getLink(obj,field);
		},
		
		/*
		 * Best score in, e.g., Project's plates tab
		 */
		renderPlateBestScoreCell:function(item){
			if(""===item["bestscorelabel"]){ return '-'; }
			return '<span style="float:right;width:20px;background-color:#'+item["bestscorecolor"]+'">&nbsp;</span>'+item["bestscorelabel"];
		},
		
};



let PlateType={
		
		getDropMapElement:function(plateType, parent){
			let t=document.createElement("table");
			t.style.position="relative";
			t.style.top="0";
			t.className="dropmap";
			let dropMap=plateType["dropmapping"].split(",");
			let rows=dropMap.length;
			let rowHeightPercent=100/rows;
			dropMap.each(function(row){
				let tr=document.createElement("tr");
				let cols=row.length;
				let colWidthPercent=100/cols;
				for(let c=0;c<row.length;c++){
					let td=document.createElement("td");
					td.style.height=rowHeightPercent+"%";
					td.style.maxHeight=rowHeightPercent+"%";
					td.style.width=colWidthPercent+"%";
					td.style.maxWidth=colWidthPercent+"%";
					td.style.overflowY="auto";
					let content=row[c];
					if(parseInt(content)){ 
						td.className="dropmapwell";
						td.dataset.dropnumber=content;
						let lbl=document.createElement("div");
						lbl.className="welllabel";
						lbl.innerHTML=content;
						td.appendChild(lbl);
					} else if("R"===content){
						//Reservoir
						td.className="dropmapreservoir";
					} else if("E"===content){
						//Empty well
						td.className="dropmapempty";
					} else if("X"===content){
						//Dead space on the plate
						td.className="dropmapdead";
					}
					tr.appendChild(td);
				}
				t.appendChild(tr);
			});
			if(null!=parent){
				parent.appendChild(t);
				window.setTimeout(function(){
					t.style.width=t.getHeight()+"px";
				},50);
			}
			return t;
		}
		
};


let ImagingSession={
	
		getLink: function(obj,field){
			let lighttype=obj["lighttype"];
			let label=obj[field];
			if('imageddatetime'===field){ label=ui.friendlyDate(label); }
			return ImagingSession.getLightPathIcon(lighttype)+'&nbsp;<a href="/imagingsession/{{id}}">'+label+'</a>';
		},
	
		getLightPathIcon: function(lightTypeName){
			let types=['Visible','UV'];
			let offset=60*Math.max(0, types.indexOf(lightTypeName));
			return '<span class="lightbulb" style="background-position:0 -'+offset+'px" title="'+lightTypeName+'">&nbsp;</span>';
		}
			
};


let CrystallizationScreen={ //don't call it Screen - almost guaranteed to conflict with some UI thing or other

		getConditionCell:function(obj,field){
			if("1"===data["isstandard"] && canEdit){
				return '<form action="/api/screencondition/'+obj.id+'/" method="patch"><input type="text" name="description" value="'+ obj["description"] +'" /></form>';
			}
			return obj[field];
		},
		
		adjustConditionsTableForEdit:function(){
			let cbt=$("conditions_body_table");
			if(!cbt){
				window.setTimeout(CrystallizationScreen.adjustConditionsTableForEdit,50);
				return;
			}
			if(cbt.adjustedForEdit){
				return;
			}
			let cells=$$('#conditions_body_table td+td');
			cells.each(function(c){
				c.style.width="85%";
				let inp=c.down("input");
				if(inp){
					inp.style.width="99%";
					inp.observe("keyup", function(){ ui.updateFormField(inp)});
					inp.dataset.oldvalue=inp.value;
				}
			});
			cbt.adjustedForEdit=true;
		},

};

let Container = {

	create: function (name, containerTypeId, successCallback) {
		if ("" === name.trim() || "" === containerTypeId.trim()) {
			alert("Name and container type ID are required");
			return false;
		} else if (!parseInt(containerTypeId)) {
			alert("Container type ID must be a number");
			return false;
		}
		let parameters = {
			'csrfToken': csrfToken,
			'name': name,
			'containertypeid': containerTypeId
		};
		new Ajax.Request('/api/container', {
			method: 'post',
			parameters: parameters,
			onSuccess: function (transport) {
				if (!transport.responseJSON || !transport.responseJSON.created) {
					return AjaxUtils.checkResponse(transport);
				}
				if (successCallback) {
					successCallback(transport.responseJSON.created);
				}
			},
			onFailure: function (transport) {
				AjaxUtils.checkResponse(transport);
			},
		});

	},

	addChildById: function (childId, parentId, position, shipmentId, successCallback) {
		let parameters = {
			'csrfToken': csrfToken,
			'parent': parentId,
			'child': childId,
			'position': position,
		};
		if (null != shipmentId) {
			parameters['shipmentid'] = shipmentId;
		}
		new Ajax.Request('/api/containercontent', {
			method: 'post',
			parameters: parameters,
			onSuccess: function (transport) {
				if (!transport.responseJSON || !transport.responseJSON.created) {
					return AjaxUtils.checkResponse(transport);
				}
				if (successCallback) {
					successCallback(transport.responseJSON.created);
				}
			},
			onFailure: function (transport) {
				AjaxUtils.checkResponse(transport);
			},
		});
	},

	/**
	 * Looks for a container with name/barcode "name". If found, sets the found object as
	 * the value of the supplied placeholder. If not found, errors and sets placeholder to "Not found".
	 */
	getByName: function (name, successCallback) {
		new Ajax.Request('/api/container/name/' + name, {
			method: 'get',
			onSuccess: function (transport) {
				let found = transport.responseJSON;
				if (found.rows) {
					found = found.rows[0];
				}
				if (successCallback) {
					successCallback(found);
				}
			},
			onFailure: function (transport) {
				if (404 !== transport.status) {
					return AjaxUtils.checkResponse;
				}
				alert("No container with name " + name);
			},
		});
	},

	getAllContents: function (container, successCallback) {
		new Ajax.Request('/api/container/' + container.id + '/content?recursive=yes', {
			method: 'get',
			onSuccess: function (transport) {
				let found = transport.responseJSON;
				container.childitems = found;
				if (successCallback) {
					successCallback(found);
				}
			},
			onFailure: function (transport) {
				if (404 !== transport.status) {
					return AjaxUtils.checkResponse;
				}
				alert("Container is empty");
			},
		});
	},

	showAllContents: function (container, elem, showEmptyingControls) {
		let category = container.containercategoryname.toLowerCase();
		if (!elem) {
			elem = ui.modalBox({title: "Contents of " + category + " " + container.name});
		}
		if (!container.childitems) {
			alert("Cannot show container contents - not set");
			return false;
		}
		if ("dewar" === category) {
			Container.renderDewarContents(container, elem, showEmptyingControls);
		} else if ("puck" === category) {
			Container.renderPuckContents(container, elem, showEmptyingControls);
		} else if ("pin" === category) {
			Container.renderPinContents(container, elem, showEmptyingControls);
		}
	},

	renderDewar: function (dewar, parentElement, showEmptyingControls) {
		let elem = ui.treeItem({
			header: "Dewar: " + dewar.name
		});
		Container.renderDewarContents(dewar, elem.down(".treebody, .boxbody, .tabbody"), showEmptyingControls);
	},
	renderDewarContents: function (dewar, parentElement, showEmptyingControls) {
		let childitems = dewar.childitems;
		let isEmpty = true;
		for (let i = 1; i < childitems.length; i++) {
			if (childitems[i]) {
				isEmpty = false;
				let puckElement = Container.renderPuck(childitems[i], parentElement, showEmptyingControls);
				if (puckElement) {
					puckElement.up(".treeitem").dewar = dewar;
				}
			}
		}
		if (isEmpty) {
			parentElement.innerHTML = "Dewar is empty.";
		} else {
			let buttons = parentElement.select(".removechild");
			if (buttons) {
				buttons.each(function (btn) {
					btn.stopObserving("click");
					btn.observe("click", function (evt) {
						Container.removeFromParent(evt.target);
						evt.stopPropagation();
					});
				});
			}
		}
	},

	renderPuck: function (puck, parentElement, showEmptyingControls) {
		if (!puck.containercategoryname || "puck" !== puck.containercategoryname.toLowerCase()) {
			parentElement.treeItem({header: puck.name + " is not a puck"});
			return false;
		}
		let controls = '';
		if (showEmptyingControls) {
			controls = 'Remove puck and <input value="Keep all contents" type="button" class="removechild" data-containercategoryname="puck" data-recursive="0" />&nbsp;<input value="Wash all pins" type="button" class="removechild" data-containercategoryname="puck" data-recursive="1" />';
		}
		let elem = parentElement.treeItem({
			record: puck,
			header: "Puck: " + puck.name + '<span style="float:right;line-height:2.5em">' + controls + '</span>'
		}).down(".treebody");
		Container.renderPuckContents(puck, elem, showEmptyingControls);
		return elem;
	},
	renderPuckContents: function (puck, parentElement, showEmptyingControls) {
		let childitems = puck.childitems;
		let numChildItems = childitems.length;
		let fieldsToCopy = ['spacegroup', 'unitcella', 'unitcellb', 'unitcellc', 'unitcellalpha', 'unitcellbeta', 'unitcellgamma'];
		for (let i = 1; i < numChildItems; i++) {
			if (!childitems[i]) {
				childitems[i] = {
					position: i,
					name: '',
					rendername: '(empty)',
					samplename: '',
					spacegroup: '',
					unitcella: '',
					unitcellb: '',
					unitcellc: '',
					unitcellalpha: '',
					unitcellbeta: '',
					unitcellgamma: '',
				};
			}

			let pin = childitems[i];
			if (0 === pin.name.indexOf("dummypin")) {
				pin.rendername = "";
				pin.controls = 'Remove from puck and <input style="float:none" value="Wash pin" type="button" class="removechild" data-containercategoryname="pin" data-recursive="1" />';
			} else {
				pin.rendername = pin.name;
				pin.controls = 'Remove from puck and <input style="float:none" value="Keep crystal" type="button" class="removechild" data-containercategoryname="pin" data-recursive="0" />&nbsp;<input style="float:none" value="Wash pin" type="button"  class="removechild" data-containercategoryname="pin" data-recursive="1"/>';
			}
			pin.position = i;
			pin.isEmpty = 0;
			pin.puckid = puck.id;
			pin.puckname = puck.name;
			pin.puckpositions = puck.positions;
			if (childitems[i].childitems && childitems[i].childitems[1]) {
				let xtal = childitems[i].childitems[1];
				if (xtal.name) {
					pin.proteinacronym = xtal.proteinacronym ? xtal.proteinacronym : '';
					pin.samplename = xtal.name;
					pin.shippingcomment = xtal.shippingcomment ? xtal.shippingcomment : '';
					fieldsToCopy.forEach(function (f) {
						pin[f] = xtal[f];
					});
					pin.crystalid = xtal.id;
					pin.crystallocalurl = '/crystal/' + xtal.id;
					pin.crystalremoteurl = xtal.urlatremotefacility;
				} else {
					//User can't see contents
					if(!isShipper){
						pin.proteinacronym = '(hidden)';
						pin.samplename = '(hidden)';
						pin.shippingcomment = '(hidden)';
					} else {
						pin.proteinacronym = xtal.proteinacronym ? xtal.proteinacronym : '';
						pin.samplename = xtal.name;
						pin.shippingcomment = xtal.shippingcomment ? xtal.shippingcomment : '';
					}
				}
			} else {
				//position is empty
				pin.proteinacronym = '';
				pin.samplename = '';
				pin.shippingcomment = '';
				pin.controls = ''; //nothing to remove
				pin.isEmpty = 1;
			}
		}
		let headers = ['Slot', 'Pin', 'Sample name', 'Protein acronym', 'Spacegroup', 'a', 'b', 'c', '&alpha;', '&beta;', '&gamma;', 'Remarks'];
		let cellTemplates = ['{{position}}', '{{rendername}}',/*'{{samplename}}'*/[Container.getCrystalLinkForPin, 'samplename'], '{{proteinacronym}}',
			'{{spacegroup}}', '{{unitcella}}', '{{unitcellb}}', '{{unitcellc}}', '{{unitcellalpha}}', '{{unitcellbeta}}', '{{unitcellgamma}}',
			'{{shippingcomment}}'];
		if (showEmptyingControls) {
			headers = ['Slot', 'Pin', 'Sample name', 'Protein acronym', '&nbsp;'];
			cellTemplates = ['{{position}}', '{{rendername}}', '{{samplename}}', '{{proteinacronym}}', '{{controls}}'];
		}
		ui.table({
			headers: headers,
			cellTemplates: cellTemplates
		}, {rows: childitems/*.splice(1)*/}, parentElement);
		let buttons = parentElement.select(".removechild");
		if (buttons) {
			buttons.each(function (btn) {
				btn.stopObserving("click");
				btn.up("td").style.textAlign = "right";
				btn.observe("click", function (evt) {
					Container.removeFromParent(evt.target);
					evt.stopPropagation();
				});
			});
		}
	},

	getCrystalLinkForPin: function (pin) {
		if (!pin.crystallocalurl || "" === pin.crystallocalurl) {
			return pin.samplename;
		}
		return '<a href="' + pin.crystallocalurl + '">' + pin.samplename + '</a>';
	},

	/**
	 * Use this only for a loose pin. renderPuckContents will handle pins in a puck.
	 * @param pin An object representing the pin
	 * @param parentElement The element to render into
	 * @param showEmptyingControls if true, render a Remove button
	 */
	renderPinContents: function (pin, parentElement, showEmptyingControls) {
		let childitems = pin.childitems;
		let out = '';
		if (2 === childitems.length && !childitems[1]['name'] && !childitems[1]['crystalhidden']) {
			out = "Pin is empty.";
		}
		parentElement.pin = pin;
		for (let i = 1; i < childitems.length; i++) {
			if (childitems[i]) {
				let xtal = childitems[i];
				let xtalName = xtal["name"] || "(hidden)";
				let construct = xtal["constructname"] || "(hidden)";
				let protein = "(hidden)";
				if (xtal["proteinname"] && xtal["proteinacronym"]) {
					protein = xtal["proteinname"] + " (" + xtal["proteinacronym"] + ")";
				}
				out += "<h3>Crystal " + i + "</h3>";
				out += "Sample name: " + xtalName;
				out += "<br/>Protein: " + protein;
				out += "<br/>Construct: " + construct;
				out += '<br/><br/><span id="pincontent_datefished"></span><br/>';
				if (showEmptyingControls) {
					//Assumes only one crystal per pin.
					out += '<label><input value="Wash pin" type="button" class="removechild" data-recursive="1" data-containercontentid="' + xtal.containercontentid + '"/></label>';
				}
				new Ajax.Request("/api/baseobject/" + xtal['containercontentid'], {
					method: "get",
					onSuccess: function (transport) {
						$("pincontent_datefished").innerHTML = "Date fished: " + ui.friendlyDate(transport.responseJSON["createtime"]);
					},
					onFailure: function () {
					}
				});

			}
		}
		parentElement.innerHTML = out;
		let buttons = parentElement.select(".removechild");
		if (buttons) {
			buttons.each(function (btn) {
				btn.stopObserving("click");
				btn.up("label").style.textAlign = "left";
				btn.observe("click", function (evt) {
					Container.removeFromParent(evt.target);
					evt.stopPropagation();
				});
			});
		}
	},

	removeFromParent: function (removeButton) {
		if("puck"===removeButton.dataset.containercategoryname && !confirm("Really remove puck from dewar?")){
			return false;
		}
		let childItemId = 0;
		let recursive = 1 * removeButton.dataset.recursive;
		if (undefined !== removeButton.dataset.containercontentid) {
			childItemId = removeButton.dataset.containercontentid;
			if (removeButton.up("label")) {
				removeButton.up("label").classList.add("updating");
			}
		} else if (removeButton.up("tr.datarow")) {
			let tr = removeButton.up("tr.datarow");
			childItemId = tr.rowData.containercontentid;
			tr.classList.add("updating");
		} else if (removeButton.up("div.treeitem")) {
			let ti = removeButton.up("div.treeitem");
			childItemId = ti.record.containercontentid;
			ti.down("h3").classList.add("updating");
		}
		if (!childItemId) {
			return false;
		}
		let uri = '/api/containercontent/' + childItemId;
		let postBody = "csrfToken=" + csrfToken;
		if (recursive) {
			postBody += "&recursive=1";
		}
		new Ajax.Request(uri, {
			method: "delete",
			postBody: postBody,
			onSuccess: function (transport) {
				Container.removeFromParent_onSuccess(removeButton, transport);
			},
			onFailure: function (transport) {
				Container.removeFromParent_onFailure(removeButton, transport);
			},
		})
	},
	removeFromParent_onSuccess: function (removeButton) {
		removeButton.up(".updating").classList.remove("updating");
		if (removeButton.up("tr")) {
			removeButton.up("tr").select("td+td").each(function (td) {
				td.innerHTML = "";
			});
		} else if (removeButton.up(".treeitem")) {
			removeButton.up(".treeitem").remove();
		} else if (removeButton.up(".pincontent")) {
			removeButton.up(".pincontent").remove();
			ui.closeModalBox();
		}
	},
	removeFromParent_onFailure: function (removeButton, transport) {
		removeButton.up(".updating").classList.remove("updating");
		AjaxUtils.checkResponse(transport);
	},


};

let Protein = {

	codonAsAmino: {
		"ATT": "I", "ATC": "I", "ATA": "I", "CTT": "L", "CTC": "L", "CTA": "L", "CTG": "L", "TTA": "L",
		"TTG": "L", "GTT": "V", "GTC": "V", "GTA": "V", "GTG": "V", "TTT": "F", "TTC": "F", "ATG": "M",
		"TGT": "C", "TGC": "C", "GCT": "A", "GCC": "A", "GCA": "A", "GCG": "A", "GGT": "G", "GGC": "G",
		"GGA": "G", "GGG": "G", "CCT": "P", "CCC": "P", "CCA": "P", "CCG": "P", "ACT": "T", "ACC": "T",
		"ACA": "T", "ACG": "T", "TCT": "S", "TCC": "S", "TCA": "S", "TCG": "S", "AGT": "S", "AGC": "S",
		"TAT": "Y", "TAC": "Y", "TGG": "W", "CAA": "Q", "CAG": "Q", "AAT": "N", "AAC": "N", "CAT": "H",
		"CAC": "H", "GAA": "E", "GAG": "E", "GAT": "D", "GAC": "D", "AAA": "K", "AAG": "K", "CGT": "R",
		"CGC": "R", "CGA": "R", "CGG": "R", "AGA": "R", "AGG": "R", "TAA": "*", "TAG": "*", "TGA": "*",
	},

	dnaToProtein: function (dna) {
		dna = dna.replace(/\s/g, "");
		let protein = "";
		let dnaParts = dna.match(/.{1,3}/g);
		dnaParts.each(function (codon) {
			let amino = Protein.codonAsAmino[codon];
			if (!amino) {
				return false;
			}
			protein += amino;
		});
		return protein;
	},

	formatDnaSequence: function (seq) {
		if (!seq || "" === seq) {
			return false;
		}
		let seqParts = seq.match(/.{1,3}/g);
		seq = seqParts.join(" ").toUpperCase();
		return seq;
	},

	formatProteinSequence: function (seq) {
		if (!seq || "" === seq) {
			return false;
		}
		let seqParts = seq.match(/.{1,10}/g);
		seq = seqParts.join(" ").toUpperCase();
		return seq;
	},

};

let Shipment = {

	keepingAlive:false,

	cleanOldDewarTabs: function (currentContainers) {
		$$(".containertab").each(function (head) {
			let dewarName = head.dataset.containername;
			let inShipment = false;
			currentContainers.each(function (cont) {
				if (cont.name === dewarName) {
					inShipment = true;
				}
			});
			if (!inShipment) {
				head.next(".tabbody").remove();
				head.remove();
			}
		});
		if (!$("shiptabs").down("h2.current")) {
			let t = $("shipmentdetails");
			t.classList.add("current");
			window.document.location.hash = t.id;
		}
	},

	getShipmentDestination: function () {
		new Ajax.Request('/api/shipmentdestination/' + data["shipmentdestinationid"], {
			method: 'get',
			onSuccess: function (transport) {
				window.shipmentDestination = transport.responseJSON;
				Shipment.getShipmentSubmissionHandler();
			},
			onFailure: function (transport) {
				AjaxUtils.checkResponse(transport);
			},
		});
	},

	getShipmentSubmissionHandler: function () {
		if ("" === shipmentDestination["shipmenthandler"]) {
			return false;
		} //TODO default handler
		new Ajax.Request('/js/model/shipping/handlers/' + shipmentDestination["shipmenthandler"] + '.js?t=' + Date.now(), {
			method: 'get',
			onSuccess: function (transport) {
				//Not nice, but not much choice.
				eval(transport.responseText);
			},
			onFailure: function () {
				alert("Could not retrieve shipment submission handler for " + shipmentDestination['name'] + " (" + shipmentDestination["shipmenthandler"] + ")");
			},
		});
	},

	getAndRenderContainers: function () {
		new Ajax.Request('/api/shipment/' + data.id + '/container', {
			method: 'get',
			onSuccess: function (transport) {
				if (transport.responseJSON.rows) {
					//First remove any tabs for dewars that are NOT in the shipment
					Shipment.cleanOldDewarTabs(transport.responseJSON.rows);
					//Then render all dewars that ARE in the shipment, reusing any existing tabs
					transport.responseJSON.rows.each(function (cont) {
						if (!cont.containercategoryname) {
							Shipment.renderPlate(cont);
						} else if ('dewar' === cont.containercategoryname.toLowerCase()) {
							Shipment.renderDewar(cont);
						} else {
							alert(cont.name + " is directly inside the shipment but is not a plate or dewar.");
						}
					});
				}
			},
			onFailure: function (transport) {
				if (404 !== transport.status) {
					return AjaxUtils.checkResponse(transport);
				}
				$$(".containertab").each(function (head) {
					head.next(".tabbody").remove();
					head.remove();
				});
				Shipment.cleanOldDewarTabs();
			},
		});
	},
	renderDewar: function (dewar) {
		let t = $(dewar.name);
		if (!t) {
			t = $("shiptabs").tab({
				label: dewar.name,
				id: dewar.name,
				classes: 'containertab',
				content: ''
			});
		}
		t.dataset.containercontentid = dewar.containercontentid;
		t.dataset.containername = dewar.name;
		t.dataset.containerid = dewar.id;
		let tb = t.next();
		tb.innerHTML = "";
		Container.renderDewarContents(dewar, tb);
		if ("" !== data.dateshipped) {
			tb.insert({top: ui.infoMessageBar("This is the data sent to the synchrotron. Some information may have changed since then.")});
		} else if (userCanShip) {
			//TODO Remove Dewar control
			tb.insert({top: ui.infoMessageBar('<input type="button" class="noprint" title="Remove from shipment (keeping all contents)" style="cursor:pointer;float:none;margin-right:5em;" onclick="Shipment.removeContainer(this)" value="Remove dewar from shipment" /> Add a puck: <form class="noprint" style="display:inline" onsubmit="return Shipment.addPuckByBarcode(this)"><input class="noprint" type="text" name="barcode" style="width:12em" value="" placeholder="Scan barcode"/></form>')});
			tb.down(".msgbar").classList.add("noprint");
			tb.select(".treeitem").each(function (ti) {
				//Remove Puck control
				ti.down(".treehead").insert({bottom: '<input type="button" class="noprint" title="Remove from dewar (keeping all contents)" style="cursor:pointer;float:none;margin-left:5em;" onclick="Shipment.removeContainer(this)" value="Remove puck" /> '});
			});
			tb.select("tr.datarow").each(function (tr) {
				let barcodeCell = tr.down("td").next();
				if (tr.rowData.containercontentid) {
					//Insert "Remove Pin" control before barcode (or, if no barcode, loose)
					barcodeCell.insert({'top': '<input type="button" class="noprint" title="Remove from puck (keeping crystal in pin)" style="cursor:pointer;float:none" onclick="Shipment.removeContainer(this)" value="X" /> '});
				} else {
					//Nothing here. Add pin by scanning barcode.
					barcodeCell.innerHTML = '<form class="noprint" style="display:inline" onsubmit="return Shipment.addPinByBarcode(this)"><input class="noprint" style="width:8em" type="text" name="barcode" value="" placeholder="Scan barcode"/></form>';
				}
			});
		}

	},
	renderPlate: function (plate) {

	},

	addDewarByBarcode: function () {
		let b = $("addtoplevelbybarcode");
		let barcode = b.value.trim();
		if ("" === barcode) {
			return false;
		}
		let alreadyIn = false;
		let numContainers = 0;
		$$('.tab').each(function (t) {
			if (t.dataset.containername) {
				numContainers++;
				if (barcode === t.dataset.containername) {
					alreadyIn = true;
				}
			}
		});
		if (alreadyIn) {
			b.value = "";
			alert(barcode + " is already in this shipment");
			return false;
		}
		let bar = b.up("label");
		bar.classList.add("updating");
		let shipmentId = data.id;
		let position = numContainers + 1;
		Container.getByName(barcode, function (found) {
			if ("dewar" !== found.containercategoryname.toLowerCase()) {
				alert(barcode + " is not a dewar.");
				bar.classList.remove("updating");
				return false;
			}
			Container.addChildById(found.id, shipmentId, position, shipmentId, Shipment.getAndRenderContainers);
			bar.classList.remove("updating");
		});
		b.value = "";
		return false; //stop form submission and page reload
	},

	addPuckByBarcode: function (frm) {
		let barcode = frm.down("input").value.trim();
		if ("" === barcode) {
			alert("Puck barcode cannot be empty");
			return false;
		}
		let tb = frm.up(".tabbody");
		let bar = frm.up(".infobar");
		bar.classList.add("updating");
		let dewarId = tb.previous(".tab").dataset.containerid;
		let position = $$(".treeitem").length + 1;
		Container.getByName(barcode, function (found) {
			if ("puck" !== found.containercategoryname.toLowerCase()) {
				alert(barcode + " is not a puck.");
				bar.classList.remove("updating");
				return false;
			}
			Container.addChildById(found.id, dewarId, position, null, Shipment.getAndRenderContainers);
		});
		frm.down("input").value = "";
		return false; //stop form submission and page reload
	},

	addPinByBarcode: function (frm) {
		let field = frm.down("input");
		let barcode = field.value.trim();
		if ("" === barcode) {
			alert("Pin barcode cannot be empty");
			return false;
		}
		let tr = frm.up("tr");
		let puckId = frm.up(".treeitem").record.id;
		let position = tr.rowData.position;
		tr.classList.add("updating");
		Container.getByName(barcode, function (found) {
			if ("pin" !== found.containercategoryname.toLowerCase()) {
				alert(barcode + " is not a pin.");
				tr.classList.remove("updating");
				return false;
			}
			//Get the contents; will 404 if no crystal in pin.
			new Ajax.Request('/api/container/' + found.id + '/content', {
				method: 'get',
				onSuccess: function (transport) {
					if (!transport.responseJSON) {
						return AjaxUtils.checkResponse(transport);
					}
					Container.addChildById(found.id, puckId, position, null, Shipment.getAndRenderContainers);
				},
				onFailure: function (transport) {
					if (404 !== transport.status) {
						return AjaxUtils.checkResponse(transport);
					}
					alert(barcode + " has no crystal. Cannot add it to the shipment.");
					tr.classList.remove("updating");
					field.value = "";
				}
			});
		});
		return false; //stop form submission and page reload
	},

	removeContainer: function (btn) {
		let deletedTopLevel = false;
		let containerContentId = null;
		if (btn.up("tr")) {
			//Pin
			containerContentId = btn.up("tr").rowData.containercontentid;
			btn.up("tr").classList.add("updating");
		} else if (btn.up(".treeitem")) {
			//Puck
			containerContentId = btn.up(".treeitem").record.containercontentid;
			btn.up(".treeitem").classList.add("updating");
		} else if (btn.up(".tabbody")) {
			//Dewar
			//TODO Could also be a plate
			containerContentId = btn.up(".tabbody").previous("h2").dataset.containercontentid;
			btn.up("div.msgbar").classList.add("updating");
			deletedTopLevel = true;
		}
		if (!containerContentId) {
			alert("Cannot determine container type for removal.");
			return false;
		}
		new Ajax.Request('/api/containercontent/' + containerContentId, {
			method: 'delete',
			postBody: 'csrfToken=' + csrfToken,
			onSuccess: function (transport) {
				if (!transport.responseJSON) {
					return AjaxUtils.checkResponse(transport);
				}
				Shipment.getAndRenderContainers();
				if (deletedTopLevel) {
					window.setTimeout(Shipment.renumberTopLevelContainers, 50);
				}
			},
			onFailure: AjaxUtils.checkResponse
		});

	},

	/**
	 * After deleting a top level container, iterate through tabs and set the "position" number
	 * to consecutive values starting from 1. Otherwise, adding another may fail due to clashing
	 * position numbers. Strictly speaking, we don't care what position a dewar occupies within a
	 * shipment - but the server does.
	 */
	renumberTopLevelContainers: function () {
		let pos = 1;
		$$(".containertab").each(function (head) {
			new Ajax.Request('/api/containercontent/' + head.dataset.containercontentid, {
				method: 'patch',
				parameters: {
					'csrfToken': csrfToken,
					'position': pos
				},
			});
			pos++;
		});
	},

	validate: function (proteinAcronyms) {
		let errors = Shipment.getShipmentErrors(proteinAcronyms);
		if (!errors || 0 === errors.length) {
			return true;
		}
		alert("You must fix these errors before the shipment can be sent:\n\n* " + errors.join("\n* "));
		return false;
	},
	getShipmentErrors: function (proteinAcronyms) {
		let errors = [];
		let dewars = $$(".containertab");
		if (0 === dewars.length) {
			errors.push("Shipment must contain at least one dewar.");
		}
		dewars.each(function (d) {
			Shipment.unhighlightValidationFailure(d);
			let dewarName = d.dataset.containername;
			let pucks = d.next().select(".treeitem");
			if (0 === pucks.length) {
				errors.push("Dewar " + dewarName + " is empty. Remove it or add a puck.");
				Shipment.highlightValidationFailure(d);
			}
			pucks.each(function (p) {
				Shipment.unhighlightValidationFailure(p);
				let puckName = p.record.name;
				let slots = p.select("tr.datarow");
				let numPins = 0;
				slots.each(function (tr) {
					Shipment.unhighlightValidationFailure(tr);
					if (!tr.rowData.childitems || 2 < tr.rowData.childitems.length) {
						//position is empty
					} else {
						numPins++;
						if (1 !== 1*tr.rowData.childitems[1]["hasacronym"]) {
							errors.push("Dewar " + dewarName + ", puck " + puckName + ", position " + tr.rowData.position + ": Crystal has no protein acronym. Set the parent plate's protein.");
							Shipment.highlightValidationFailure(tr);
						} else if (proteinAcronyms) {
							let matched = false;
							proteinAcronyms.each(function (pa) {
								if (pa === tr.rowData.childitems[1].proteinacronym) {
									matched = true;
								}
							});
							if (!matched) {
								errors.push("Dewar " + dewarName + ", puck " + puckName + ", position " + tr.rowData.position + ": Crystal has a protein acronym (" + tr.rowData.childitems[1].proteinacronym + ") that is not in the approved list.");
								Shipment.highlightValidationFailure(tr);
							}
						}
					}
				});
				if (0 === numPins) {
					errors.push("Puck " + puckName + " in dewar " + dewarName + " is empty. Remove it or add a pin.");
					Shipment.highlightValidationFailure(p);
				}
			});
		});
		return errors;
	},

	highlightValidationFailure: function (elem) {
		elem.classList.add("haserror");
		//find a parent puck and highlight it
		if (elem.up(".treeitem")) {
			elem.up(".treeitem").down(".treehead").classList.add("haserror");
		}
		//find a parent dewar and highlight it
		if (elem.up(".tabbody")) {
			elem.up(".tabbody").previous("h2").classList.add("haserror");
		}
	},

	unhighlightValidationFailure: function (elem) {
		elem.classList.remove("haserror");
		//Don't do parent elements. We call this on the parent then validate its
		//children, so a good child would unhighlight its parent after a bad sibling.
	},

	/**
	 * Submits the shipment contents to the synchrotron
	 */
	send: function () {
		if ("" !== data.dateshipped) {
			alert("Shipment has already shipped");
			return false;
		}
		if (!Shipment.validate()) {
			return false;
		}
		if ("" !== shipmentDestination["shipmenthandler"] && window[shipmentDestination["shipmenthandler"]]) {
			window[shipmentDestination["shipmenthandler"]].begin();
		} else {
			alert("No shipment submission handler defined for " + window.shipmentDestination.name);
		}
	},

	markReturned: function (evt) {
		if (!data["objecttype"] || "shipment" !== data["objecttype"]) {
			//Not on a shipment page!
			return false;
		}
		let btn = evt.target;
		if (!confirm("Really mark this shipment as returned?")) {
			return false;
		}
		btn.up("label").classList.add("updating");
		new Ajax.Request("/api/shipment/" + data.id, {
			method: "patch",
			parameters: {
				csrfToken: csrfToken,
				datereturned: (new Date().toISOString().split("T"))[0]
			},
			onSuccess: function () {
				ui.forceReload();
			},
			onFailure: function (transport) {
				btn.up("label").classList.remove("updating");
				AjaxUtils.checkResponse(transport);
			}
		});
	},

	hasDatasetRetrieval:function(){
		return !!window[shipmentDestination["shipmenthandler"]] && !!window[shipmentDestination["shipmenthandler"]].DatasetRetrieval;
	},

	getCollectedDatasets: function () {
		if ("" === data.dateshipped) {
			alert("Shipment has not shipped");
			return false;
		}
		if(Shipment.hasDatasetRetrieval()){
			window[shipmentDestination["shipmenthandler"]].DatasetRetrieval.begin();
		} else {
			if ("" === shipmentDestination["shipmenthandler"] || !window[shipmentDestination["shipmenthandler"]]) {
				alert("No shipment submission handler defined for " + window.shipmentDestination.name);
			} else {
				alert("No dataset retrieval handler defined for " + window.shipmentDestination.name);
			}
		}
	},



	/**
	 * Handles the recording of information during the synchrotron data collection process.
	 */
	DataCollection:{

		quickNotes:[
			{
				"label":"Ice rings",
				"note":"Ice rings"
			},
			{
				"label":"Empty pin",
				"note":"Pin was empty"
			},
			{
				"label":"Not visible",
				"note":"Crystal was not visible"
			},
		],

		renderTab: function () {
			let tb=$("shiptabs");
			if (!tb || !data["manifest"]) {
				return;
			}
			let t = tb.tab({
				'id': 'shipment_datacollection',
				'label': 'Data collection'
			});
			t.observe("click", function () {
				window.setTimeout(Shipment.DataCollection.setHeightsInTab, 50);
			});
			t = t.next(); //body, not the header
			let puckDiv = document.createElement("div");
			puckDiv.id = "dcpucks";
			puckDiv.setStyle({position: "absolute", top: 0, left: 0, right: 0});
			puckDiv.innerHTML = "";
			let pinDiv = document.createElement("div");
			pinDiv.id = "dcpins";
			pinDiv.setStyle({position: "absolute", bottom: 0, left: 0, right: 0});

			data["manifest"].rows.each(function (dewar) {
				if (!dewar.childitems) {
					return;
				}
				dewar.childitems.each(function (puck) {
					if (!puck || !puck.childitems) {
						return;
					}
					let puckButton = document.createElement("div");
					puckButton.classList.add("ship_containertype");
					puckButton.id = "dcpuck" + puck.id;
					let puckName = document.createElement("div");
					puckName.addClassName("ship_containertypename ship_puck");
					puckName.innerHTML = puck.name;
					puckButton.appendChild(puckName);
					puckButton.style.cursor = "pointer";
					puckButton.pins=puck.childitems;
					puckDiv.appendChild(puckButton);
				});
			});
			puckDiv.innerHTML += '<hr style="clear:both;margin:0 1em"/>';
			t.appendChild(puckDiv);
			t.appendChild(pinDiv);
			data["manifest"].rows.each(function (dewar) {
				if (!dewar.childitems) {
					return;
				}
				dewar.childitems.each(function (puck) {
					if (!puck || !puck.childitems) {
						return;
					}
					let puckButton = $("dcpuck" + puck.id);
					puckButton.pins = puck.childitems;
					puckButton.pins.each(function(pin){
						if(!pin.childitems){ return; }
						let diffractionRequestId=pin.childitems[1].diffractionrequestid;
						pin.actionupdated=false;
						new Ajax.Request("/api/diffractionrequest/"+diffractionRequestId,{
							method:"get",
							onFailure:function(){
								pin.actionupdated=true;
							},
							onSuccess:function(transport){
								pin.childitems[1]["actiononreturn"]=transport.responseJSON.actiononreturn;
								pin.actionupdated=true;
							}
						});
					});
				});
			});
			window.setTimeout(function () {
				Shipment.DataCollection.setHeightsInTab();
				puckDiv.select(".ship_containertype").each(function (p) {
					p.onclick = Shipment.DataCollection.renderPuck;
					window.setTimeout(function(){ Shipment.DataCollection.updatePuckCompletedStatus(p) },1000);
				});
			}, 10);
		},

		setHeightsInTab: function () {
			$("dcpins").style.top = $("dcpucks").getHeight() + "px";
		},

		renderPuck: function (evt) {
			if(!Shipment.keepingAlive){
				ui.keepAlive();
				Shipment.keepingAlive=true;
			}
			let puckButton = evt.target;
			if (puckButton.up(".ship_containertype")) {
				puckButton = puckButton.up(".ship_containertype");
			}
			$("dcpucks").select(".ship_containertype").each(function (p) {
				p.classList.remove("dccurrentpuck");
			});
			puckButton.classList.add("dccurrentpuck");

			let dcPins=$("dcpins");
			dcPins.innerHTML = "";
			let ts = ui.tabSet({}, dcPins);
			ts.style.left = "1em";
			ts.style.right = "1em";
			ts.style.top = "1em";
			ts.style.bottom = ".5em";
			for (let i = 1; i < puckButton.pins.length; i++) {
				let pin = puckButton.pins[i];
				let content = "Puck position is empty.";
				if (!pin.isEmpty) {
					content = "Puck " + pin.puckname + " position " + i;
				}
				let t = ts.tab({
					label: "Pin " + i,
					content: content,
					disabled: pin.isEmpty
				});
				if(pin.childitems && ""!==pin.childitems[1].actiononreturn){
					t.classList.add("complete");
				}
				t.pin = pin;
			}
			ts.select(".enabledtab").each(function (t) {
				Shipment.DataCollection.renderPin(t.next());
			});
			dcPins.down("h2.enabledtab").click();
		},

		renderPin: function (tb) { //gets tab body
			let t = tb.previous(); //gets tab header
			tb.innerHTML = '';
			let pin = t.pin;
			let f = tb.form("#", {});

			let pos = f.formField({
				'label': pin.position,
				'content': ''
			});
			let lbl = '<a target="_blank" href="' + pin.crystallocalurl + '">View crystal in IceBear</a>';
			if ("" !== pin.childitems[1].crystalurlatremotefacility) {
				lbl += '<br/>' +
					'<a target="_blank" href="' + pin.childitems[1].crystalurlatremotefacility + '">View crystal at ' + shipmentDestination.name + '</a>';
			}

			let det = f.formField({
				'label': lbl,
				'content': pin.samplename + ' (Protein: ' + pin.proteinacronym + ')<br/>Puck ' + pin.puckname + ' position ' + pin.position
			});
			pos.title = "Position in puck";
			pos.classList.add("pinnumber");
			det.classList.add("xtaldetails");

			if (!isAdmin && !isShipper && -1 === userUpdateProjects.indexOf(parseInt(pin.childitems[1].projectid))) {
				Shipment.DataCollection._renderPinControlsReadOnly(pin, f);
			} else {
				Shipment.DataCollection._renderPinControls(pin, f);
			}

		},

		_renderPinControlsReadOnly: function (pin, frm) {
			frm.formField({
				label: "Not your crystal",
				content: "You do not have permission to add notes to this crystal, or to decide whether to keep it."
			});
			let btn = frm.buttonField({
				label: "Go to next pin",
			});
			btn.id = "pin" + pin.position + "save";
			let tb = frm.up(".tabbody");
			btn = btn.down("input");
			if (!tb.next(".enabledtab")) {
				btn.value = "Done";
			}
			btn.onclick = function () {
				Shipment.DataCollection.advanceToNextPin(btn);
			};

		},

		_renderPinControls: function (pin, frm) {
			let isBarcodedPin = true;
			if (!pin.name || 0 === pin.name.indexOf("dummypin")) {
				isBarcodedPin = false;
			}
			//Notes box, with any previous note (since this page was loaded)
			let notes = frm.textArea({
				'id':'dcnote'+pin.position,
				'label':'Data collection notes'
			});
			notes.down("textarea").rows=5;
			notes.down("textarea").insert({"before":'<div id="dcnote'+pin.position+'_buttons" style="text-align:right">Quick notes:</div>'});
			Note.writeQuickNoteButtons(
				$("dcnote"+pin.position+"_buttons"),
				"dcnote"+pin.position,
				Shipment.DataCollection.quickNotes,
				"margin:0.5em"
			);

			if (pin.dcnote && "" !== pin.dcnote) {
				notes.down("textarea").insert({"before": pin.dcnote + "<br/><br/>"});
			}
			if (isBarcodedPin) {
				let defaultAction = Crystal.WASH;
				let pinAction=pin.childitems[1].actiononreturn;
				if (pinAction && "" !== pinAction) {
					defaultAction = pinAction;
				}
				frm.radioButtons({
					name: 'pin' + pin.position + 'action',
					cssClasses: 'pinaction',
					label: 'When the shipment is returned...',
					defaultValue: defaultAction,
					options: [
						{
							'value': Crystal.WASH,
							'label': '<span style="font-weight:bold;color:#600">Wash the pin.</span> I don\'t want to keep the crystal.'
						},
						{
							'value': Crystal.KEEP,
							'label': '<span style="font-weight:bold;color:#060">Keep the crystal</span> in the pin. I want to send it again.'
						}
					]
				})
			} else {
				frm.formField({
					name: "pin" + pin.position + "action",
					label: "When the shipment is returned...",
					content: "Pin has no barcode, so it will be washed on return."
				});
			}
			let save = frm.buttonField({
				label: "Save",
			});
			save.down("input").onclick = Shipment.DataCollection.savePin;
			save.id = "pin" + pin.position + "save";
			let tb = frm.up(".tabbody");
			if (tb.next(".enabledtab")) {
				save.down("input").insert({"after": " and go to next"});
			} else {
				save.down("input").insert({"after": " and finish this puck"});
			}
		},

		savePin: function (evt) {
			let clicked = evt.target;
			let lbl = clicked.up("label");
			let f = clicked.up("form");
			let tb = f.up(".tabbody");
			let t = tb.previous();
			let note = f.down("textarea").value;
			let returnAction = "wash"; //default for non-barcoded pins
			if (f.down("label.selected")) {
				//barcoded pins have options
				returnAction = f.down("label.selected").down("input").value;
			}
			let diffractionRequestId = t.pin.childitems[1]["diffractionrequestid"];
			lbl.classList.add("updating");

			//save note
			if ("" !== note.trim()) {
				new Ajax.Request("/api/note", {
					method: "post",
					parameters: {
						"csrfToken": csrfToken,
						"parentid": t.pin.crystalid,
						"text": note.trim()
					},
					onSuccess: function () {
						clicked.noteSaved = true;
					},
					onFailure: function (transport) {
						AjaxUtils.checkResponse(transport);
					},
				});
			} else {
				//set success flag, no note to add.
				clicked.noteSaved = true;
			}

			//save return action
			new Ajax.Request("/api/diffractionrequest/" + diffractionRequestId, {
				method: "patch",
				parameters: {
					"csrfToken": csrfToken,
					"actiononreturn": returnAction
				},
				onSuccess: function () {
					clicked.actionSaved = true;
				},
				onFailure: function (transport) {
					AjaxUtils.checkResponse(transport);
				},
			});
			clicked.dcnote = note.trim();
			clicked.actiononreturn = returnAction;
			window.setTimeout(function () {
				Shipment.DataCollection.checkPinSaved(clicked);
			}, 100);
		},

		checkPinSaved: function (saveButton) {
			if (!saveButton || !saveButton.actionSaved) {
				window.setTimeout(function () {
					Shipment.DataCollection.checkPinSaved(saveButton);
				}, 100);
				return false;
			}
			let t = saveButton.up(".tabbody").previous();
			t.classList.add("complete");
			let puckDiv = $("dcpuck" + t.pin.puckid);
			puckDiv["pins"][t.pin.position].dcnote = saveButton.dcnote;
			puckDiv["pins"][t.pin.position].childitems[1]["actiononreturn"] = saveButton.actiononreturn;
			Shipment.DataCollection.advanceToNextPin(saveButton);
		},

		advanceToNextPin: function (saveButton) {
			let nextPinHeader = saveButton.up(".tabbody").next(".enabledtab");
			if (nextPinHeader) {
				saveButton.noteSaved = false;
				saveButton.actionSaved = false;
				nextPinHeader.click();
				saveButton.up("label").classList.remove("updating");
				window.setTimeout(function () {
					nextPinHeader.next().down(".pinnumber span").classList.add("flash");
				}, 100);
				window.setTimeout(function () {
					nextPinHeader.next().down(".pinnumber span").classList.remove("flash");
				}, 5000);
			} else {
				Shipment.DataCollection.closePuck(saveButton);
			}
		},

		closePuck: function (saveButton) {
			let header = saveButton.up(".tabbody").previous();
			let pin = header.pin;
			let puckDiv = $("dcpuck" + pin.puckid);
			let ts = header.up(".tabset");
			puckDiv.classList.remove("dccurrentpuck");
			ts.remove();
		},

		updatePuckCompletedStatus:function(puckButton){
			let completed=true;
			puckButton.pins.each(function(pin){
				if(!pin.childitems){ return; } //from this iteration
				if(!pin.actionupdated || ""===pin.childitems[1].actiononreturn){
					completed=false;
				}
			});
			if(completed){
				puckButton.classList.add("dccompletedpuck");
			} else {
				window.setTimeout(function () {
					Shipment.DataCollection.updatePuckCompletedStatus(puckButton);
				}, 1000);
			}
		},

	}, //end DataCollection

	/**
	 * Handles the return of pins, pucks and dewars.
	 */
	DewarReturn: {

		renderTab: function () {
			if (!data["objecttype"] || "shipment" !== data["objecttype"]) {
				//Not on a shipment page!
				return false;
			} else if ("" === data.datereturned) {
				//Shipment has not been returned
				return false;
			} else if (!data["manifest"]) {
				//no idea what was in the shipment
				return false;
			}
			$("shiptabs").tab({
				label: "Shipment return",
				id: "shipmentreturn",
				content: ""
			});
			let dewars = data["manifest"].rows;
			dewars.forEach(function (dewar) {
				Shipment.DewarReturn.renderDewar(dewar);
			});
		},

		renderDewar: function (dewar) {
			//Ensure that the dewar was removed from the shipment when shipment was marked returned.
			//If the containercontent for the shipment-dewar relationship exists, delete it.
			new Ajax.Request("/api/containercontent/" + dewar.containercontentid, {
				"method": "get",
				onSuccess: function () {
					new Ajax.Request("/api/containercontent/" + dewar.containercontentid, {
						"method": "delete",
						"parameters": {"csrfToken": csrfToken},
						onSuccess: function () {
						},
						onFailure: function () {
							alert("Dewar was not removed from shipment when shipment was returned. Could not remove it now.")
						}
					});
				},
				onFailure: function (transport) {
					if (404 !== transport.status) {
						return AjaxUtils.checkResponse(transport);
					}
				}
			});
			//Render a tree item for this dewar
			let ti = $("shipmentreturn_body").treeItem({
				header: "Dewar " + dewar.name + ': <span class="returndetails" style="margin-right:1em">...Checking...</span>'
			});
			let th = ti.down(".treehead");
			let tb = ti.down(".treebody");
			th.classList.add("updating");
			//And render each puck within the dewar, after checking whether the dewar-puck relationship still exists
			dewar.childitems.forEach(function (puck) {
				if ("" === puck || "dummy" === puck) {
					return;
				}
				new Ajax.Request("/api/containercontent/" + puck["containercontentid"], {
					method: "get",
					onSuccess: function () {
						Shipment.DewarReturn.renderPuck(puck, tb, true);
					},
					onFailure: function (transport) {
						if (404 !== transport.status) {
							AjaxUtils.checkResponse(transport);
						} else {
							Shipment.DewarReturn.renderPuck(puck, tb, false);
						}
					},
				});
			});
			window.setTimeout(function () {
				Shipment.DewarReturn.updateDewarHeader(th);
			}, 1000);
		},

		updateDewarHeader: function (dewarHeader) {
			let treeItem = dewarHeader.closest(".treeitem");
			window.setTimeout(function () {
				Shipment.DewarReturn.updateDewarHeader(dewarHeader);
			}, 1000);
			if (treeItem.querySelector(".treebody .updating")) {
				return false;
			}
			let pucksStillInDewar = 0;
			let crystalsStoredElsewhere=0;
			let pucks = treeItem.querySelectorAll(".treebody .treeitem");
			pucks.forEach(function (puck) {
				if (1 * puck.dataset.puckindewar || "true"===puck.dataset.puckindewar) {
					pucksStillInDewar++;
				}
				crystalsStoredElsewhere+=1*(puck.dataset.crystalsStoredElsewhere);
			});

			let out = "";
			if (!pucksStillInDewar) {
				out += "All pucks have been removed.";
			} else if (1 === pucksStillInDewar) {
				out += "1 puck is still in this dewar.";
			} else {
				out += pucksStillInDewar + " pucks are still in this dewar.";
			}
			if(crystalsStoredElsewhere){
				out+='<span style="color:#aaf">&nbsp;&nbsp;&nbsp;&nbsp;Crystals stored elsewhere: ' + crystalsStoredElsewhere+'</span>';
			}
			dewarHeader.querySelector(".returndetails").innerHTML = out;
			dewarHeader.classList.remove("updating"); //Only use this for initial render, then auto-update.
		},

		renderPuck: function (puck, dewarBody, puckStillInDewar) {
			let ti = dewarBody.treeItem({
				header: '<span class="removebuttons" style="float:right;margin:.25em 1em">...Checking...</span> Puck ' + puck.name + ': <span class="returndetails">...Checking...</span>'
			});
			let th = ti.down(".treehead");
			let tb = ti.down(".treebody");
			th.classList.add("updating");
			ti.dataset.puckindewar = puckStillInDewar;
			ti.dataset.ccPuckInDewar = puck.containercontentid;

			let positions = {"rows": puck.childitems.slice(1)};
			let t = tb.table({
				headers: ["Pos", "Pin", "Crystal", "Protein", "Action on return", ""],
				cellTemplates: ["{{position}}", "", "", "", "", ""] //Pin renderer will populate
			}, positions);

			t.querySelectorAll("tr.datarow").forEach(function (row) {
				let pin = row.rowData;
				if (pin.isEmpty) {
					let cells = row.querySelectorAll("td");
					cells[5].innerHTML = "";
					row.dataset.crystalinpin = "";
					row.dataset.pininpuck = "";
					row.dataset.actiononreturn = "";
				} else {
					row.querySelector("td").classList.add("updating");
					//is pin still in puck?
					new Ajax.Request("/api/containercontent/" + pin.containercontentid, {
						method: "get",
						onSuccess: function () {
							row.dataset.pininpuck = "1";
						},
						onFailure: function () {
							row.dataset.pininpuck = "0";
						}
					});
					//is crystal still in pin?
					new Ajax.Request("/api/containercontent/" + pin.childitems[1].containercontentid, {
						method: "get",
						onSuccess: function () {
							row.dataset.crystalinpin = "1";
						},
						onFailure: function () {
							row.dataset.crystalinpin = "0";
						}
					});
					//What should be done with the pin?
					new Ajax.Request("/api/diffractionrequest/" + pin.childitems[1]["diffractionrequestid"], {
						method: "get",
						onSuccess: function (transport) {
							row.dataset.actiononreturn = transport.responseJSON.actiononreturn;
						},
						onFailure: function () {
							row.dataset.actiononreturn = "fail";
						}
					});
					//Render the pin.
					Shipment.DewarReturn.renderPin(row);
				}
			});
			window.setTimeout(function () {
				Shipment.DewarReturn.updatePuckHeader(th);
			}, 500);
		},

		updatePuckHeader: function (puckHeader) {
			window.setTimeout(function () {
				Shipment.DewarReturn.updatePuckHeader(puckHeader);
			}, 1000);
			let treeItem = puckHeader.closest(".treeitem");
			// if (treeItem.querySelector(".updating")) {
			// 	return false;
			// }

			let rows = treeItem.querySelectorAll("tr.datarow");
			let crystalsToCheckWithOwner = 0;
			let crystalsToKeepInPuck = 0;
			let crystalsToKeepElsewhere = 0;
			let emptyPinsInPuck = 0;
			let crystalsToWash = 0;

			//Do pins still need to be washed? y/n
			//Have crystals been kept, and still in the puck? y/n
			rows.forEach(function (row) {
				let actionOnReturn=row.dataset.actiononreturn;
				let crystalInPin=1*row.dataset.crystalinpin;
				let pinInPuck=1*row.dataset.pininpuck;
				if (1 * row.rowData.isEmpty || (!crystalInPin && !pinInPuck)) {
					//crystal not in pin, pin not in puck, job done
				} else if (crystalInPin && pinInPuck) {
					//crystal in pin, pin in puck, nothing has been done yet
					if("keep"===actionOnReturn){
						crystalsToKeepInPuck++;
					} else if("wash"===actionOnReturn){
						crystalsToWash++;
					} else {
						crystalsToCheckWithOwner++;
					}
				} else if (!crystalInPin && pinInPuck) {
					//Pin is empty but still in puck
					emptyPinsInPuck++;
				} else if (crystalInPin && !(pinInPuck)) {
					//Crystal is in pin, but removed from puck
					crystalsToKeepElsewhere++;
				} else {
					//Something weird happened
				}
			});

			let parts = [];
			if (crystalsToKeepInPuck) {
				parts.push('<span style="color:#9f9">'+"Crystals to keep, in puck: " + crystalsToKeepInPuck+"</span>");
			}
			if (crystalsToKeepElsewhere) {
				parts.push('<span style="color:#aaf">' +"Crystals stored elsewhere: " + crystalsToKeepElsewhere+"</span>");
				treeItem.dataset.crystalsStoredElsewhere=crystalsToKeepElsewhere+"";
			}
			if (crystalsToWash) {
				parts.push('<span style=\"color:#f99">' +"Pins to wash: " + crystalsToWash+"</span>");
			}
			if (emptyPinsInPuck) {
				parts.push('<span style=\"color:#f99\">' +"Empty pins in puck: " + emptyPinsInPuck+"</span>");
			}
			if (crystalsToCheckWithOwner) {
				parts.push('<span style=\"color:#ff9\">' +"Check with pin owner: " + crystalsToCheckWithOwner+"</span>");
			}
			let out = parts.join("&nbsp;&nbsp;&nbsp;&nbsp;");
			if ("" === out) {
				out = "All pins have been removed.";
			} else if(crystalsToKeepElsewhere){
				out = "All pins have been removed.&nbsp;&nbsp;&nbsp;&nbsp;"+out;
			}
			puckHeader.querySelector(".returndetails").innerHTML = out;

			//Do I need a "remove puck" button? y/n
			if ("1"===treeItem.dataset.puckindewar || "true"===treeItem.dataset.puckindewar) {
				treeItem.querySelector(".removebuttons").innerHTML = '<input onclick="Shipment.DewarReturn.removePuckFromDewar(this);return false" value="Remove puck from dewar" type="button" style="float:right" />';
			} else {
				treeItem.querySelector(".removebuttons").innerHTML = "";
			}
			puckHeader.classList.remove("updating"); //Only use this for initial render, then auto-update.
		},

		renderPin: function (row) {
			//Wait for checks on current status to complete
			if (undefined===row.dataset.crystalinpin || undefined===row.dataset.pininpuck || ""===row.dataset.crystalinpin || ""===row.dataset.pininpuck || undefined===row.dataset.actiononreturn) {
				window.setTimeout(function () {
					Shipment.DewarReturn.renderPin(row);
				}, 250);
				return false;
			}

			let cells = row.select("td");
			let pin = row.rowData;

			row.dataset.ccPinInPuck = pin.containercontentid;
			row.dataset.ccCrystalInPin = pin.childitems[1].containercontentid;

			if (0 === pin.name.indexOf("dummypin")) {
				cells[1].innerHTML = "(no barcode)";
			} else {
				cells[1].innerHTML = '<a href="/container/' + pin.id + '">' + pin.name + '</a>';
			}

			cells[2].innerHTML = '<a href="' + pin.crystallocalurl + '">' + pin.childitems[1].name + '</a>';

			cells[3].innerHTML = pin.proteinacronym;

			let keepCrystal = true;
			let action = '<span style="font-weight:bold;color:#990">Check with owner</span>';
			if ("keep" === row.dataset.actiononreturn) {
				keepCrystal = true;
				action = '<span style="font-weight:bold;color:#090">Keep crystal</span>';
			} else if ("wash" === row.dataset.actiononreturn) {
				keepCrystal = false;
				action = '<span style="font-weight:bold;color:#900">Wash pin</span>';
			}
			cells[4].innerHTML = action;

			let pinInPuck = 1 * row.dataset.pininpuck;
			let crystalInPin = 1 * row.dataset.crystalinpin;

			let controls="";
			if (pinInPuck) {
				if (crystalInPin) {
					//pin still in puck, with crystal
					if (pin.name.indexOf("dummypin")) {
						controls += 'Remove and ' +
							'<input onclick="Shipment.DewarReturn.removePinFromPuck(this)" value="Keep crystal" type="button" style="float:none" /> ' +
							'<input onclick="Shipment.DewarReturn.removePinAndWashCrystal(this)" value="Wash pin" type="button" style="float:none" />';
					} else {
						controls += 'Non-barcoded pin. ' +
							'<input onclick="Shipment.DewarReturn.removePinAndWashCrystal(this)" value="Remove and wash" type="button" style="float:none" />';
					}
				} else {
					//pin still in puck, crystal washed
					//Shouldn't happen. Remove the pin.
					controls += 'Pin washed, but still in puck. <input onclick="Shipment.DewarReturn.removePinFromPuck(this)" value="Remove" type="button" style="float:none" />';
				}
			} else {
				if (crystalInPin) {
					//pin removed, crystal kept
					controls += 'Pin moved to storage, crystal kept. <input onclick="Shipment.DewarReturn.removeCrystalFromPin(this)" value="Wash pin" type="button" style="float:none" />';
					if (keepCrystal) {
						row.classList.remove("outstanding");
					}
				} else {
					//pin removed and washed
					controls += "Pin removed and washed.";
					row.classList.remove("outstanding");
				}
			}
			cells[5].innerHTML = controls;

			row.querySelector("td").classList.remove("updating");
		},

		removePuckFromDewar: function (btn) {

			let puckHeader = btn.up(".treehead");
			let treeItem = puckHeader.up(".treeitem");
			puckHeader.classList.add("updating");
			if ("true"===treeItem.dataset.puckindewar) {
				new Ajax.Request("/api/containercontent/" + treeItem.dataset.ccPuckInDewar, {
					method: "delete",
					parameters: {csrfToken: csrfToken},
					onSuccess: function () {
						treeItem.dataset.puckindewar = "false";
					},
					onFailure: function (transport) {
						AjaxUtils.checkResponse(transport);
					},
				});
			}
			return false;
		},

		removePinFromPuck: function (btn) {
			let pinTr = btn.up("tr");
			Shipment.DewarReturn.waitForPinStateAndReRender(pinTr, false, pinTr.dataset.crystalinpin);
			Shipment.DewarReturn._doRemovePinFromPuck(pinTr);
		},
		removeCrystalFromPin: function (btn) {
			let pinTr = btn.up("tr");
			if ("keep" === pinTr.dataset.actiononreturn && !confirm("This crystal is to be kept. Really wash the pin?")) {
				return false;
			}
			Shipment.DewarReturn.waitForPinStateAndReRender(pinTr, pinTr.dataset.pininpuck, false);
			Shipment.DewarReturn._doRemoveCrystalFromPin(pinTr);
		},
		removePinAndWashCrystal: function (btn) {
			let pinTr = btn.up("tr");
			if ("keep" === pinTr.dataset.actiononreturn && !confirm("This crystal is to be kept. Really wash the pin?")) {
				return false;
			}
			Shipment.DewarReturn.waitForPinStateAndReRender(pinTr, false, false);
			Shipment.DewarReturn._doRemoveCrystalFromPin(pinTr);
			Shipment.DewarReturn._doRemovePinFromPuck(pinTr);
		},

		_doRemovePinFromPuck: function (pinTr) {
			if (1 * pinTr.dataset.pininpuck) {
				new Ajax.Request("/api/containercontent/" + pinTr.dataset.ccPinInPuck, {
					method: "delete",
					parameters: {csrfToken: csrfToken},
					onSuccess: function () {
						pinTr.dataset.pininpuck = "0";
						if("1"===pinTr.dataset.crystalinpin){
							let treeItem=pinTr.closest(".treeitem");
							if(treeItem.dataset.crystalsStoredElsewhere){
								treeItem.dataset.crystalsStoredElsewhere=""+(1+parseInt(treeItem.dataset.crystalsStoredElsewhere));
							} else {
								treeItem.dataset.crystalsStoredElsewhere="1";
							}
						}
					},
					onFailure: function (transport) {
						AjaxUtils.checkResponse(transport);
					},
				});
			}
		},

		_doRemoveCrystalFromPin: function (pinTr) {
			if (1 * pinTr.dataset.crystalinpin) {
				new Ajax.Request("/api/containercontent/" + pinTr.dataset.ccCrystalInPin, {
					method: "delete",
					parameters: {csrfToken: csrfToken},
					onSuccess: function () {
						pinTr.dataset.crystalinpin = "0";
					},
					onFailure: function (transport) {
						AjaxUtils.checkResponse(transport);
					},
				});
			}
		},

		waitForPinStateAndReRender: function (pinTr, pinInPuck, crystalInPin) {
			if (1 * pinInPuck !== 1 * pinTr.dataset.pininpuck || 1 * crystalInPin !== 1*pinTr.dataset.crystalinpin) {
				window.setTimeout(function () {
					Shipment.DewarReturn.waitForPinStateAndReRender(pinTr, pinInPuck, crystalInPin);
				}, 250);
				return false;
			}
			Shipment.DewarReturn.renderPin(pinTr); //Puck and dewar above will update on their regular schedule.
		},

	}, //end DewarReturn

	DatasetRetrieval:{

		datasets:[],

		beamlines:[],
		beamlineNameToId:{},

		renderProgressNumbers:function (){
			let mb=document.getElementById("modalBox").querySelector(".boxbody");
			if(!document.getElementById("progressnumbers")){
				let pn=document.createElement("div");
				pn.id="progressnumbers";
				let counts=["Found","Processed","Succeeded","Failed"];
				counts.forEach(function (c){
					let dv=document.createElement("div");
					dv.style.display="inline-block";
					dv.style.margin="1em";
					dv.style.textAlign="center";
					dv.style.width="8em";
					let num=document.createElement("div");
					num.id="num"+c;
					num.classList.add("bignumbers");
					dv.appendChild(num);
					num.innerHTML="0";
					num.dataset.numDatasets="0";
					dv.innerHTML+=c;
					pn.appendChild(dv);
				});
				mb.appendChild(pn);
			}
			Shipment.DatasetRetrieval.setFoundCount();
		},
		setFoundCount:function(){
			document.getElementById("numFound").innerHTML=Shipment.DatasetRetrieval.datasets.length+"";
		},
		incrementFailedCount:function (message){
			ui.logToDialog(message,"error");
			Shipment.DatasetRetrieval.incrementProgressCount("Processed");
			Shipment.DatasetRetrieval.incrementProgressCount("Failed");
		},
		incrementSucceededCount:function (message){
			ui.logToDialog(message,"success");
			Shipment.DatasetRetrieval.incrementProgressCount("Processed");
			Shipment.DatasetRetrieval.incrementProgressCount("Succeeded");
		},
		incrementProgressCount:function (name){
			let elem=document.getElementById("num"+name);
			elem.innerHTML=(1+parseInt(elem.innerHTML))+"";
			if("Processed"===name && parseInt(document.getElementById("numFound").innerHTML)===parseInt(document.getElementById("numProcessed").innerHTML)){
				ui.logToDialog("All datasets processed");
			}
		},

		/**
		 * Creates or updates IceBear beamline records matching the supplied objects.
		 * @param beamlines an array of beamline objects, containing at minimum "name" and "shipmentdestinationid"
		 * 					properties. "detectormanufacturer", "detectormodel" and "detectortype" are optional.
		 */
		updateIceBearBeamlines:function (beamlines){
			Shipment.DatasetRetrieval.beamlines=beamlines;
			beamlines.forEach(function(bl){
				new Ajax.Request("/api/beamline/shipmentdestinationid/"+data["shipmentdestinationid"]+"/name/"+encodeURIComponent(bl["name"]), {
					method:'get',
					onSuccess:function(transport){
						let found=transport.responseJSON.rows;
						if(1===found.length){
							found=found[0];
							ui.logToDialog("Found IceBear beamline with name "+bl.name+", updating");
							Shipment.DatasetRetrieval.createOrUpdateIceBearBeamline(bl, found.id);
						} else {
							ui.logToDialog("Error on checking for IceBear beamline, more than one found","error");
						}
					},
					onFailure:function(transport){
						if(404===transport.status){
							ui.logToDialog("No IceBear beamline for this synchrotron with name "+bl.name+", creating");
							Shipment.DatasetRetrieval.createOrUpdateIceBearBeamline(bl, null);
						} else {
							ui.logToDialog("Error on checking for IceBear beamline, HTTP "+transport.status, "error");
						}
					}
				});
			});
		},

		createOrUpdateIceBearBeamline:function(beamline, localId){
			let uri="/api/beamline/";
			let method="post";
			if(localId){
				uri+=localId;
				method="patch";
				ui.logToDialog("Updating IceBear beamline "+beamline["name"]);
			} else {
				ui.logToDialog("Creating IceBear beamline "+beamline["name"]);
			}
			beamline["csrfToken"]=csrfToken;
			new Ajax.Request(uri,{
				method:method,
				parameters:beamline,
				onSuccess:function (transport){
					ui.logToDialog("Created/updated IceBear beamline OK","success");
					let beamlineId=0;
					if(transport.responseJSON["created"]){ beamlineId=transport.responseJSON["created"].id; }
					if(transport.responseJSON["updated"]){ beamlineId=transport.responseJSON["updated"].id; }
					Shipment.DatasetRetrieval.beamlineNameToId[beamline["name"]]=beamlineId;
				},
				onFailure:function (transport){
					ui.logToDialog("Error on creating/updating IceBear beamline, HTTP "+transport.status, "error");
				}
			});
		}

	} //end DatasetRetrieval


}; //end Shipment

