<?php 

class validator {
	
	const ANY='any';
	const REQUIRED='required';
	const ALPHANUMERIC='alphanumeric';
	const ALPHANUMERIC_PLUS_UNDERSCORE='alphanumericplusunderscore';
	const INTEGER='integer';
	const FLOAT='float';
	const BOOLEAN='boolean';
	const EMAIL='email';
	const USERNAME='username';
	const DATETIME='datetime';
	const DATE='date';
	const COLOR='color';
	
	const GROUPVISIBILITY='groupvisibility';
	const GROUPMEMBERSHIPVISIBILITY='membershipvisibility';
	const GROUPJOINING='joining';
	const PERMISSIONTYPE='permissiontype';

	//TODO Application-specific, extend this base class
	const DNASEQUENCE='dnaSequence';
	const PROTEINSEQUENCE='proteinSequence';
	const DROPMAPPING='dropmapping';
	const PDBCODE='pdbcode';
	const PDBCODE_COMMASEPARATED='pdbcodemultiple';


    /**
     * ValidateS $fieldValue and returns true if valid or throws a BadRequestException if invalid.
     * @param string $fieldName The name of the field being validated. Used only in the failure case.
     * @param string $fieldValue The value to validate
     * @param string|array $validationNames The name of the validation to perform. Use constants like validator::REQUIRED to take advantage of your IDE's autocomplete.
     * @return boolean true if $fieldValue is valid.
     * @throws ServerException
     * @throws BadRequestException if $fieldValue is not valid.
     */
	public static function validate($fieldName, $fieldValue, $validationNames){
		if(!is_array($validationNames)){
			$validationNames=array($validationNames);
		}
		foreach($validationNames as $vn){
			if(!self::isValid($fieldValue, $vn)){
				$message='"'.$fieldValue.'" is not valid for field "'.$fieldName.'"';
				$validation=self::$validations[$vn];
				if($validation && isset($validation['message'])){
					$message='Field "'.$fieldName.'"'.$validation['message'];
				}
				throw new BadRequestException($message);
			}
		}
		return true;
	}
	
	/**
	 * Determines whether $fieldValue is valid, in a way specified by $validationName. Returns true if valid, false otherwise.
	 * @param string $fieldValue The value to validate
	 * @param array|string $validationNames The name(s) of the validation(s) to perform. Use constants like validator::REQUIRED to take advantage of your IDE's autocomplete.
	 * @throws ServerException if $validationName is not a known validation.
	 * @return boolean true if $fieldname validates, false otherwise
	 */
	public static function isValid($fieldValue, $validationNames){
		if(!is_array($validationNames)){
			$validationNames=array($validationNames);
		}
		$isValid=true;
		/*
		if(!in_array(validator::REQUIRED, $validationNames) && empty($fieldValue)){
			return true;
		}
		if($fieldValue==database::$nullValue){ return true; }
		*/
		if(!in_array(validator::REQUIRED, $validationNames) && (empty($fieldValue)||database::$nullValue==$fieldValue)){
		    return true;
		}
		foreach($validationNames as $validationName){
			if(!isset(validator::$validations[$validationName])){
				throw new ServerException('Server tried to validate string "'.$fieldValue.'" with non-existent validation type "'.$validationName.'"');
			}
			$validation=validator::$validations[$validationName];
			if(isset($validation['pattern']) && !preg_match('/^'.$validation['pattern'].'$/', $fieldValue)) { $isValid=false; break; }
			if(isset($validation['helper']) && !forward_static_call('self::'.$validation['helper'], $fieldValue)) { $isValid=false; break; }
		}
		return $isValid;
	}
	
	public static function getValidationPatterns(){
		return self::$validations;
	}
	
	/**
	 * 
	 * NOTE: If adding helper functions, these need to be mirrored in the client Javascript validator.
	 */
	private static $validations=array(
			self::ANY=>array(
				'message'=>'' //No validation, so shouldn't ever need a message
			),
			self::REQUIRED=>array(
				'pattern'=>'.+',
				'message'=>' is required.'
			),
    	    self::ALPHANUMERIC=>array(
    	        'pattern'=>'[A-Za-z0-9]*',
    	        'message'=>' must be alphanumeric.'
    	    ),
    	    self::ALPHANUMERIC_PLUS_UNDERSCORE=>array(
    	        'pattern'=>'[A-Za-z0-9_]*',
    	        'message'=>' must be alphanumeric (underscores allowed).'
    	    ),
    	    self::INTEGER=>array(
				'pattern'=>'-?[0-9]*',
				'message'=>' must be a whole number.'
			),
			self::FLOAT=>array(
				'pattern'=>'-?(\d*[,\.])?\d+',
				'message'=>' must be a number.'
			),
			
			self::COLOR=>array(
					'pattern'=>'[0-9A-Fa-f]{3,6}',
					'message'=>' must be an HTML hex colour code.'
			),
				
			self::DATETIME=>array(
				'pattern'=>'\d\d\d\d-\d\d-\d\d \d\d:\d\d:\d\d',
				'message'=>' must be a date/time in "YYYY-MM-DD hh:mm:ss" format.'
			),
			self::DATE=>array(
				'pattern'=>'\d\d\d\d-\d\d-\d\d',
				'message'=>' must be a date in "YYYY-MM-DD" format.'
			),
			
			self::BOOLEAN=>array(
				'pattern'=>'[01]',
				'message'=>' must be either 1 (true) or 0 (false).'
			),
			self::EMAIL=>array(
				'helper'=>'isValidEmailAddress',
				'message'=>' must be a valid email address.'
			),
			self::USERNAME=>array(
				'pattern'=>'[A-za-z][A-Za-z0-9_-]+',
				'message'=>' must begin with a letter and contain only letters, numbers, underscore and dash.'
			),
			
			self::GROUPVISIBILITY=>array(
				'pattern'=>'visible|hidden|membersonly',
				'message'=>' must be one of visible, hidden or membersonly.'
			),
			self::GROUPMEMBERSHIPVISIBILITY=>array(
				'pattern'=>'visible|hidden|membersonly',
				'message'=>' must be one of visible, hidden or membersonly.'
			),
			self::GROUPJOINING=>array(
				'pattern'=>'open|request|closed|auto',
				'message'=>' must be open, request, closed or auto.'
			),
			self::PERMISSIONTYPE=>array(
				'pattern'=>'create|read|update|delete',
				'message'=>' must be create, read, update or delete.'
			),
			
			/* TODO Application-specific, should move */
			
			self::DNASEQUENCE=>array(
				'helper'=>'isValidDnaSequence',
				'message'=>' must be a valid DNA sequence. Only A, C, G and T are valid, and the length must be divisible by 3.'
			),
			self::PROTEINSEQUENCE=>array(
				'pattern'=>'^[\*\sACDEFGHIKLMNPQRSTVWY]*$',
				'message'=>' must be a valid protein sequence.'
			),
            self::DROPMAPPING=>array(
                'pattern'=>'[0-9ERX,]+',
                'message'=>' must be a valid drop mapping.'
            ),

            //https://proteopedia.org/wiki/index.php/PDB_identification_code#Future_Plans_for_Expanded_PDB_Codes
            self::PDBCODE=>array(
                'pattern'=>'(pdb_\d\d\d\d)?\d[a-zA-Z0-9]{3}',
                'message'=>' must be a valid PDB code.'
            ),
            self::PDBCODE_COMMASEPARATED=>array(
                'pattern'=>'(?:pdb_\d\d\d\d)?\d[a-zA-Z0-9]{3}(?:, ?(?:pdb_\d\d\d\d)?\d[a-zA-Z0-9]{3})*',
                'message'=>' must be a comma-separated list of PDB codes.'
            )
	);

    /**
     * @param $seq
     * @return bool
     * @noinspection PhpUnusedPrivateMethodInspection
     */
	private static function isValidDnaSequence($seq){
		$seq=strtoupper(str_replace(' ','',$seq));
		if(!preg_match('/^[ACGT]*$/', $seq)){ return false; }
		if(strlen($seq)%3 != 0){ return false; }
		return true;
	}

    /**
     * @param $email
     * @return bool
     * @noinspection PhpUnusedPrivateMethodInspection
     */
	private static function isValidEmailAddress($email){
		if(filter_var($email, FILTER_VALIDATE_EMAIL)===false){
			return false;
		}
		return true;
	}
	
}