<?php 

/**
 * This class is used to manage the default layout of the homepage for new users.
 * Nothing in this class affects the homepage layout of existing users.
 * There is one homepagedefaultlayout entry per brick on the default homepage.
 */

class homepagedefaultbrick {
	
	private static $select='SELECT db.id AS id, b.id AS homepagebrickid,
				db.row AS `row`, db.col AS `col`,
				b.name AS name, b.version AS version, b.title AS title,
				b.description AS description, b.height AS height, b.width AS width,
				b.apiurl AS apiurl, b.scriptblock AS scriptblock, b.content AS content,
				b.headertemplates AS headertemplates, b.rowtemplates AS rowtemplates,
				b.adminonly AS adminonly
				FROM homepagedefaultbrick AS db, homepagebrick AS b
				WHERE db.homepagebrickid=b.id';

    /**
     * @throws ForbiddenException
     */
	private static function checkCanUpdate(){
		if(!session::isAdmin()){
			throw new ForbiddenException('Only administrators can configure the default homepage');
		}
	}

    /**
     * @param $id
     * @return array
     * @throws BadRequestException
     * @throws ServerException
     */
	public static function getById($id){
	    return database::queryGetOne(
	        static::$select.' AND db.id=:id',
	        array(':id'=>$id)
	        );
	}

    /**
     * @param $name
     * @throws BadRequestException
     * @noinspection PhpUnusedParameterInspection
     */
	public static function getByName($name){
	    throw new BadRequestException('getByName not implemented on homepagedefaultbrick');
	}

    /**
     * @param array $request
     * @return array
     * @noinspection PhpUnusedParameterInspection
     * @throws BadRequestException
     * @throws ServerException
     */
	public static function getAll($request=array()){
		return database::queryGetAll(
				static::$select,
				array()
		);	
	}

    /**
     * @return array
     */
	public static function getFieldHelpTexts(){
		return array();
	}

    /**
     * @return array
     */
	public static function getFieldValidations(){
		return array();
	}

    /**
     * @param array $request
     * @return array
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws ServerException
     */
	public static function create($request=array()){
		static::checkCanUpdate();
		if(!isset($request['homepagebrickid']) || !isset($request['row']) || !isset($request['col'])){
			throw new BadRequestException('Brick ID, row and col are required');
		}
		$sqlStatement='INSERT INTO homepagedefaultbrick(`homepagebrickid`, `row`, `col`)
				VALUES(:homepagebrickid, :row, :col)';
		$parameters=array(
				':homepagebrickid'=>$request['homepagebrickid'],
				':row'=>$request['row'],
				':col'=>$request['col']
		);
		database::query($sqlStatement, $parameters);
		$newId=database::getLastInsertId();
		return array('type'=>'homepageuserbrick', 'created'=>self::getById($newId));
	}

    /**
     * @param $id
     * @param array $request
     * @return array
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
	public static function update($id, $request=array()){
		static::checkCanUpdate();
		$defaultBrick=static::getById($id);
		if(!$defaultBrick){
			throw new NotFoundException('No default homepage brick with ID '.$id);
		}
		$parts=array();
		foreach($request as $k=>$v){
			if(!in_array($k, array('row','col'))){
				unset($request[$k]);
				continue;
			}
			$v=(int)$v;
			//We assume that the client threw out any overlapping/non-fitting layouts, but check for the truly stupid
			if($v<1 ||$v>3){
				throw new BadRequestException($k.' must be between 1 and 3, was '.$v);
			}
			$parts[]='`'.$k.'`=:'.$k;
		}
		$request['id']=$id;
		$sqlStatement='UPDATE homepagedefaultbrick SET '.implode(', ',$parts).' WHERE id=:id';
		database::query($sqlStatement, $request);
		return static::getById($id);
	}

    /**
     * @param $id
     * @return array
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
	public static function delete($id){
		static::checkCanUpdate();
		$defaultBrick=static::getById($id);
		if(!$defaultBrick){
			throw new NotFoundException('No default homepage brick with ID '.$id);
		}
		$sqlStatement='DELETE FROM homepagedefaultbrick WHERE id=:id';
		database::query($sqlStatement, array(':id'=>$id));
		return array('deleted'=>$id);
	}

}