<?php 

class database {

	static $connection;
	static $inTransaction=false;
	
	static $nullValue='NULL';
	static $lastAffectedRowCount;
	/**
	 * Connect to the database using credentials supplied in the config file.
	 */
	public static function connect(){
		try {
			$conf=parse_ini_file(realpath(__DIR__).'/../../conf/config.ini');
			if(!$conf){ die('Could not read database connection details from config file'); }
			$db = new PDO('mysql:host='.$conf['dbHost'].';dbname='.$conf['dbName'], $conf['dbUser'], $conf['dbPass'], array(PDO::MYSQL_ATTR_FOUND_ROWS => true));
			$conf=null; //for security, do not keep the DB credentials
			$db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
			self::$connection=$db;
			self::$connection->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
		} catch (PDOException $e) {
			die("Could not connect to database");
		}
	}
	
	/**
	 * Begin a database transaction.
     * @noinspection PhpUndefinedMethodInspection
     */
	public static function begin(){
		if(!self::$connection){ self::connect(); }
		self::$connection->beginTransaction();
		self::$inTransaction=true;
	}
	/**
	 * Commit a database transaction.
     * @noinspection PhpUndefinedMethodInspection
     */
	public static function commit(){
		if(!self::$inTransaction){ return; }
		self::$connection->commit();
		self::$inTransaction=false;
	}
	/**
	 * Abort a database transaction.
     * @noinspection PhpUndefinedMethodInspection
     */
	public static function abort(){
		if(!self::$inTransaction){ return; }
		self::$connection->rollBack();
		self::$inTransaction=false;
	}
	
	/**
	 * After a SELECT, return the number of rows that would have matched, regardless of the LIMIT clause.
	 * For MYSQL, the query must begin SELECT SQL_CALC_FOUND_ROWS.
     * @throws ServerException
     * @throws BadRequestException
	 */
	public static function getFoundRows(){
		$result=self::doQuery('SELECT FOUND_ROWS()', array(), 'one');
		if(!$result){ return 0; }
		return (int)$result['FOUND_ROWS()'];
	}
	
	public static function getAffectedRows(){
		return (int)self::$lastAffectedRowCount;
	}

    /**
     * @noinspection PhpUndefinedMethodInspection
     * @param $sqlStatement
     * @param $parameters
     * @param $returnType
     * @return array|null
     * @throws BadRequestException
     * @throws ServerException
     */
    private static function doQuery($sqlStatement, $parameters, $returnType){
	    try {
	        //Original - cant handle null
	        foreach($parameters as $k=>$v){
				if(is_bool($v)){
					$parameters[$k]=(int)$v;
				} else if(strtoupper($v)==database::$nullValue){
				    $parameters[$k]=null;
				}
			}
			$stmt=self::$connection->prepare($sqlStatement);
			$stmt->execute($parameters);
			static::$lastAffectedRowCount=$stmt->rowCount();
			
			if('one'==$returnType){
				$result=$stmt->fetch();
				if(!$result){return null; }
				$keys=array_keys($result);
				foreach($keys as $k){
					if(is_int($k)){
						unset($result[$k]);
					}
				}
				return $result;
			} else if('all'==$returnType){
				$result=$stmt->fetchAll();
				if(!$result){return null; }
				foreach ($result as &$row){ //by reference
					$keys=array_keys($row);
					foreach($keys as $k){
						if(is_int($k)){
							unset($row[$k]);
						}
					}
				}
				return $result;
			} else {
				return $stmt->rowCount();
			}
		} catch (PDOException $e) {
			self::handlePDOException($e, $sqlStatement, $parameters);
		}
	}

    /**
     * Prepares and executes a PDO statement using the supplied SQL and parameters, and returns the number of affected rows.
     * Note that a row "updated" with the same values counts toward the total affected rows.
     * @param string $sqlStatement The SQL statement, with placeholders
     * @param array $parameters The array of placeholders and values
     * @return array|null The number of rows affected by the query.
     * @throws BadRequestException
     * @throws ServerException
     */
	public static function query($sqlStatement, $parameters=array()){
		return self::doQuery($sqlStatement, $parameters, null);
	}

    /**
     * Prepares and executes a PDO statement using the supplied SQL and parameters, and returns an array representing a single row.
     * @param string $sqlStatement The SQL statement, with placeholders
     * @param array $parameters The array of placeholders and values
     * @return array The first match.
     * @throws BadRequestException
     * @throws ServerException
     */
	public static function queryGetOne($sqlStatement, $parameters=array()){
		return self::doQuery($sqlStatement, $parameters, 'one');
	}

    /**
     * Prepares and executes a PDO statement using the supplied SQL and parameters, and returns an array containing two keys: "total" and "rows".
     * "total" is the number of rows that would have been matched, ignoring any LIMIT clause.
     * "rows" contains the actual data, one item per result row, as key-value pairs.
     * @param string $sqlStatement The SQL statement, with placeholders
     * @param array $parameters The array of placeholders and values
     * @return array The results, along with the total rows matched
     * @throws BadRequestException
     * @throws ServerException
     */
	public static function queryGetAll($sqlStatement, $parameters=array()){
		$result=self::doQuery($sqlStatement, $parameters, 'all');
		if(!$result){ return null; }
		$total=self::getFoundRows();
        return array(
                'total'=>$total,
                'rows'=>$result
        );
	}

    /**
     * @return mixed
     * @noinspection PhpUndefinedMethodInspection
     */
	public static function getLastInsertId(){
		return self::$connection->lastInsertId();
	}

    /**
     * @param Exception $e
     * @param string $sqlStatement
     * @param array $parameters
     * @throws BadRequestException
     * @throws ServerException
     */
	private static function handlePDOException($e, $sqlStatement, $parameters){
		$message=$e->getMessage();
        if(database::$inTransaction){
            database::abort();
        }
		if(false!==strpos($message, 'constraint violation')){
			throw new BadRequestException('Item may already exist, try another name '.$message);
		} else {
 			if(!session::isAdmin()){
 				$message='Database error, could not continue';
 			} else {
				$message.="\nStatement: $sqlStatement\nParameters:";
				foreach($parameters as $k=>$v){
					$message.=' '.$k.'='.$v;
				}
 			}
			throw new ServerException($message);
		}
	}

    /**
     * Returns a LIMIT clause for pagination, using $request['pagenumber'] and $request['pagesize']. If either is missing, returns an empty string.
     * @param $requestParameters
     * @return string the Limit clause if both parameters found, or an empty string
     */
	public static function getLimitClause($requestParameters){
		if(!isset($requestParameters['pagenumber']) && !isset($requestParameters['pagesize'])){
			$requestParameters['pagenumber']=1;
			$requestParameters['pagesize']=25;
		}
		if(isset($requestParameters['pagenumber']) && isset($requestParameters['pagesize'])){
			$pageNum=(int)($requestParameters['pagenumber']);
			$pageSize=(int)($requestParameters['pagesize']);
 			$start=(($pageNum-1)*$pageSize);
			return ' LIMIT '.$start.', '.$pageSize.' ';
		}
		return '';
	}

    /**
     * @param $requestParameters
     * @param string $className
     * @return string
     * @throws BadRequestException
     */
	public static function getOrderClause($requestParameters,$className=''){
		$sortOrder='ASC';
		if(''!=$className && !preg_match('/^[a-z0-9_]+$/',$className)){
			throw new BadRequestException('Class name not recognised');
		}
		if(isset($requestParameters['sortby'])){ 
			$sortBy=$requestParameters['sortby']; 
			if(isset($requestParameters['sortdescending']) && true==$requestParameters['sortdescending']){ 
				$sortOrder='DESC'; 
			}
			if(!preg_match('/^[a-z0-9_.]+$/',$sortBy)){
				throw new BadRequestException('Sort field not recognised');
			}
			if(''!=$className){
				$className.='.';
			}
			$sql=' ORDER BY LOWER('.$className.$sortBy.') '.$sortOrder.' ';
			if(stripos($sortBy,'.')!==FALSE){
			    $sql=' ORDER BY '.$sortBy.' '.$sortOrder.' ';
			} else if('id'==$sortBy){
				$sql=' ORDER BY '.$className.$sortBy.' '.$sortOrder.' ';
			} else {
				$validations=forward_static_call_array(array(trim($className,'.'), 'getFieldValidations'), array());
				$validation=(array)$validations[$sortBy];
				if(in_array(validator::INTEGER, $validation) || in_array(validator::FLOAT, $validation)){
					$sql=' ORDER BY '.$className.$sortBy.' '.$sortOrder.' ';
				}
			}
		} else if(isset($className::$defaultSortOrder)){
			$sql=' ORDER BY '.$className::$defaultSortOrder;
		} else {
			$sql='';
		}
		return $sql;
	}

    /**
     * Filter the results against supplied text values. This function expects the request parameters to contain a key
     * "filter", whose value is a JSON object with keys being property names and values being the text that must appear
     * in the corresponding property. For example {"name":"ab"} will only return rows whose name contains ab. This filter
     * is CASE-INSENSITIVE.
     *
     * Note that the standard UI tables do not use this functionality, retrieving all rows and filtering client-side.
     *
     * @param $requestParameters array The request parameters.
     * @param string $className
     * @return string
     * @throws BadRequestException
     */
	public static function getFilterClause($requestParameters,$className=''){
		if(!isset($requestParameters['filter'])){
			return '';
		}

        if(''!=$className){
            if(!preg_match('/^[a-z0-9_]+$/',$className)){
                throw new BadRequestException('Class name not recognised');
            }
            $className.='.';
        }

		$filters=json_decode($requestParameters['filter']);
		if(!$filters){
		    throw new BadRequestException('Filter is not JSON');
        }
		$filterClause='';
        foreach ($filters as $filterBy=>$filterText) {
            if(!preg_match('/^[A-Za-z0-9]*$/',$filterBy)){
                throw new BadRequestException('Bad filter key '.$filterBy);
            }
            if(!preg_match('/^[\sA-Za-z0-9_-]*$/',$filterText)){
                throw new BadRequestException('Bad filter text '.$filterText);
            }
            $filterClause.=' AND LOWER('.$className.$filterBy.') LIKE "%'.strtolower($filterText).'%" ';
		}
        return $filterClause;
	}

    /**
     * Returns an AND clause with the projects for which the current user is authorised to perform the specified operation.
     *
     * For administrators this is an empty string. Otherwise returns:
     * ' AND 1=0 ' for users who cannot do this operation on any project
     * ' AND project.id=123 ' for users who can do this on only one project
     * ' AND project.id IN (123,124) ' for users who can do this on more than one project
     *
     * @param string $accessType One of 'create','read','update' or 'delete';
     * @param bool $forceSharedProject
     * @return string The SQL clause
     * @throws NotFoundException
     * @throws ServerException
     * @throws BadRequestException
     */
	public static function getProjectClause($accessType, $forceSharedProject=false){
		if(session::isAdmin()){ 
			return ''; 
		}
		$projects=session::getProjectPermissions($accessType);
		if($forceSharedProject){
		    $sharedProjectId=project::getSharedProjectId();
		    if(!in_array($sharedProjectId, $projects)){
		        $projects[]=$sharedProjectId;
		    }
		}
		if(empty($projects)){ 
			return ' AND 1=0 '; //Always evaluates to false, so they won't get anything.
		} else if(1==count($projects)){
			return ' AND project.id='.(int)($projects[0]).' ';				
		} else {
			$projects=implode(',', $projects);
			if(preg_match('/[^0-9,]/',$projects)){
				throw new ServerException('Internal list of user projects corrupted');
			}
			return ' AND project.id IN('.$projects.') ';
		}
	}

    /**
     * Adds the column to the table, if it doesn't exist, otherwise alters it to the supplied definition.
     * @param $table
     * @param $column
     * @param $definition
     * @param string $comment
     * @return boolean true if created, false if already existed
     * @throws BadRequestException
     * @throws ServerException
     */
	public static function addOrAlterColumn($table, $column, $definition, $comment=''){
	    if(!Log::isInited()){
	        throw new ServerException('Log must be init()ed before calling database::addOrAlterColumn');
	    }
	    Log::write(Log::LOGLEVEL_DEBUG, "In database:addOrAlterColumn, $table, $column, $definition");
	    if(!preg_match('/^[A-Za-z0-9_-]+$/', $table)){ throw new BadRequestException('Bad table name "'.htmlentities($table).'" got to addOrAlterColumn'); }
	    if(!preg_match('/^[A-Za-z0-9_-]+$/', $column)){ throw new BadRequestException('Bad column name "'.htmlentities($column).'" got to addOrAlterColumn'); }
	    if(!preg_match('/^[\(\)\s\.\,\\\'\"\/A-Za-z0-9_-]+$/', $definition)){ throw new BadRequestException('Bad definition "'.htmlentities($definition).'" got to addOrAlterColumn'); }
	    if(!preg_match('/^[\(\)\s\.\,\/A-Za-z0-9_-]*$/', $comment)){ throw new BadRequestException('Bad comment "'.htmlentities($comment).'" got to addOrAlterColumn'); }
	    if(!empty(database::queryGetAll('SHOW COLUMNS in '.$table.' LIKE "'.$column.'"'))){
	        Log::write(Log::LOGLEVEL_INFO, "Table $table already has column $column");
	        Log::write(Log::LOGLEVEL_INFO, "Altering to $definition...");
	        database::query("ALTER TABLE $table CHANGE $column $column $definition ");
	        Log::write(Log::LOGLEVEL_INFO, "Altered $table column $column");
	        return false;
	    }
	    Log::write(Log::LOGLEVEL_INFO, "Table $table does not column $column");
	    Log::write(Log::LOGLEVEL_INFO, "Adding column $column to $table");
	    Log::write(Log::LOGLEVEL_INFO, "Definition: $definition");
	    database::query("ALTER TABLE $table ADD $column $definition ");
        Log::write(Log::LOGLEVEL_INFO, "Added column $column to $table");
	    return true;
	}

	/**
	 * 
	 * @param string table The table name
	 * @param string column The column name
	 * @throws ServerException
	 * @throws BadRequestException
	 * @return boolean true if column existed and was dropped, false if column did not exist
	 */
	public static function dropColumnIfExists($table, $column){
	    if(!Log::isInited()){
	        throw new ServerException('Log must be init()ed before calling dropColumnIfExists');
	    }
	    Log::write(Log::LOGLEVEL_DEBUG, "In database:dropColumnIfExists, $table, $column");
	    if(!preg_match('/^[A-Za-z0-9_-]+$/', $table)){ throw new BadRequestException('Bad table name '.htmlentities($table).' got to dropColumnIfExists'); }
	    if(!preg_match('/^[A-Za-z0-9_-]+$/', $column)){ throw new BadRequestException('Bad column name '.htmlentities($column).' got to dropColumnIfExists'); }
	    if(empty(database::queryGetAll('SHOW COLUMNS in '.$table.' LIKE "'.$column.'"'))){
	        Log::write(Log::LOGLEVEL_WARN, "Table $table has no column $column");
	        return false;
	    }
	    database::query("ALTER TABLE $table DROP $column ");
	    Log::write(Log::LOGLEVEL_INFO, "Dropped column $column in $table");
	    return true;
	}
	
}

