<?php
/**
 * This is the base user class. It contains functionality for access control.
 * All application-specific user methods should go in classes/model/user.php. Do not modify this class unless adding generic access-related functionality.
 */
class baseuser implements crudable {

	private static $fields=array(
		'name'=>validator::USERNAME,
		'email'=>validator::EMAIL,
 		'fullname'=>validator::REQUIRED,
		'password'=>validator::ANY,
		'isactive'=>validator::BOOLEAN,
		'supervisorid'=>validator::INTEGER
	);

	private static $helpTexts=array(
		'name'=>'A unique username identifying the user',
		'email'=>'The user\'s email address',
 		'fullname'=>'The user\'s full name, e.g., John Smith',
		'password'=>'The user\'s password for access',
		'isactive'=>'If inactive, the user cannot log in',
		'supervisorid'=>'The person supervising this user'
	);

	private static $adminSelect='SELECT SQL_CALC_FOUND_ROWS id,name,fullname, email, isactive, supervisorid FROM user WHERE 1=1 ';
	private static $normalSelect='SELECT SQL_CALC_FOUND_ROWS id,fullname, supervisorid FROM user WHERE 1=1 ';

	public static $defaultSortOrder='UPPER(fullname) ASC';
	
	public static function getFieldValidations(){
		return self::$fields;
	}
	public static function getFieldHelpTexts(){
		return self::$helpTexts;
	}
	
	public static function canCreate(){
		return session::isAdmin();
	}
	
	public static function canUpdate($id){
		return session::isAdmin();
	}
	
	private static function getSelectClause(){
		if(session::isAdmin()){
			return self::$adminSelect;
		}
		return self::$normalSelect;
	}

    /**
     * @param $id
     * @return array
     * @throws BadRequestException
     * @throws ServerException
     */
	public static function getById($id){
		$sqlStatement=self::getSelectClause().' AND id=:id';
		$result=database::queryGetOne($sqlStatement, array(':id'=>$id));
		if(empty($result)){ return $result; }
		$adminIds=static::getGroupMemberUserIds(usergroup::ADMINISTRATORS);
		if(in_array($id, $adminIds)){
			$result['isadmin']=1;
		} else {
			$result['isadmin']=0;
		}
		return $result;
	}

    /**
     * @param $name
     * @return array
     * @throws BadRequestException
     * @throws ServerException
     */
	public static function getByName($name){
		$sqlStatement=self::getSelectClause().' AND name=:name';
		$result=database::queryGetOne($sqlStatement, array(':name'=>$name));
		if(empty($result)){ return $result; }
		$adminIds=static::getGroupMemberUserIds(usergroup::ADMINISTRATORS);
		if(in_array($result['id'], $adminIds)){
			$result['isadmin']=1;
		} else {
			$result['isadmin']=0;
		}
		return $result;
	}

    /**
     * @param $searchTerms
     * @param array $request
     * @return array
     * @throws BadRequestException
     * @throws ServerException
     * @noinspection PhpUnused
     */
	public static function getByNameLike($searchTerms,$request=array()){
		$sqlStatement=self::getSelectClause().' AND name LIKE :name';
		$sqlStatement.=database::getOrderClause($request, 'user');
		$sqlStatement.=database::getLimitClause($request);
		$result=database::queryGetAll($sqlStatement, array(':name'=>'%'.$searchTerms.'%'));
		$adminIds=static::getGroupMemberUserIds(usergroup::ADMINISTRATORS);
		if(!empty($result)){
			foreach($result['rows'] as &$r){
				if(in_array($r['id'], $adminIds)){
					$r['isadmin']=1;
				} else {
					$r['isadmin']=0;
				}
			}
		}
		return $result;
	}

    /**
     * @param $key
     * @param $value
     * @param array $request
     * @return array
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws ServerException
     */
	public static function getByProperty($key,$value,$request=array()){
		//verify key in fields array
		if(!in_array($key, self::$fields)){
			throw new BadRequestException('Property '.$key.' not recognised on user');
		}
		if(!session::isAdmin()){
			$keys=array('fullname');
			if(!in_array($key, $keys)){
				throw new ForbiddenException('Property '.$key.' not recognised on user');
			}
		}
		$sqlStatement=self::getSelectClause().' AND '.$key.'=:val';
		$sqlStatement.=database::getOrderClause($request, 'user');
		$sqlStatement.=database::getLimitClause($request);
		$params=array(':val'=>$value);
		$result=database::queryGetAll($sqlStatement,$params);
		if(!empty($result)){
			$adminIds=static::getGroupMemberUserIds(usergroup::ADMINISTRATORS);
			foreach($result['rows'] as &$r){
				if(in_array($r['id'], $adminIds)){
					$r['isadmin']=1;
				} else {
					$r['isadmin']=0;
				}
			}
		}
		return $result;
	}

    /**
     * @param array $request
     * @return array
     * @throws BadRequestException
     * @throws ServerException
     */
	public static function getAll($request=array()){
		$sqlStatement=self::getSelectClause();
        $sqlStatement.=database::getFilterClause($request, 'user');
        $sqlStatement.=database::getOrderClause($request, 'user');
		$sqlStatement.=database::getLimitClause($request);
		$params=array();
		$result=database::queryGetAll($sqlStatement,$params);
		$adminIds=static::getGroupMemberUserIds(usergroup::ADMINISTRATORS);
		if(!empty($result)){
			foreach($result['rows'] as &$r){
				if(in_array($r['id'], $adminIds)){
					$r['isadmin']=1;
				} else {
					$r['isadmin']=0;
				}
			}
		}
		return $result;
	}

    /**
     * @param $groupNameOrId
     * @return array
     * @throws BadRequestException
     */
	private static function getGroupMemberUserIds($groupNameOrId){
		return usergroup::getGroupMemberUserIds($groupNameOrId);
	}

    /**
     * @param array $request
     * @return array
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
	public static function create($request=array()){
		if(!session::isAdmin()){
			throw new ForbiddenException('You do not have permission to create users.');
		}
		$keys=array();
		foreach (self::$fields as $fieldName=>$validations){
			if(isset($request[$fieldName])){
				validator::validate($fieldName, $request[$fieldName], $validations);
				$keys[]=$fieldName;
				if('password'==$fieldName){
					$request[$fieldName]=password_hash($request[$fieldName], PASSWORD_BCRYPT);
				}
			} else {
				validator::validate($fieldName, null, $validations);
			}
			
		}
		$fields=implode(',', $keys);
		$values=':'.implode(',:', $keys);
        /** @noinspection SqlInsertValues */
        $sqlStatement='INSERT INTO user('.$fields.') VALUES('.$values.')';
		$params=array();
		foreach($keys as $k){
			if(isset($request[$k])){
				$params[':'.$k]=$request[$k];
			}
		}
		database::query($sqlStatement,$params);
		$newUserId=database::getLastInsertId();
		$everyone=usergroup::getByName(usergroup::EVERYONE);
		database::query(
				'INSERT INTO groupmembership(userid,usergroupid) VALUES(:user,:grp)',
				array(':user'=>$newUserId, ':grp'=>$everyone['id'])
		);
		
		$defaultHomepageBricks=database::queryGetAll('SELECT * FROM homepagedefaultbrick', array());
		if(!empty($defaultHomepageBricks)){
			foreach($defaultHomepageBricks['rows'] as $d){
			    homepageuserbrick::create(array(
                    'userid'=>$newUserId,
                    'homepagebrickid'=>$d['homepagebrickid'],
                    'row'=>$d['row'],
                    'col'=>$d['col']
                ));
			}
		}
		
		return array('type'=>'user', 'created'=>self::getById($newUserId));	
	}

    /**
     * @param $id
     * @param array $request
     * @return array
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws ServerException
     */
	public static function update($id, $request=array()){
		if(!session::isAdmin()){
			if($id==session::getUserId()){
				//User can only update his password, and only if the old one is supplied;
				if(isset($request['password']) && isset($request['oldpassword'])){
					$result=database::queryGetOne('SELECT password FROM user WHERE id=:id AND isactive=true', array(':id'=>$id));
					if(!$result || !password_verify($request['oldpassword'], $result['password'])){
						throw new ForbiddenException('User ID does not exist or old password did not match');
					}
					$sqlStatement='UPDATE user set password=:password WHERE id=:id';
					$params=array(
						':password'=>password_hash($request['password'], PASSWORD_BCRYPT),
						':id'=>$id
					);
					database::query($sqlStatement, $params);
				} else {
					throw new ForbiddenException('You do not have permission to update users');
				}
			}

		} else {
			//Admin
			foreach($request as $k=>$v){
				//if('name'==$k){ throw new BadRequestException('Username cannot be changed'); }
				if(in_array($k, array_keys(self::$fields))){
					validator::validate($k, $v, self::$fields[$k]);
					if("password"==$k){
						$v=password_hash($v, PASSWORD_BCRYPT);
					}
					database::query('UPDATE user SET '.$k.'=:val WHERE id=:id', array(':id'=>$id, ':val'=>$v));
				}
			}
		}
		return array('updated'=>self::getById($id));
	}

    /**
     * @param $id
     * @throws ForbiddenException
     */
	public static function delete($id){
		if(session::isAdmin()){
			throw new ForbiddenException('Users cannot be deleted. Make the user inactive instead.');
		}
		throw new ForbiddenException('You do not have permission to manage users');
	}

    /**
     * @param $id
     * @param array $request
     * @return array
     * @throws ForbiddenException
     * @noinspection PhpUnused
     */
	public static function getusergroups($id, $request=array()){
		if(session::isAdmin()){
			//Show an administrator this user's group memberships
			$sqlStatement='SELECT ug.name AS name, ug.id AS id, gm.isgroupadmin, gm.id AS groupmembershipid FROM usergroup AS ug, groupmembership AS gm
				WHERE ug.id=gm.usergroupid AND gm.userid=:userid ORDER BY ug.issystem DESC, ug.name ';
		} else if($id==session::getUserId()){
			//Show the current user only groups he is authorised to see - he may be in hidden groups, but don't tell him
			$sqlStatement="SELECT ug.name AS name, ug.id AS id, gm.isgroupadmin, gm.id AS groupmembershipid FROM usergroup AS ug, groupmembership AS gm
				WHERE ug.id=gm.usergroupid AND ug.groupvisibility IN('visible','membersonly') AND gm.userid=:userid ORDER BY ug.issystem DESC, ug.name ";
		} else {
			//If not an administrator or looking at own groups, bail
			throw new ForbiddenException('Only administrators can see this.');
		}
		$sqlStatement.=database::getLimitClause($request);
		$parameters=array(':userid'=>$id);
		return database::queryGetAll($sqlStatement, $parameters);
	}

    /**
     * @param $id
     * @param array $request
     * @return array
     * @throws ForbiddenException
     * @noinspection PhpUnused
     */
	public static function getnonmemberusergroups($id, $request=array()){
		if(!session::isAdmin() && $id!=session::getUserId()){
			throw new ForbiddenException('Only administrators can see this.');
		}
		$sqlStatement='SELECT ug.id AS id FROM usergroup AS ug, groupmembership AS gm
				WHERE ug.id=gm.usergroupid AND gm.userid=:userid ORDER BY ug.issystem DESC, ug.name ';
		$parameters=array(':userid'=>$id);
		$memberGroups=database::queryGetAll($sqlStatement, $parameters);
		$inClause='';
		if(!empty($memberGroups) && 0!=count($memberGroups['rows'])){
			$inClause=' AND id NOT IN('.implode(',', array_column($memberGroups['rows'], 'id')) .')';
		}
		$sqlStatement='SELECT id, name FROM usergroup WHERE 1=1 '.$inClause;
		$sqlStatement.=database::getLimitClause($request);
		return database::queryGetAll($sqlStatement, array());
	}

    /**
     * @param $id
     * @param array $request
     * @return array
     * @throws BadRequestException
     * @throws ForbiddenException
     * @noinspection PhpUnused
     * @noinspection PhpUnusedParameterInspection
     */
	public static function getprojects($id, $request=array()){
		if(!session::isAdmin() && $id!=session::getUserId()){
			throw new ForbiddenException('Only administrators can see this.');
		}
		$projects=array();
		
		$sqlStatement='SELECT project.name as projectname, project.id AS projectid, 1 AS isowner 
				FROM project
				WHERE owner=:userid
		';
		$sqlStatement.=database::getOrderClause(array("sortby"=>"name"),'project');
		$parameters=array(':userid'=>$id);
		$owned=database::queryGetAll($sqlStatement, $parameters);
		if(!empty($owned)){
			foreach($owned['rows'] as $row){
				if(!isset($projects[$row['projectname']])){
					$projects[$row['projectname']]=array(
							'id'=>$row['projectid'],
							'name'=>$row['projectname'],
							'isowner'=>1,
							'permissions'=>array()
					);
				}
			}
		}		
		
		
		$sqlStatement='SELECT project.name as projectname, project.id AS projectid, 
					usergroup.name AS usergroupname, usergroup.id AS usergroupid,
					groupmembership.id AS groupmembershipid, permission.type AS type
				FROM groupmembership, usergroup, permission, project
				WHERE groupmembership.userid=:userid
					AND groupmembership.usergroupid=usergroup.id
					AND permission.usergroupid=usergroup.id 
					AND permission.projectid=project.id
		';
		if(!empty($owned) && count($owned['rows'])>0){
			$ownedIds=array_column($owned['rows'],'projectid');
			$sqlStatement.=' AND project.id NOT IN('.implode(',',$ownedIds).')';
		}
		$sqlStatement.=database::getOrderClause(array("sortby"=>"name"),'project');
		$parameters=array(':userid'=>$id);
		$result=database::queryGetAll($sqlStatement, $parameters);
		foreach($result['rows'] as $row){
			if(!isset($projects[$row['projectname']])){
				$projects[$row['projectname']]=array(
					'id'=>$row['projectid'],
					'name'=>$row['projectname'],
					'isowner'=>0,
					'permissions'=>array()
				);
			}
			if(!isset($projects[$row['projectname']]['permissions'][$row['type']])){
				$projects[$row['projectname']]['permissions'][$row['type']]=array();
			}
			$projects[$row['projectname']]['permissions'][$row['type']][]=array(
					'usergroupid'=>$row['usergroupid'],
					'usergroupname'=>$row['usergroupname']
			);
		}
		$projects=array_values($projects);
		return array('total'=>count($projects), 'rows'=>$projects);
	}

    /**
     * Returns users below the specified user in the supervision hierarchy.
     * If $request['allbelow'] is set, users under this one (supervisees' supervisees, etc.) will be returned in a single array.
     * @param $id
     * @param array|null $request The parameters supplied in the request.
     * @return array|null
     * @noinspection PhpUnused
     */
	public static function getsupervisees($id, $request=array()){
		$request['pagenumber']=1;
		$request['pagesize']=100000;
		if(!isset($request['allbelow'])){
			return database::queryGetAll(
					'SELECT id, name, fullname, supervisorid FROM user WHERE supervisorid=:userid', 
					array(':userid'=>$id)
			);
		} else {
			$ids=array();
			$supervisees=database::queryGetAll(
					'SELECT id FROM user WHERE supervisorid=:userid', 
					array(':userid'=>$id)
			);
			if(empty($supervisees)){return null; }
			while(!empty($supervisees)){
				$newIds=array_column($supervisees['rows'],'id');
				$ids=array_merge($ids,$newIds);
				$count=1;
				$placeholders=array();
				$values=array();
				foreach($newIds as $n){
					$placeholders[]=':u'.$count;
					$values[':u'.$count]=$n;
					$count++;
				}
				$supervisees=database::queryGetAll(
					'SELECT id FROM user WHERE supervisorid IN('.implode(', ',$placeholders).')',
					$values
				);	
			}
			
			$placeholders=array();
			$values=array();
			$count=0;
			foreach($ids as $n){
				$placeholders[]=':u'.$count;
				$values[':u'.$count]=$n;
				$count++;
			}
			return database::queryGetAll(
				'SELECT id, name, fullname, supervisorid FROM user WHERE id IN('.implode(', ',$placeholders).')',
				$values
			);	
			
		}
		
	}

    /**
     * @param $id
     * @return bool
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ServerException
     */
	public static function isAdmin($id){
		$adminGroup=usergroup::getByName(usergroup::ADMINISTRATORS);
		$membership=database::queryGetOne(
				'SELECT * FROM groupmembership WHERE userid=:uid AND usergroupid=:gid', 
				array(':uid'=>$id, ':gid'=>$adminGroup['id'])
		);
		return !empty($membership);
	}

    /**
     * Returns the first admin user (by ID)
     * @return mixed
     * @throws BadRequestException
     * @throws NotFoundException
     */
	public static function getFirstAdmin(){
        $adminGroup=usergroup::getByName(usergroup::ADMINISTRATORS);
        if(!$adminGroup){ throw new NotFoundException('No usergroup called '.usergroup::ADMINISTRATORS.' found'); }
        $members=usergroup::getGroupMemberUserIds($adminGroup['id']);
        if(!$members){ throw new NotFoundException('Administrators usergroup has no memmbers'); }
        return user::getById($members[0]);
    }

}