<?php 
/**
 * This is the base project class. It contains functionality for access control.
 * All application-specific project methods should go in classes/model/project.php. Do not modify this class unless adding generic access-related functionality.
 * 
 * A project is essentially a container for application-specific records, serving as a way to apply common access rights to them all.
 * One user is designated the "project owner". The project owner can perform certain actions such as archiving the project or updating its details. They may also have some application-specific privileges.
 * Users are members of usergroups. The project owner can grant those groups read, update, create and delete permissions "on the project" - actually on the records within the project.
 * 
 */
class baseproject implements crudable {

	const SHARED='Shared'; //The shared project for stuff that anyone might need, for example rooms, shared equipment...
	const DEFAULTPROJECT='Default Project'; //The default for unknown items
	public static $defaultSortOrder='issystem DESC, UPPER(project.name) ASC';
	
	/**
	 * This is the base project class, which handles only general access-related functions.
	 * If you want to add methods that are specific to your application, you should do these in
	 * classes/model/project.class.php instead. 
	 */
	
	protected static $fields=array(
			'name'=>validator::REQUIRED,
			'description'=>validator::REQUIRED,
			'owner'=>array(validator::INTEGER, validator::REQUIRED),
			'isarchived'=>validator::BOOLEAN
	);
	
	protected static $helpTexts=array(
			'name'=>'A unique name for the project',
			'description'=>'A short description of why the project exists',
			'owner'=>'The user in charge of this project',
			'isarchived'=>'If archived, the project cannot be changed'
	);

    /**
     * @return array
     */
	public static function getFieldValidations(){
		return self::$fields;
	}

    /**
     * @return array
     */
	public static function getFieldHelpTexts(){
		return self::$helpTexts;
	}

    /**
     * @param $id
     * @return array
     * @throws BadRequestException
     * @throws NotFoundException
     */
	public static function getById($id){
		$sqlStatement='SELECT project.*,user.fullname AS ownername 
				FROM project, user WHERE user.id=project.owner AND project.id=:id ';
		if(!session::isAdmin()){
			$readProjects=session::getReadProjects();
			if(empty($readProjects)){
				$readProjects=array();
				$readProjects[]=0;
			}
			$sqlStatement.=' AND project.id IN('.implode(',', $readProjects).')';
		}
		$data=database::queryGetOne($sqlStatement, array(':id'=>$id));
		if(!$data){
			throw new NotFoundException('Project does not exist, or you do not have permission to see it.');
		}
		return $data;
	}

    /**
     * @param $name
     * @return array
     * @throws BadRequestException
     * @throws NotFoundException
     */
	public static function getByName($name){
		$sqlStatement='SELECT project.*,user.fullname AS ownername
				FROM project, user WHERE user.id=project.owner AND project.name=:name ';
		if(!session::isAdmin()){
			$readProjects=session::getReadProjects();
			if(empty($readProjects)){
				$readProjects=array();
				$readProjects[]=0;
			}
			$sqlStatement.=' AND project.id IN('.implode(',', $readProjects).')';
		}
		$data=database::queryGetOne($sqlStatement, array(':name'=>$name));
		if(!$data){
			throw new NotFoundException('Project does not exist, or you do not have permission to see it.');
		}
		return $data;
	}

    /**
     * @param $key
     * @param $value
     * @param array $request
     * @return array
     * @throws BadRequestException
     */
	public static function getByProperty($key,$value,$request=array()){
		$keys=array_keys(self::getFieldValidations());
		if(!in_array($key, $keys)){
			throw new BadRequestException('Property '.$key.' not recognised on project');
		}
		$sqlStatement='SELECT project.*,user.fullname AS ownername
				FROM project, user WHERE user.id=project.owner AND project.'.$key.'=:val ';
		if(!session::isAdmin()){
			$readProjects=session::getReadProjects();
			if(empty($readProjects)){
				$readProjects=array();
				$readProjects[]=0;
			}
			$sqlStatement.=' AND project.id IN('.implode(',', $readProjects).')';
		}
		$sqlStatement.=database::getOrderClause($request, 'project');
		$sqlStatement.=database::getLimitClause($request);
		return database::queryGetAll($sqlStatement, array(':val'=>$value));
	}

    /**
     * @param array $request
     * @return array
     * @throws BadRequestException
     */
	public static function getAll($request=array()){
		$sqlStatement='SELECT project.*,user.fullname AS ownername
				FROM project, user WHERE user.id=project.owner ';
		if(!session::isAdmin()){
			$readProjects=session::getReadProjects();
			if(empty($readProjects)){
				$readProjects=array();
				$readProjects[]=0;
			}
			$sqlStatement.=' AND project.id IN('.implode(',', $readProjects).')';
		}
		$sqlStatement.=database::getOrderClause($request, 'project');
		$sqlStatement.=database::getLimitClause($request);
		return database::queryGetAll($sqlStatement, array());
	}

    /**
     * @param array $request
     * @return array
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
	public static function create($request=array()){
		if(!session::isAdmin() &&!session::canCreateProjects()){
			throw new ForbiddenException('You do not have permission to create projects.');
		}
		$keys=array();
		foreach (static::$fields as $fname=>$validations){
			if(isset($request[$fname])){
				validator::validate($fname, $request[$fname], $validations);
				$keys[]=$fname;
			} else {
				validator::validate($fname, null, $validations);
			}
		}
		$fields=implode(',', $keys);
		$values=':'.implode(',:', $keys);
        /** @noinspection SqlInsertValues */
        $sqlStatement='INSERT INTO project('.$fields.') VALUES('.$values.')';
		$params=array();
		foreach($keys as $k){
			if(isset($request[$k])){
				$params[':'.$k]=$request[$k];
			}
		}
		database::query($sqlStatement,$params);
		$newProjectId=database::getLastInsertId();

		/* We need to refresh the permissions in order to see the project we just created.
		 * However, dummy sessions (as used in importers) do not derive admin permissions
		 * from group membership and have it set explicitly; refreshing the permissions
		 * nukes that setting, so we have to do this little dance:
		 */
		session::refreshProjectPermissions();
		$wasAdmin=session::isAdmin();
		session::set('isAdmin',true);
		$created=static::getById($newProjectId);
		session::set('isAdmin',$wasAdmin);
		
		return array('type'=>'project', 'created'=>$created);
	}

    /**
     * @param $id
     * @param array $request
     * @return array
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
	public static function update($id, $request=array()){
		if(!session::isAdmin()){
			$project=self::getById($id);
			if($project['owner']!=session::getUserId()){
				throw new ForbiddenException('You do not have permission to update users');
			}
		}
		foreach($request as $k=>$v){
			if(in_array($k, array_keys(self::$fields))){
				validator::validate($k, $v, self::$fields[$k]);
				database::query('UPDATE project SET '.$k.'=:val WHERE id=:id', array(':id'=>$id, ':val'=>$v));
			}
		}
		return array('updated'=>self::getById($id));
	}

    /**
     * @param $id
     * @throws BadRequestException
     */
	public static function delete($id){
		throw new BadRequestException('Projects cannot be deleted');
	}

    /**
     * @return bool
     */
	public static function canCreate(){
		return session::isAdmin() || session::canCreateProjects();
	}

    /**
     * @param $id
     * @return bool
     * @throws NotFoundException
     * @throws BadRequestException
     */
	public static function canUpdate($id){
		if(session::isAdmin()){ return true; }
		$project=project::getById($id);
		if($project['owner']==session::getUserId()){ return true; }
			return false;
	}

    /**
     * @param $id
     * @param array $request
     * @return array
     * @noinspection PhpUnused
     * @noinspection PhpUnusedParameterInspection
     */
	public static function getpermissions($id,$request=array()){
		$sqlStatement='SELECT id,name FROM usergroup WHERE 1=1 ';
		if(!session::isAdmin()){
			$readables=session::getVisibleGroupIds();
			if(empty($readables)){ $readables=array('0'); }
			$sqlStatement.=' AND id IN('.implode(',', $readables).')';
		}
		$sqlStatement.=' ORDER BY issystem DESC, name ASC';
		//TODO Limit clause
		$result=database::queryGetAll($sqlStatement, array());
		$groups=array();
		foreach($result['rows'] as $r){
			$r['permissions']=array('read'=>false, 'update'=>false, 'create'=>false, 'delete'=>false);
			$groups['grp'.$r['id']]=$r;
		}
		$groupIds=array_column($groups,'id');
		$sqlStatement='SELECT id, usergroupid, type FROM permission WHERE projectid=:id AND usergroupid IN('.implode(',',$groupIds).')';
		$perms=database::queryGetAll($sqlStatement, array(':id'=>$id));
		if(!empty($perms)){
			foreach($perms['rows'] as $p){
				$groups['grp'.$p['usergroupid']]['permissions'][$p['type']]=$p['id'];
			}
		}
		return array(
				'total'=>count($groups), //TODO more than this if LIMIT clause
				'rows'=>array_values($groups)
		);
	}

    /**
     * Returns the ID of the shared project.
     * @return int
     * @throws NotFoundException
     * @throws BadRequestException
     */
	public static function getSharedProjectId(){
		$project=static::getByName(static::SHARED);
		return 1*$project['id'];
	}

    /**
     * Returns the ID of the default project.
     * @return int
     * @throws NotFoundException
     * @throws BadRequestException
     * @noinspection PhpUnused
     */
	public static function getDefaultProjectId(){
		$project=static::getByName(static::DEFAULTPROJECT);
		return 1*$project['id'];
	}

    /**
     * @param $id
     * @param $request
     * @return array|null
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ServerException
     */
	public static function getNotes($id, $request){
        return note::getByProperties(array(
            'parentid'=>database::$nullValue,
            'projectid'=>$id
        ), $request);
    }

    /**
     * @param $id
     * @param $request
     * @return array
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ServerException
     */
    public static function getFiles($id, $request){
        return file::getByProperties(array(
            'parentid'=>database::$nullValue,
            'projectid'=>$id
        ), $request);
    }

}