<?php 
/**
 * Main handler script for the API.
 * 
 */

date_default_timezone_set('UTC');

//Array of actions not needing authentication here. Dedicated handlers may enforce it.
$patternsNotNeedingAuthentication=array(
    'GET /api/diffraction',
    'GET /api/diffraction/.*',
    
);

//GET methods usually abort the transaction to ensure no changes are committed by accident.
//If GET method results in db changes (for example, creating now-mandatory records relating to historical ones),
//set this true to force a database commit after processing
$forceCommit=false;

/*
 * Valid combinations of HTTP verb and URI. Code iterates through these and takes the FIRST match, so
 * put the longest ones first. Optional trailing slash is handled below (as are beginning and end match),
 * so no need to do it in these patterns.
 */
$validPatterns=array(
		
	//Diffraction API call
	'GET /api/diffraction'=>'diffraction',
	'GET /api/diffraction/.*'=>'diffraction',
		
		
	//get file (or image thumbnail) by ID, but with filename appended to URL. Browsers may not 
	//recognise file type without the file extension even when mimetype is correctly set.
	//Special handling is needed because this looks like a "get by property" API URL.
	'GET /api/[^/]*file/[0-9]+/.*'=>'getById',
	'GET /api/[^/]*thumb/[0-9]+/.*'=>'getById',
		
	//get one or more by partial name match, e.g, /api/plate/search/90a6
	'GET /api/search/[^/]+'=>'getByNameLikeMultiClass',
		
	//get one or more by partial name match, e.g, /api/plate/search/90a6
	'GET /api/[A-Za-z]+/search/[^/]+'=>'getByNameLike',
		
	//get associated records for a given record, e.g, /api/group/1234/user
	'GET /api/[A-Za-z]+/[0-9]+/[A-Za-z0-9]+'=>'getAssociated',
		
	//get one or more by three properties, e.g, /api/plate/projectid/1/owner/2/bestscore/9
	'GET /api/[A-Za-z]+/[A-Za-z0-9_]+/[^/]+/[A-Za-z0-9_]+/[^/]+/[A-Za-z0-9_]+/[^/]+'=>'getByProperties',
		
	//get one or more by two properties, e.g, /api/plate/projectid/1/owner/2
	'GET /api/[A-Za-z]+/[A-Za-z0-9_]+/[^/]+/[A-Za-z0-9_]+/[^/]+'=>'getByProperties',
		
	//get one or more by property, e.g, /api/plate/barcode/90a6
	'GET /api/[A-Za-z]+/[A-Za-z0-9_]+/[^/]+'=>'getByProperty',

	//get by database ID, e.g, /api/plate/1234
	'GET /api/[A-Za-z]+/[0-9]+'=>'getById',

	//Delete a record by its database ID
	'DELETE /api/[A-Za-z]+/[0-9]+'=>'delete',
	
    //Call a dedicated function on a record found by its name, e.g, /api/plate/destroyByName/9abc
    'PATCH /api/[A-Za-z]+/[^/]+ByName/[^/]+'=>'actionByName',
    
    //update by database ID, e.g, /api/plate/1234
    'PATCH /api/[A-Za-z]+/[0-9]+'=>'update',
    
    //update site-wide configuration item, e.g, /api/config/parameterNameHere
    'PATCH /api/config/[A-Za-z0-9_]+'=>'updateConfig',
    
    //update user configuration item, e.g, /api/userconfig/parameterNameHere
    'PATCH /api/userconfig/[A-Za-z0-9_]+'=>'updateUserConfig',
    
    //get all, e.g., /api/plate - pagination in request parameters
	'GET /api/[A-Za-z]+'=>'getAll',
	
    //create a new record, e.g., /api/plate - details in request parameters
    'POST /api/[A-Za-z]+'=>'create',
    
    //create a new record in remote AJAX calls
    'PUT /api/corsproxy'=>'create',
    
);


//define class autoloader
spl_autoload_register(function($className){
    global $projectWebsiteEnabled;
    $paths=array(
			'../classes/',
			'../classes/core/',
			'../classes/core/exception/',
			'../classes/core/authentication/',
			'../classes/core/interface/',
	       '../classes/model/',
	       './XraySourceInterface/',
	);
    if($projectWebsiteEnabled){
        $paths[]='../projectWebsite/classes/';
    }
    foreach($paths as $path){
		if(file_exists($path.$className.'.class.php')){
            /** @noinspection PhpIncludeInspection */
            include_once($path.$className.'.class.php');
		}
	}
});

//connect to the database
database::connect();
database::begin();

//set response type
$_REQUEST['responseType']='application/json';
//TODO Later support xml

//determine request method and URI
$wwwroot=rtrim($_SERVER['DOCUMENT_ROOT'],'/');
$method=$_SERVER['REQUEST_METHOD'];

//$uri=rtrim($_SERVER['REQUEST_URI'],'/');
$uri=strtok($_SERVER['REQUEST_URI'],'?');
$uri=rtrim($uri,'/');
$uriArgs=explode('/', substr($uri,1));
$req=$method.' '.$uri;

//get request parameters
$parameters=array();
if('GET'==$method){
	$parameters=$_GET;
} else {
	parse_str(file_get_contents("php://input"),$parameters);
	if(empty($parameters) && !empty($_POST)){
		$parameters=$_POST;
	}
	foreach($parameters as $k=>$v){
		if(is_array($v)){
			for($i=0;$i<count($v);$i++){
				$parameters[$k][$i]=urldecode($parameters[$k][$i]);
			}
		} else {
			$parameters[$k]=urldecode($v);
		}
	}
}
//attempt to start session
$sid=null;
if(isset($parameters['sid'])){
	$sid=$parameters['sid'];
}
session::init(new PhpSession());

/*
 * Handle login/logout here
 */
if('POST /api/Login'==$req){
	try {
		$success=session::login($parameters);
		if($success){ 
		    header('Content-Type: application/json');
		    echo json_encode($success); 
		}
	} catch(Exception $e){
		respond(array('error'=>$e->getMessage()),$e->getCode());
		//TODO send stacktrace if admin
		//TODO Log error
	}
	database::abort();
	exit;
} else if('POST /api/Logout'==$req || 'GET /api/Logout'==$req){
	$success=session::logout($parameters);
	if($success){ echo json_encode($success); }
	database::abort();
	exit;
}

/*
 * If no username in the session by this point: 
 * - Was not logged in, did not attempt login.
 * - Assume machine user, and authenticate based on IP+method+URI
 */
if(empty(session::getUsername())){
	// @TODO Authenticate via IP+method+URI exceptions
}



//get arguments from URI - ignore any initial /api/
if(1==count($uriArgs) && 'api'==$uriArgs[0]){
	header('Content-Type: application/json');
	echo json_encode(array('message'=>'This is the API'));
	exit;
} else if(0<count($uriArgs) && 'api'==$uriArgs[0]){
	array_shift($uriArgs);
} 

/*
if(!session::isLoggedIn()){
	respond(array('error'=>'Login required'),401);
	exit;
}
*/

$objectType=null;
$action=null;

$keys=array_keys($validPatterns);
$numPatterns=count($keys);
for($i=0;$i<$numPatterns;$i++){
	$pattern=$keys[$i];
	if(preg_match('|^'.$pattern.'/?$|', $req)){
		$action=$validPatterns[$pattern];
		$objectType=$uriArgs[0];
		break;
	}
}

$exceptionParameters=array();
$data=array();
try {

    //validate the CSRF token, if changing anything...
    if('GET'!=$method /* AND NOT LOGGING IN OR OUT! */){
        session::validateCsrfToken($parameters);
    }
    //...but don't pass the CSRF token into the handling classes
    unset($parameters['csrfToken']);


    if(!session::isLoggedIn()){
        $actionOK=false;
        foreach($patternsNotNeedingAuthentication as $p){
            if(preg_match('|^'.$p.'/?$|', $req)){
                $actionOK=true;
                break;
            }
        }
        if(!$actionOK){ throw new AuthorizationRequiredException('Login required'); }
	}

	$responseCode=200;
	switch($action){
		case 'getByName':
		case 'getById':
		case 'delete':
		case 'update':
			$args=array($uriArgs[1], $parameters);
			$data=callOrThrow($objectType, $action, $args);
			if($data && ("getById"==$action ||"getByName"==$action)){
				$data['objecttype']=$objectType;
			}
			break;
		case 'create':
			$args=array($parameters);
			$data=callOrThrow($objectType, $action, $args);
			$responseCode=201; // HTTP 201 Created
			break;
		case 'getAll':
			$args=array($parameters);
			$data=callOrThrow($objectType, $action, $args);
			break;
		case 'getByNameLikeMultiClass':
			$args=array(urldecode($uriArgs[1]), $parameters);
			$data=callOrThrow('baseobject', $action, $args);
			break;
		case 'getByNameLike':
			$args=array(urldecode($uriArgs[2]), $parameters);
			$data=callOrThrow($objectType, $action, $args);
			break;
		case 'getByProperties':
			$num=1;
			$args=array();
			while(isset($uriArgs[$num]) && isset($uriArgs[$num+1])){
				$args[$uriArgs[$num]]=urldecode($uriArgs[$num+1]);
				$num=$num+2;
			}
			$args=array($args, $parameters);
			$data=callOrThrow($objectType, $action, $args);
			break;
		case 'getByProperty':
			$args=array($uriArgs[1],urldecode($uriArgs[2]), $parameters);
			$data=callOrThrow($objectType, $action, $args);
			break;
		case 'actionByName':
		    // /api/plate/destroyByName/9abc
		    $args=array(urldecode($uriArgs[2]), $parameters);
		    $data=callOrThrow($objectType, $uriArgs[1], $args);
		    break;
		case 'getAssociated':
			$args=array($uriArgs[1], $parameters);
			$data=callOrThrow($objectType, 'get'.urldecode($uriArgs[2]).'s', $args);
			break;
		case 'updateConfig':
		    $itemName=$uriArgs[1];
		    $itemValue=$parameters[$itemName];
		    config::set($itemName, $itemValue);
		    $data=array('updated'=>array($itemName=>$itemValue));
		    break;
		case 'updateUserConfig':
		    $itemName=$uriArgs[1];
		    $itemValue=$parameters[$itemName];
		    userconfig::set($itemName, $itemValue);
		    $data=array('updated'=>array($itemName=>$itemValue));
		    break;
		case 'diffraction':
			include('diffraction/index.php');
			$data=callDiffractionApi($uriArgs);
			break;
	}
	if(false===$data){
		throw new BadRequestException('Class or method does not exist');
	} else if(null===$data){
		throw new NotFoundException('None found');
	}
	if("GET"==$method && !$forceCommit){
		database::abort();
	} else {
		database::commit();
	}
	respond($data,$responseCode, $action);
} catch(Exception $e){
	database::abort();
	session::revertAdmin();
	$statusCode=$e->getCode();
	$trace=$e->getTrace();
	$prettytrace=array();
	$prettytrace[]=$e->getFile().' line '.$e->getLine();
 	foreach($trace as $t){
 	    $traceline='';
 	    $line=(isset($t['line'])) ? $t['line'] : '--';
 	    $file=(isset($t['file'])) ? str_replace($_SERVER['DOCUMENT_ROOT'], '', (str_replace('\\','/',$t['file']))) : '--';
 	    if (array_key_exists('class',$t)){
 	        $traceline=sprintf("%s:%s %s::%s",
 	            $file,
 	            $line,
 	            $t['class'],
 	            $t['function']
            );
 	    } else {
 	        $traceline=sprintf("%s:%s %s",
                $file,
                $line,
                $t['function']
            );
 	    }
 	    $prettytrace[]=$traceline;
 	}
 	if(Log::isInited()){
 	    Log::write(Log::LOGLEVEL_ERROR, 'Exception was thrown, details follow:');
 	    Log::write(Log::LOGLEVEL_ERROR, $e->getMessage());
 	    Log::write(Log::LOGLEVEL_ERROR, 'Thrown at '.$prettytrace[0]);
 	    foreach($prettytrace as $t){
 	        Log::write(Log::LOGLEVEL_ERROR, $t);
  	    }
  	    //For 5xx exceptions, also log code and database versions.
  	    if($statusCode>=500 && $statusCode<600){
  	        $codeVersion=trim(@file_get_contents(rtrim($_SERVER['DOCUMENT_ROOT'],'/').'/conf/codeversion'));
  	        $databaseVersion=trim(config::get('core_icebearversion'));
  	        Log::write(Log::LOGLEVEL_INFO, 'Code version: '.$codeVersion);
  	        Log::write(Log::LOGLEVEL_INFO, 'Database version: '.$databaseVersion);
  	        if($codeVersion!=$databaseVersion){
  	            Log::write(Log::LOGLEVEL_ERROR, 'Code version and database version do not match.');
  	        }
  	    }
 	}
	respond(array(
	    'error'=>$e->getMessage(),
	    'parameters'=>$exceptionParameters,
	    'thrownat'=>$prettytrace[0],
	    'trace'=>$prettytrace
	),$statusCode);
	//TODO send stacktrace if admin
	//TODO Log error
}

/****************************************
 * Supporting functions below
 ***************************************/

/**
 * Returns the data to the client in the appropriate format, with the supplied HTTP response code.
 * @param array $data
 * @param int $responseCode
 * @param string $action
 */
function respond($data, $responseCode, $action=''){
    header('Access-Control-Allow-Origin: *');
	http_response_code($responseCode);
	$responseType='application/json';
	if(isset($_REQUEST['responseType'])){
		$responseType=$_REQUEST['responseType'];
	}
	header('Content-Type: '.$responseType);
	if('application/json'==$responseType){
		header('Content-Type: application/json');
		$json=json_encode(sanitize($data));
		if('diffraction'==$action){
			$json=DiffractionAPI::postProcessEncodedShipmentJson($json);
		}
		echo $json;
	} else if('text/xml'==$responseType){
		echo 'XML not supported yet';
	}	
}

function sanitize($data){
	foreach($data as $k=>&$v){
		if(is_array($v)){
			$v=sanitize($v);
		} else {
			$v=htmlspecialchars($v);
		}
	}
	return $data;
}

/**
 * Calls a class method with the supplied arguments, throwing a BadRequestException if class or method does not exist.
 * @param string $className The class name
 * @param string $methodName The method name
 * @param array $args The arguments to pass to the method
 * @throws BadRequestException if the class or method does not exist
 * @return mixed The results of the method call
 */
function callOrThrow($className, $methodName, $args){
	if(!class_exists($className)){
		throw new BadRequestException('Class '.$className.' does not exist');
	}
	if(!method_exists($className, $methodName)){
		throw new BadRequestException('Class '.$className.': Method '.$methodName.' does not exist');
	}
	return forward_static_call_array(array($className, $methodName), $args);
}
