<?php
/**
 * Encodes and decodes shipments according to the ISPyB standard shipping API.
 * @author edaniel
 *
 * GET /api/diffraction/v0_1_0/plate/93a7
 *
 */
class DiffractionAPI {
    
    private static function getServerName(){
        return rtrim($_SERVER['SERVER_NAME'],'/');
    }
    private static function getRequestProtocol(){
        return (isset($_SERVER['SERVER_PROTOCOL']) && 'off'!=$_SERVER['SERVER_PROTOCOL']) ? 'https://' : 'http://';
    }
    
    private static function checkServerNameFullyQualified(){
        if(false===strpos(static::getServerName(), '.')){
            throw new ServerException('IceBear server not properly configured with fully-qualified domain name.');
        }
        return true;
    }
    
    public static function getVersion(){
        return '0.1.0';
    }
    
    /**
     * Cleans up the output of PHP's json_encode() function to ensure that it validates against the schema.
     *
     * By default, json_encode converts ALL values to strings, meaning that (for example) x/y coordinates cause
     * the schema validation to fail. Specifying JSON_NUMERIC_CHECK forces ALL number-like values to int/float,
     * regardless of their original type, which causes schema validation to fail in the case of, for example,
     * an all-numeric barcode (should be type "string").
     *
     * This should not be necessary, and is something of a filthy hack until PHP gets its act together.
     */
    public static function postProcessEncodedShipmentJson($jsonString){
        $numericFields=array('x','y','r','micronsPerPixel','positionInPuck','rowNumber','columnNumber','dropNumber');
        $pattern = '/"('.implode('|',$numericFields).')":"([0-9.-]+)"/i';
        $replacement = '"$1":$2';
        $jsonString=preg_replace($pattern, $replacement, $jsonString);
        return $jsonString;
    }
    
    /*
     * Functions for sending info from IceBear, e.g., shipment, plate task list.
     */
    
    public static function encodeShipment($shipmentId, $format='json'){
        static::checkServerNameFullyQualified();
        //if encoding plates, specify 'array' as format!
    }
    
    public static function encodePlateByBarcode($barcode, $format='json'){
        Log::write(Log::LOGLEVEL_DEBUG, 'In encodePlateByBarcode, barcode='.$barcode);
        Log::write(Log::LOGLEVEL_DEBUG, 'Checking server name fully qualified');
        static::checkServerNameFullyQualified();
        Log::write(Log::LOGLEVEL_DEBUG, 'Getting plate');
        $plate=plate::getByName($barcode);
        if(!$plate){
            throw new NotFoundException('No plate with barcode '.$barcode);
        }
        Log::write(Log::LOGLEVEL_DEBUG, 'Plate exists');
        $drops=plate::getwelldrops($plate['id']);
        $drops=$drops['rows'];
        Log::write(Log::LOGLEVEL_DEBUG, 'Plate has '.count($drops).' drops');
        $dropsWithCrystals=array();
        Log::write(Log::LOGLEVEL_DEBUG, 'About to encode drops');
        foreach($drops as $wd){
            $crystals=welldrop::getcrystals($wd['id']);
            if(!empty($crystals)){
                $dropsWithCrystals[]=static::encodePlateDrop($wd);
            }
        }
        
        $response=array(
            'barcode'=>$plate['name'],
            'plateType'=>$plate['platetypename'],
            'dropMapping'=>$plate['dropmapping'],
            'drops'=>$dropsWithCrystals
        );
        return static::formatArray($response, $format);
    }
    
    /**
     * Temporary function for demo around ISPyB meeting Jan 2018
     * @param string $barcode
     * @param string $format
     * @throws NotFoundException
     * @return array|string
     */
    public static function encodeDewarByBarcode($barcode, $format='json'){
        static::checkServerNameFullyQualified();
        $dewar=container::getByName($barcode);
        if(!$dewar){
            throw new NotFoundException('No dewar with barcode '.$barcode);
        }
        return static::encodeDewar($dewar['id']);
    }
    
    public static function encodePlate($plateId, $format='json'){
        $plate=plate::getById($plateId);
        return static::encodePlateByBarcode($plate['name'], $format);
    }
    
    private static function encodeCrystal($crystalId, $format='array'){
        if(is_int($crystalId) || (int)$crystalId!=0){
            $crystal=crystal::getById($crystalId);
        } else {
            $crystal=$crystalId;
        }
        
        $wellDrop=welldrop::getById($crystal['welldropid']);
        if(empty($wellDrop['constructid'])){
            throw new BadRequestException('Crystal with ID '.$crystalId.' is not associated with a protein in IceBear');
        }
        $construct=construct::getById($wellDrop['constructid']);
        $crystalUrl=static::getRequestProtocol().static::getServerName().'/crystal/'.$crystalId;
        $dropImage=dropimage::getById($crystal['dropimageid']);
        $extension=pathinfo($dropImage['imagepath'], PATHINFO_EXTENSION);
        $imageUrl=static::getRequestProtocol().static::getServerName().'/dropimagefile/'.$dropImage['id'].'/full.'.$extension;
        $response=array(
            'sampleName'=>$crystal['name'], //required
            'proteinAcronym'=>$construct['proteinacronym'], //required
            'sendersId'=>$crystalId,
            //'receiversId'=>'',
            'sendersUrl'=>$crystalUrl,
            'sendersImage'=>array(
                'imageUrl'=>$imageUrl,
                'sendersId'=>$dropImage['id'],
                'micronsPerPixel'=>1*$dropImage['micronsperpixelx'],
                'marks'=>array(
                    
                    array(
                        'regionType'=>'point',
                        'x'=>1*$crystal['pixelx'],
                        'y'=>1*$crystal['pixely']
                    )
                    
                )
            )
        );
        return static::formatArray($response, $format);
    }
    
    /**
     * Returns the specified plate drop, encoded in the format requested.
     * @param int|array $dropId The drop ID, or the result of calling welldrop::getById
     * @param string $format The return format
     */
    private static function encodePlateDrop($dropId, $format='array'){
        Log::write(Log::LOGLEVEL_DEBUG, 'In encodePlateDrop');
        if(is_int($dropId)){
            Log::write(Log::LOGLEVEL_DEBUG, '$dropId was an int: '.$dropId);
            $drop=welldrop::getById($dropId);
        } else {
            $drop=$dropId;
            Log::write(Log::LOGLEVEL_DEBUG, '$dropId was an array: ID in that array is'.$drop['id']);
        }
        
        $encodedCrystals=array(); //one element per crystal marked in drop (regardless of which image)
        Log::write(Log::LOGLEVEL_DEBUG, 'Getting crystals in drop');
        $crystals=welldrop::getcrystals($drop['id']);
        $usedImages=array();
        foreach($crystals['rows'] as $c){
            Log::write(Log::LOGLEVEL_DEBUG, 'About to encode crystal');
            $encodedCrystal=static::encodeCrystal($c['id']);
            Log::write(Log::LOGLEVEL_DEBUG, 'Encoded crystal');
            $encodedCrystals[]=$encodedCrystal;
            Log::write(Log::LOGLEVEL_DEBUG, 'Added crystal to list');
        }
        
        $overviewImageUrl='';
        $highestImageId=0;
        if(1==count($encodedCrystals)){
            Log::write(Log::LOGLEVEL_DEBUG, 'One crystal is present');
            $overviewImageUrl=$encodedCrystals[0]['sendersImage']['imageUrl'];
        } else if(!empty($encodedCrystals)){
            Log::write(Log::LOGLEVEL_DEBUG, 'More than one crystal is present');
            foreach($encodedCrystals as $c){
                if($c['sendersImage']['sendersId']>$highestImageId){
                    $highestImageId=$c['sendersImage']['sendersId'];
                    $overviewImageUrl=$c['sendersImage']['imageUrl'];
                }
            }
        }
        Log::write(Log::LOGLEVEL_DEBUG, 'Overview image is '.$overviewImageUrl);
        
        Log::write(Log::LOGLEVEL_DEBUG, 'Getting well label, row='.$drop['row'].', col='.$drop['col'].', drop='.$drop['dropnumber']);
        $dropLabel=platewell::getWellLabel($drop['row'], $drop['col']).'.'.$drop['dropnumber'];
        Log::write(Log::LOGLEVEL_DEBUG, 'Well label is '.$dropLabel);
        
        $response=array(
            'position'=>$dropLabel, //A01.1
            'rowNumber'=>$drop['row'],
            'columnNumber'=>$drop['col'],
            'dropNumber'=>$drop['dropnumber'],
            'overviewImageUrl'=>$overviewImageUrl,
            'crystals'=>$encodedCrystals
        );
        Log::write(Log::LOGLEVEL_DEBUG, 'formatting response');
        $response=static::formatArray($response, $format);
        Log::write(Log::LOGLEVEL_DEBUG, 'returning from encodePlateDrop');
        return $response;
    }
    
    private static function encodePin($pinId, $format='array'){
        
        $cc=containercontent::getByProperties(array(
            'child'=>$pinId,
            'iscurrent'=>1
        ));
        if(!$cc || 1!=count($cc['rows'])){
            throw new BadRequestException('Pin is not in a puck, or appears to be in two places at once');
        }
        
        $crystals=array();
        $positionInPuck=$cc['rows'][0]['position'];
        $pin=container::getById($pinId);
        
        $cc=containercontent::getByProperties(array(
            'parent'=>$pinId,
            'iscurrent'=>1
        ));
        if(!$cc || 1<count($cc['rows'])){
            throw new BadRequestException('Pin has no contents');
        }
        foreach ($cc['rows'] as $c){
            $crystals[]=static::encodeCrystal($c['child'],$format);
        }
        
        $response=array();
        if($pin){
            $response['barcode']=$pin['name'];
        }
        $response['positionInPuck']=$positionInPuck;
        $response['crystals']=$crystals;
        
        return static::formatArray($response, $format);
    }
    
    private static function encodePuck($puckId, $format='array'){
        $pins=array();
        $puck=container::getById($puckId);
        if(!$puck || "puck"!=strtolower($puck['containercategoryname'])){
            throw new BadRequestException('Supplied puck ID is not a puck');
        }
        $contents=container::getcontents($puckId);
        if(!$contents){
            throw new BadRequestException('Puck '.$puck['name'].' is empty, cannot include in shipment');
        }
        $contents=$contents['rows'];
        foreach ($contents as $c){
            $pins[]=static::encodePin($c['child']);
        }
        $response=array(
            'barcode'=>$puck['name'],
            'pins'=>$pins
        );
        return static::formatArray($response, $format);
    }
    
    private static function encodeDewar($dewarId, $format='array'){
        $pucks=array();
        $dewar=container::getById($dewarId);
        if(!$dewar || "dewar"!=strtolower($dewar['containercategoryname'])){
            throw new BadRequestException('Supplied dewar ID is not a dewar');
        }
        $contents=container::getcontents($dewarId);
        if(!$contents){
            throw new BadRequestException('Dewar '.$dewar['name'].' is empty, cannot include in shipment');
        }
        $contents=$contents['rows'];
        foreach ($contents as $c){
            $pucks[]=static::encodePuck($c['child']);
        }
        $response=array(
            'barcode'=>$dewar['name'],
            'pucks'=>$pucks
        );
        return static::formatArray($response, $format);
    }
    
    
    private static function formatArray($array, $format){
        if('array'==$format){
            return $array;
        } else if('json'==$format){
            return json_encode($array);
        }
        throw new BadRequestException('Unrecognised format '.$format);
    }
    
}
